/***************************************************************************
 *                                                                         *
 *    NoMoRe++                                                             *
 *                                                                         *
 *    Copyright (C) 2003-2005 NoMoRe Developing Group                      *
 *                                                                         * 
 *    For more information, see http://www.cs.uni-potsdam.de/nomore/       *
 *    or email to nomore-dg@cs.uni-potsdam.de                              *
 *                                                                         *
 *                                                                         *
 *    This program is free software; you can redistribute it and/or        *
 *    modify it under the terms of the GNU General Public License          *
 *    as published by the Free Software Foundation; either version 2       *
 *    of the License, or (at your option) any later version.               *
 *                                                                         *
 *    This program is distributed in the hope that it will be useful,      *
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of       *
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the        *
 *    GNU General Public License for more details.                         *
 *                                                                         *
 *    You should have received a copy of the GNU General Public License    *
 *                                                                         * 
 ***************************************************************************/

#ifndef NOMORE_CHOICE_OPERATOR_H
#define NOMORE_CHOICE_OPERATOR_H

#include <operator.h>
#include <color.h>
#include <heuristic.h>
namespace NS_NOMORE {

class Node;
class Constraint;

//! Base class for all non deterministic choice-operators.
/*! ChoiceOperator implements the operator()() of the base class Operator. 
 *  Concrete sub-classses must override the method selectChoice for choosing 
 *  the next node to be colored as choice point. 
 *  
 *  There are two ways to use a concrete choice-operator.
 *
 *  - <b>Method 1:</b> Use the choice-operator just like any other operator
 *    by calling its function-operator. This way an uncolored node is selected
 *    (using the installed heuristics) and automatically colored as choice point. 
 *    \code
 *      ColorOpResult result = anChoiceOperator();
 *      if (result == ColorOperation::succeeded) {
 *        // choice point found and successfully colored
 *      }
 *      else if (result == ColorOperation::failed) {
 *        // choice point found but color error while coloring the node
 *      }
 *      else {
 *        // no choice point found
 *      }
 *    \endcode
 *    
 *  - <b>Method 2:</b> Select and color a node manually. 
 *  To do this call the selectChoice method then coolor the returned node (e.g.
 *  by calling graph::colorChoicePoint).
 *  selectChoice returns a null-ptr if no choices are left.
 *    \code
 *      node = anChoiceOperator.selectChoice();
 *      if(node == NULL) {
 *        // no choice point found
 *      } 
 *      else {
 *        graph.colorChoicePoint(*node, myFirstColor, mySecondColor);
 *      }
 *    \endcode
 */
class ChoiceOperator : public Operator {
public:
  //! Creates the choice operator.
  /*! 
   * \param grp The graph the operator should work on. 
   * \param h The heuristic to use for valuating possible choice points
   */
  ChoiceOperator(Graph& grp, Heuristic* h);
  ~ChoiceOperator();

  //! Executes this choice-operator.
  /*!
   *  
   * \pre choiceType is either ColorOpType::first_choice or ColorOpType::next_choice
   *
   * This base class implements the following algorithm:
   * \code
   *  ColorOpResult result = ColorOperation::unchanged;
   *  Node* n = selectNode();
   *  if (n != 0) {
   *    if (graph_->colorChoicePoint(n, n->preferredChoicePointColor(), n->alternativeChoicePointColor()))
   *      result = ColorOperation::succeeded;
   *    else
   *      result = ColorOperation::failed;
   *  }
   *  return result;
   *
   * \endcode
   *
   * \note 
   *  concrete subclasses shall not override this method. Instead they should
   *  override the hook-method ChoiceOperator::selectChoice
   */
  ColorOpResult operator()();
  ColorOpResult operator()(ColorOpType::Value choiceType);
  
  
  //! Selects a choice point.
  /*! Returns a choice object containing the node to be colored as choice point as well
   *  the choice's preferred and alternative choice color. 
   */
  virtual Choice selectChoice() = 0;

  virtual const Constraint& getConstraint() const = 0;

  const Heuristic& getHeuristic() const {return *heuristic_;}
  Heuristic& getHeuristic() {return *heuristic_;}

  
private:
  Graph* graph_;
  Heuristic* heuristic_;
};

//! SimpleChoiceOperator uses a given constraint and a given heuristic to choose nodes.
/*!
 * The heuristic is used to evaluate those nodes that satisfy the given
 * constraint. The SimpleChoiceOperator then returns the node with the best
 * heuristic value.
 *
 * \par Known Combinations
 *
 *  - C  = SimpleChoiceOperator + NoConstraint: Unconditionally selects both head- and body nodes
 *  - CH = SimpleChoiceOperator + HeadConstraint: Selects only head nodes.
 *  - CB = SimpleChoiceOperator + BodyConstraint: Select only body nodes.
 *  - D  = SimpleChoiceOperator + SupportedBodyConstraint: Selects only supported body nodes.
 *  - DT = SimpleChoiceOperator + SupportedOrTrivialBodyConstraint: Selects only body nodes that are either supported or trivally connected.
 */
class SimpleChoiceOperator : public ChoiceOperator {
public:
  SimpleChoiceOperator(Constraint* theConstraint, const char* name, Graph& g, Heuristic* h);
  ~SimpleChoiceOperator();

  Choice selectChoice();

  const Constraint& getConstraint() const;
  const char* getName() const;
private:
  Constraint* theConstraint_;
  const char* name_;
};

///////////////////////////////////////////////////////////////////////////////
// Basic choice constraints
///////////////////////////////////////////////////////////////////////////////
class NoConstraint : public Constraint {
public:
  //! returns true if n.getColor() == Color::none
  bool isSatisfiedBy(const Node& n) const;
  //! always true
  bool supersedes(const Node&, const Node&) const;
  //! returns Constraint::hybrid_constraint
  Type getType() const;
};

class HeadConstraint : public Constraint {
public:
  //! returns true iff n is an uncolored head node
  bool isSatisfiedBy(const Node& n) const;
  //! always true
  bool supersedes(const Node&, const Node&) const;
  //! returns Constraint::uniform_constraint
  Type getType() const;
};

class BodyConstraint : public Constraint {
public:
  //! returns true iff n is an uncolored body node
  bool isSatisfiedBy(const Node& n) const;
  //! always true
  bool supersedes(const Node&, const Node&) const;
  //! returns Constraint::uniform_constraint
  Type getType() const;
};

class SupportedBodyConstraint : public Constraint {
public:
  //! returns true iff n is an uncolored and supported body node
  bool isSatisfiedBy(const Node& n) const;
  
  /*!
   * returns true iff
   * - n2 is a head node or
   * - n1 is a head node and n2 is a supported body node or
   * - n1 is a supported body node or
   * - n2 is a body node that is not supported
   */
  bool supersedes(const Node& n1, const Node& n2) const;
  //! returns Constraint::uniform_constraint
  Type getType() const;
};

class SupportedOrTrivialBodyConstraint : public Constraint {
public:
  /*!
   * returns true iff n is a uncolored body node that is either supported or
   * part of a trivial component.
   */
  bool isSatisfiedBy(const Node& n) const;
  
  /*!
   * returns true iff
   * - n2 is a head node or
   * - n1 is a head node and n2 is a supported or trivial connected body node or
   * - n1 is a supported or trivial connected body node or
   * - n2 is a body node that is neither supported nor trivial connected.
   */
  bool supersedes(const Node& n1, const Node& n2) const;
  //! returns Constraint::uniform_constraint
  Type getType() const;
  
};

template <class T>
ChoiceOperator* createChoiceOp(Graph&, Heuristic*, T*);

}

#endif
