/***************************************************************************
 *                                                                         *
 *    NoMoRe++                                                             *
 *                                                                         *
 *    Copyright (C) 2003-2005 NoMoRe Developing Group                      *
 *                                                                         *
 *    For more information, see http://www.cs.uni-potsdam.de/nomore/       *
 *    or email to nomore-dg@cs.uni-potsdam.de                              *
 *                                                                         *
 *                                                                         *
 *    This program is free software; you can redistribute it and/or        *
 *    modify it under the terms of the GNU General Public License          *
 *    as published by the Free Software Foundation; either version 2       *
 *    of the License, or (at your option) any later version.               *
 *                                                                         *
 *    This program is distributed in the hope that it will be useful,      *
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of       *
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the        *
 *    GNU General Public License for more details.                         *
 *                                                                         *
 *    You should have received a copy of the GNU General Public License    *
 *                                                                         *
 ***************************************************************************/

#if defined (_MSC_VER) && _MSC_VER <= 1300
#pragma warning (disable : 4786)
#endif

#include <choice_operator.h>
#include <graph.h>
#include <heuristic.h>
#include <node.h>

namespace NS_NOMORE {

ChoiceOperator::ChoiceOperator(Graph& grp, Heuristic* h)
  : Operator(grp)
  , graph_(&grp)
  , heuristic_(h) {
  heuristic_->setGraph(grp);
}

ChoiceOperator::~ChoiceOperator() {
  delete heuristic_;
}

ColorOpResult ChoiceOperator::operator()() {
  return (*this)(ColorOpType::first_choice);
}

ColorOpResult ChoiceOperator::operator()(ColorOpType::Value choiceType) {
  assert(choiceType == ColorOpType::first_choice || choiceType == ColorOpType::next_choice);
  if (Choice c = selectChoice()) {
    return graph_->colorChoicePoint(*c.node(), c.preferredColor(), c.alternativeColor(), choiceType)
      ? ColorOperation::succeeded 
      : ColorOperation::failed;
  }
  return ColorOperation::unchanged;
}

///////////////////////////////////////////////////////////////////////////////
SimpleChoiceOperator::SimpleChoiceOperator(Constraint* theConstraint, const char* name, Graph& g, Heuristic* h)
  : ChoiceOperator(g, h)
  , theConstraint_(theConstraint)
  , name_(name) {

  assert(theConstraint_);
}

SimpleChoiceOperator::~SimpleChoiceOperator() {
  delete theConstraint_;
}

Choice SimpleChoiceOperator::selectChoice() {
  return getHeuristic().selectNode(*theConstraint_);
}

const Constraint& SimpleChoiceOperator::getConstraint() const {
  return *theConstraint_;
}

const char* SimpleChoiceOperator::getName() const {
  return name_;
}

///////////////////////////////////////////////////////////////////////////////
// Basic choice constraints
///////////////////////////////////////////////////////////////////////////////
bool NoConstraint::isSatisfiedBy(const Node& n) const { return n.getColor() == Color::none; }
bool NoConstraint::supersedes(const Node&, const Node&) const { return true; }
Constraint::Type NoConstraint::getType() const { return Constraint::hybrid_constraint;}


bool HeadConstraint::isSatisfiedBy(const Node& n) const { 
  return n.getColor() == Color::none && n.getType() == NodeType::head_node;
}

bool HeadConstraint::supersedes(const Node& l, const Node& c) const {  
  return isSatisfiedBy(l) || !isSatisfiedBy(c);
}

Constraint::Type HeadConstraint::getType() const { return Constraint::uniform_constraint;}

bool BodyConstraint::isSatisfiedBy(const Node& n) const { 
  return n.getColor() == Color::none && n.getType() == NodeType::body_node;
}
bool BodyConstraint::supersedes(const Node&, const Node&) const { return true; }
Constraint::Type BodyConstraint::getType() const { return Constraint::uniform_constraint;}


bool SupportedBodyConstraint::isSatisfiedBy(const Node& n) const { 
  return n.getColor() == Color::none && n.getType() == NodeType::body_node
  && static_cast<const BodyNode&>(n).isSupported();
}

bool SupportedBodyConstraint::supersedes(const Node& n1, const Node& n2) const { 
  if(n2.getType() == NodeType::head_node)
    return true;

  const BodyNode *bn2 = static_cast<const BodyNode*>(&n2);

  if(n1.getType() == NodeType::head_node)
    return !bn2->isSupported();

  const BodyNode *bn1 = static_cast<const BodyNode*>(&n1);

  return bn1->isSupported() || !bn2->isSupported();
}

Constraint::Type SupportedBodyConstraint::getType() const { return Constraint::uniform_constraint;}


Constraint::Type SupportedOrTrivialBodyConstraint::getType() const { return Constraint::uniform_constraint;}

bool SupportedOrTrivialBodyConstraint::isSatisfiedBy(const Node& n) const {
  return n.getColor() == Color::none 
    && n.getType() == NodeType::body_node 
    && (n.trivialComponent() || static_cast<const BodyNode&>(n).isSupported());
}

bool SupportedOrTrivialBodyConstraint::supersedes(const Node& n1, const Node& n2) const {
  if(n2.getType() == NodeType::head_node)
    return true;

  const BodyNode *bn2 = static_cast<const BodyNode*>(&n2);

  if(n1.getType() == NodeType::head_node)
    return !bn2->isSupported() && !bn2->trivialComponent();

  const BodyNode *bn1 = static_cast<const BodyNode*>(&n1);

  return (bn1->isSupported() || bn1->trivialComponent()) || 
        !(bn2->isSupported() || bn2->trivialComponent());
}

///////////////////////////////////////////////////////////////////////////////
// Basic choice operators
///////////////////////////////////////////////////////////////////////////////
template <>
ChoiceOperator* createChoiceOp(Graph& g, Heuristic* h, NoConstraint*) {
  return new SimpleChoiceOperator(new NoConstraint, "C", g, h);
}
template <>
const char* getName(NoConstraint*) { return "C"; }

template <>
ChoiceOperator* createChoiceOp(Graph& g, Heuristic* h, HeadConstraint*) {
  return new SimpleChoiceOperator(new HeadConstraint, "CH", g, h);
}
template <>
const char* getName(HeadConstraint*) { return "CH"; }

template <>
ChoiceOperator* createChoiceOp(Graph& g, Heuristic* h, BodyConstraint*) {
  return new SimpleChoiceOperator(new BodyConstraint, "CB", g, h);
}
template <>
const char* getName(BodyConstraint*) { return "CB"; }

template <>
ChoiceOperator* createChoiceOp(Graph& g, Heuristic* h, SupportedBodyConstraint*) {
  return new SimpleChoiceOperator(new SupportedBodyConstraint, "D", g, h);
}
template <>
const char* getName(SupportedBodyConstraint*) { return "D"; }

template <>
ChoiceOperator* createChoiceOp(Graph& g, Heuristic* h, SupportedOrTrivialBodyConstraint*) {
  g.checkTightness();
  return new SimpleChoiceOperator(new SupportedOrTrivialBodyConstraint, "DT", g, h);
}
template <>
const char* getName(SupportedOrTrivialBodyConstraint*) { return "DT"; }

}
