/***************************************************************************
 *                                                                         *
 *    NoMoRe++                                                             *
 *                                                                         *
 *    Copyright (C) 2003-2005 NoMoRe Developing Group                      *
 *                                                                         * 
 *    For more information, see http://www.cs.uni-potsdam.de/nomore/       *
 *    or email to nomore-dg@cs.uni-potsdam.de                              *
 *                                                                         *
 *                                                                         *
 *    This program is free software; you can redistribute it and/or        *
 *    modify it under the terms of the GNU General Public License          *
 *    as published by the Free Software Foundation; either version 2       *
 *    of the License, or (at your option) any later version.               *
 *                                                                         *
 *    This program is distributed in the hope that it will be useful,      *
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of       *
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the        *
 *    GNU General Public License for more details.                         *
 *                                                                         *
 *    You should have received a copy of the GNU General Public License    *
 *                                                                         * 
 ***************************************************************************/

#if defined (_MSC_VER) && _MSC_VER <= 1300
#pragma warning (disable : 4786)
#define for if (0); else for
#endif
#include <lparsereader.h>
#include "detail/functors.h"
#include "detail/lparse_writer.h"
#include <graph.h>
#include <color.h>

#include <iostream>
#include <set>
#include <memory>
#include <algorithm>

using namespace std;

namespace NS_NOMORE {

namespace {

// represents the body of an lparse rule.
// A body holds three pieces of information:
// * the list of positive atoms it contains
// * the list of negative atoms it contains
// * the heads of the rules this body is element of.
class LparseBody {
public:
  typedef std::set<int> AtomSet;
  typedef AtomSet::const_iterator const_iterator;

  explicit LparseBody(istream& prg);

  // strict weak ordering based on the positive and negative body-atoms.
  friend bool operator<(const LparseBody& lhs, const LparseBody& rhs);

  void addHead(int id) {
    heads_.insert(id);
  }

  bool superfluous(int currentHead) const {
    AtomSet::const_iterator pos = positives_.begin();
    AtomSet::const_iterator end = positives_.end();
    AtomSet::const_iterator negEnd = negatives_.end();
    // set_intersection(b+, b-) != 0
    for (; pos != end; ++pos) {
      if (negatives_.find(*pos) != negEnd) {
        return true;
      }
    }
    return positives_.find(currentHead) != positives_.end();
  }

  const_iterator headsBegin() const {
    return heads_.begin();
  }
  const_iterator headsEnd() const {
    return heads_.end();
  }
  const_iterator positivesBegin() const {
    return positives_.begin();
  }
  const_iterator positivesEnd() const {
    return positives_.end();
  }
  const_iterator negativesBegin() const {
    return negatives_.begin();
  }
  const_iterator negativesEnd() const {
    return negatives_.end();
  }

  AtomSet::size_type bPlusSize() const {
    return positives_.size();
  }

  // returns true, iff this body is supported by the given set of atoms,
  // i.e. all positive atoms of this body are element of atoms.
  bool supportedBy(const AtomSet& atoms) {
    if (positives_.size() > atoms.size()) {
      return false;
    }

    AtomSet::const_iterator it = positives_.begin();
    AtomSet::const_iterator end = positives_.end();
    AtomSet::const_iterator otherIt = atoms.begin();
    AtomSet::const_iterator otherEnd = atoms.end();
    while (it != end && otherIt != otherEnd) {
      if (*it == *otherIt) {
        ++it;
        ++otherIt;
      }
      else if (*otherIt < *it) {
        if ((otherIt = atoms.find(*it)) == otherEnd) {
          return false;
        }
        ++otherIt;
        ++it;
      }
      else {
        return false;
      }
    }
    return it == end;
  }
private:
  AtomSet positives_;
  AtomSet negatives_;
  AtomSet heads_;
};

LparseBody::LparseBody(istream& prg) {
  int bodySize = 0;
  int negBodySize = 0;
  if (!(prg >> bodySize) || !(prg >> negBodySize) || negBodySize > bodySize) {
    throw ReadError("body size error");
  }
  int id = 0;
  for (long i = 0; i != negBodySize; ++i) {
    if (!(prg >> id) || id < 1) {
      throw ReadError("atom out of bounds");
    }
    negatives_.insert(id);
  }
  long plusSize = bodySize - negBodySize;
  for (long i = 0; i != plusSize; ++i) {
    if (!(prg >> id) || id < 1) {
      throw ReadError("atom out of bounds");
    }
    positives_.insert(id);
  }
}

namespace {

bool lessSet(const std::set<int>& lhs, const std::set<int>& rhs) {
	return lhs.size() < rhs.size() || (lhs.size() == rhs.size() && lhs < rhs);
}

}

inline bool operator<(const LparseBody& lhs, const LparseBody& rhs) {
  return lessSet(lhs.positives_, rhs.positives_)
    || (lhs.positives_ == rhs.positives_ && lessSet(lhs.negatives_, rhs.negatives_));
}

} // end anonymous namespace

///////////////////////////////////////////////////////////////////////////////
// class LparseReader::Impl
///////////////////////////////////////////////////////////////////////////////
class LparseReader::Impl {
public:
  Impl()
    : theGraph_(0)
    , stream_(0)
    , currentHead_(-1) {
  }
  ~Impl() {
		for_each(bodies_.begin(), bodies_.end(), DETAIL::DeleteObject());
	}
  Graph& read(istream& in, Graph& g);
private:
  Impl(const Impl&);
  Impl& operator=(const Impl&);

  void readRules();
  void insertBodies();
  void insertBody(LparseBody* b, LparseBody::AtomSet& heads);
  bool addOneArcs(LparseBody* b, BodyNode* bodyNode);
  void processDeferredOneArcs();
  void readSymbolTable();
  void readComputeStatement();
  void readSection(const char* name, Color::ColorValue c);
  inline void extractEndMarker();

  void applyPreOp(BodyNode* bodyNode);

  HeadNode* getHeadNode(long atomId) const {
    if (static_cast<size_t>(atomId) < atomToHeadNode_.size())
      return atomToHeadNode_[atomId];
    return 0;
  }
  HeadNode* addHeadNode(long atomId) {
    HeadNode* h = getHeadNode(atomId);
    if (!h) {
      h = theGraph_->insertHeadNode(atomId);
      if (static_cast<size_t>(atomId) >= atomToHeadNode_.size())
        atomToHeadNode_.resize(atomId + 1, static_cast<HeadNode*>(0));
      atomToHeadNode_[atomId] = h;
    }
    return h;
  }

  Graph* theGraph_;
  istream* stream_;
  int currentHead_;
  typedef set<LparseBody*, DETAIL::LessDeref> BodySet;
  typedef pair<BodyNode*, std::vector<int> > BodyOneArcs;
  typedef std::vector<BodyOneArcs> OneArcRelations;
  typedef std::vector<HeadNode*> HeadNodes;
  HeadNodes atomToHeadNode_;

  OneArcRelations deferredOneArcs_;
  BodySet bodies_;
};
///////////////////////////////////////////////////////////////////////////////
// LparseReader - public interface
///////////////////////////////////////////////////////////////////////////////
LparseReader::LparseReader() {
}

LparseReader::~LparseReader() {
}

Graph& LparseReader::readProgram(std::istream& input, Graph& graph) {
  if (!input) {
    throw ReadError("could not read from input!");
  }
  Impl impl;
  graph.clear();
  return impl.read(input, graph);
}

void LparseReader::writeProgram(std::ostream& os, const Graph& graph) {

	if (!os)
		throw WriteError("could not write to stream!");

  DETAIL::LparseWriter writer;
  writer.writeGraph(os, graph);
  
}
///////////////////////////////////////////////////////////////////////////////
// LparseReader::Impl - public interface
///////////////////////////////////////////////////////////////////////////////
Graph& LparseReader::Impl::read(istream& in, Graph& g) {
  theGraph_ = &g;
	stream_ = &in;
  readRules();
  insertBodies();
  processDeferredOneArcs();
  readSymbolTable();
  readComputeStatement();
  return *theGraph_;
}

///////////////////////////////////////////////////////////////////////////////
// LparseReader::Impl - private helpers
///////////////////////////////////////////////////////////////////////////////
void LparseReader::Impl::readRules() {
  istream& prg = *stream_;
  enum {
    ENDRULE,
    BASICRULE
  };
  int ruleType = ENDRULE; 
  while ((prg >> ruleType) && ruleType == BASICRULE) {
    if (!(prg >> currentHead_) || currentHead_ < 1) {
      throw ReadError("head atom out of bounds");
    }

    auto_ptr<LparseBody> body(new LparseBody(prg));
    if (!body->superfluous(currentHead_)) {
      pair<BodySet::iterator, bool> insPos = bodies_.insert(body.get());
      (*insPos.first)->addHead(currentHead_);
      if (insPos.second) {
        // Body does not yet exists.
        // transfer ownership to the Set...
        body.release();
      }
    }
    // duplicate bodies get automatically deleted here by auto_ptr's dtor.
  }
  if (!prg || ruleType > BASICRULE) {
    throw ReadError("unsupported or unknown rule type");
  }
  extractEndMarker();
}


// from the set of all bodies adds all those bodies to the graph
// that are possibly supported.
// Algorithm:
//  Set heads = {}
//  while bodies not empty
//    Set suppBodies = getBodiesSupportedBy(heads);
//    for each b in suppBodies
//      insertBody(b);
//      heads = unite(heads, b.getHeads());
//    bodies = difference(bodies, suppBodies);
void LparseReader::Impl::insertBodies() {
  LparseBody::AtomSet heads;
  bool keepScanning = true;
  while (keepScanning) {
    keepScanning = false;
    BodySet::iterator it = bodies_.begin(); 
    BodySet::iterator end = bodies_.end();
    while (it != end && (*it)->bPlusSize() <= heads.size()) {
      if ((*it)->supportedBy(heads)) {
        insertBody(*it, heads);
        delete * it;
        bodies_.erase(it++);
        keepScanning = true;
      }
      else {
        // keep scanning forward as long as following bodies can
        // possibly be supported.
        ++it;
      }
    }
    // start over only if last pass changed heads
  }
}

// adds the Body b to the Graph by adding a new body node and inserting
// all predecessor/successor arcs.
//
// adding consists of the following steps:
//  create and insert a new body node bn
//  for each Head h of body b 
//    add h to heads
//    create and insert head node hn (if necessary) 
//    set up successor/predecessor arcs between bn and hn
//  for each atom a in the positive-part of b
//    get head node hn for atom a
//    set up 0-predecessor/0-successor arcs between bn and hn
//  for each atom a in the negative-part of b
//    check if a is already represented as a head hn in the graph
//      if yes set up 0-predecessor/0-successor arcs between bn and hn
//      else remember the one-arc-relation (bn, hn)
void LparseReader::Impl::insertBody(LparseBody* b, LparseBody::AtomSet& heads) {
  BodyNode* bodyNode = theGraph_->insertBodyNode();

  LparseBody::const_iterator end = b->headsEnd();
  for (LparseBody::const_iterator it = b->headsBegin(); it != end; ++it) {
    HeadNode* h = addHeadNode(*it);
    bodyNode->insertSuccessor(*h);    // calls h.insertPredecessor
    heads.insert(*it);
  }

  end = b->positivesEnd();
  for (LparseBody::const_iterator it = b->positivesBegin(); it != end; ++it) {
    HeadNode* h = getHeadNode(*it);
    assert(h && "Missing head for body");
    h->insertZeroSuccessor(*bodyNode);  // calls bodyNode.insertZeroPredecessor
  }

  end = b->negativesEnd();
  
  if (addOneArcs(b, bodyNode)) {
    applyPreOp(bodyNode);
  }
}

bool LparseReader::Impl::addOneArcs(LparseBody* b, BodyNode* bn) {

  BodyOneArcs oneArcs;
  
  LparseBody::const_iterator end = b->negativesEnd();
  for (LparseBody::const_iterator it = b->negativesBegin(); it != end; ++it) {
    if (HeadNode* h = getHeadNode(*it)) {
      h->insertOneSuccessor(*bn);  // calls bodyNode.insertOnePredecessor
    }
    else {
      oneArcs.second.push_back(*it);
    }
  }
  if (!oneArcs.second.empty()) {
    oneArcs.first = bn;
    deferredOneArcs_.push_back(oneArcs);
    return false;
  }
  return true;
}


void LparseReader::Impl::processDeferredOneArcs() {
  
  OneArcRelations::const_iterator end = deferredOneArcs_.end();
  for (OneArcRelations::const_iterator it = deferredOneArcs_.begin(); it != end; ++it) {
    for (std::vector<int>::const_iterator headIt = it->second.begin(); headIt != it->second.end(); ++headIt) {
      if (HeadNode* h = getHeadNode(*headIt)) {
        h->insertOneSuccessor(*it->first);
      }
    }
    applyPreOp(it->first);
  }

}


void LparseReader::Impl::readSymbolTable() {
  istream& prg = *stream_;
  const int len = 1024;
  char name[len], space;
  int id = 0;
  while ((prg >> id) && id != 0 && prg.get(space) && prg.getline(name, len)) {
    if (HeadNode * h = getHeadNode(id)) {
      h->setAtomName(name);
    }
    else {
      // The atom a is not relevant to the program since
      // it does not occur in a supported body
      // Just ignore this atom.
    }
  }
  if (!prg) {
    throw ReadError("atom name too long or end of file");
  } 
  extractEndMarker();
}

void LparseReader::Impl::readComputeStatement() {
  readSection("B+", Color::plus);
  readSection("B-", Color::minus);
}

void LparseReader::Impl::readSection(const char* sec, Color::ColorValue c) {
  istream& prg = *stream_;
  const int len = 1024;
  char name[len];
  if (!prg.getline(name, len) || strcmp(sec, name) != 0) {
    throw ReadError(string(sec).append(" expected!"));
  }
  int id = 0;
  for (; (prg >> id) && id > 0;) {
    HeadNode* h = getHeadNode(id);
    //assert(h && "compute statement contains unknown head!");
    if(h != NULL)
      theGraph_->color(*h, c);
  }
  if (!prg || id < 0) {
    throw ReadError(string("error reading ").append(sec).append(" atoms"));
  }
  extractEndMarker();
}


inline void LparseReader::Impl::extractEndMarker() {
  while (stream_->get() != '\n') {
    ; // nothing to be done, only extract the character
  }
}


void LparseReader::Impl::applyPreOp(BodyNode* bodyNode) {
  if(bodyNode->countBodyAtoms() == 0ul) {
    theGraph_->color(*bodyNode, Color::plus);
  } 
  else if(bodyNode->isSelfBlocker()) {
    theGraph_->color(*bodyNode, Color::minus);
  }
}

} // NS_NOMORE
