/***************************************************************************
 *                                                                         *
 *    NoMoRe++                                                             *
 *                                                                         *
 *    Copyright (C) 2003-2005 NoMoRe Developing Group                      *
 *                                                                         * 
 *    For more information, see http://www.cs.uni-potsdam.de/nomore/       *
 *    or email to nomore-dg@cs.uni-potsdam.de                              *
 *                                                                         *
 *                                                                         *
 *    This program is free software; you can redistribute it and/or        *
 *    modify it under the terms of the GNU General Public License          *
 *    as published by the Free Software Foundation; either version 2       *
 *    of the License, or (at your option) any later version.               *
 *                                                                         *
 *    This program is distributed in the hope that it will be useful,      *
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of       *
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the        *
 *    GNU General Public License for more details.                         *
 *                                                                         *
 *    You should have received a copy of the GNU General Public License    *
 *                                                                         * 
 ***************************************************************************/

#if defined (_MSC_VER) && _MSC_VER <= 1300
#pragma warning (disable : 4786)
#pragma warning (disable : 4503)
#define for if (0); else for
#endif
#include <object_factory.h>
#include <operator_decorator_factory.h>
#include <heuristic.h>
#include <choice_operator.h>
#include <operators/lookahead.h>
#include <constraint.h>
#include <operator_string_parser.h>
#include <operators/sequence.h>
#include <graph.h>
#include <cassert>
#include <set>
#include <stack>
#include <memory>

#define INCLUDED_IN_OBJECT_FACTORY
#include "detail/exported_classes.h"
#undef INCLUDED_IN_OBJECT_FACTORY


namespace NS_NOMORE { namespace {

std::string& trim(std::string& str) {
  std::string::size_type pos = str.find_last_not_of(" \t");
  if (pos != std::string::npos)
    str.erase(pos + 1, std::string::npos);
  pos = str.find_first_not_of(" \t");
  if (pos != 0)
    str.erase(0, pos);
  return str;
}

std::vector<std::string > explode(const std::string& separator, const std::string& str) {
  std::string::size_type len = separator.length();
  std::vector<std::string> result;
  if (len == 0) {
    result.push_back(str);
    return result;
  }

  std::string::size_type startPos = 0;
  std::string::size_type endPos = str.find(separator, startPos);
  while (endPos != std::string::npos) {
    if (endPos - startPos)
      result.push_back(std::string(str, startPos, endPos - startPos));

    startPos = endPos;
    if (startPos != std::string::npos)
      startPos += len;
    endPos = str.find(separator, startPos);
  }
  if (startPos != std::string::npos) {
    result.push_back(std::string(str, startPos, std::string::npos));
  }
  return result;
}

template <class V>
std::vector<std::string>& appendKeys(const std::map<std::string, V>& m, 
                                     std::vector<std::string>& out) {
  typename std::map<std::string, V>::const_iterator it = m.begin();
  typename std::map<std::string, V>::const_iterator end = m.end();
  for (; it != end; ++it)
    out.push_back(it->first);
  return out;
}

std::pair<std::string, std::string> parseLookaheadString(const std::string& str) {
  std::string::size_type posBegin = str.find("[");
  std::string::size_type posEnd = str.find("]");
  if (posBegin == std::string::npos && posEnd == std::string::npos)
    return std::pair<std::string, std::string>(str, "");
  else if (posBegin != std::string::npos && posEnd != std::string::npos)
    return std::pair<std::string, std::string>(std::string(str, 0, posBegin),
      std::string(str, posBegin + 1, posEnd - (posBegin + 1)));
  
  throw FormatError("Illegal String Format! Expected: LOp[DetOp]");
}

///////////////////////////////////////////////////////////////////////////////
// class DetOpParser
///////////////////////////////////////////////////////////////////////////////
class DetOpParser : public OperatorStringParser {
public:
  DetOpParser(Graph& g, Operator* opSet, const ObjectFactory* opFactory)
	: opSet_(opSet)
    , factory_(opFactory)
    , graph_(g)
    , result_(0)
    , ownsResult_(false) {	
  }
  Operator* getResult();
  std::set<std::string>& getOperatorNames() {
    return opNames_;
  }
  void clearResult() {
    if (ownsResult_)
      delete getResult();
    result_ = 0;
    while (!seqs_.empty()) seqs_.pop();
    opNames_.clear();
  }

  bool ownsResult() const {
    return ownsResult_;
  }
private:
  void startSequence();
  void endSequence(bool closure);
  void addOperator(const std::string& name);
  Operator* opSet_;
  const ObjectFactory* factory_;
  Graph& graph_;
  Operator* result_;
  std::stack<Sequence*> seqs_;
  bool ownsResult_;
  std::set<std::string> opNames_;
  
};

void DetOpParser::startSequence() {
  seqs_.push(new Sequence);
  ownsResult_ = true;
}

void DetOpParser::endSequence(bool cl) {
  Sequence* top = seqs_.top();
  top->setClosure(cl);
  if (seqs_.size() > 1) {
    seqs_.pop();
    seqs_.top()->add(top);
  }
  result_ = seqs_.top();
}

void DetOpParser::addOperator(const std::string& name) {
  if (opNames_.insert(name).second) {
    Operator* o = opSet_ ? opSet_->extractOperator(name) : 0;
    bool newOp = false;
    if (!o) {
      o = factory_->createOperator(graph_, name).release();
      newOp = true;
    }
    if (seqs_.empty()) {  
      result_ = o;
      ownsResult_ = newOp;
    }
    else {
      seqs_.top()->add(o, newOp);
    }
  }
}

Operator* DetOpParser::getResult() {
  while (seqs_.size() > 1) {
    Sequence* top = seqs_.top();
    seqs_.pop();
    seqs_.top()->add(top);
    result_ = seqs_.top();
  }
  return result_;
}

} // end anonymous-namespace

///////////////////////////////////////////////////////////////////////////////
// ObjectFactory::Operators
///////////////////////////////////////////////////////////////////////////////
ObjectFactory::Operators::Operators()
{}

ObjectFactory::Operators::Operators(const Operators& other)
  : detOp_(other.detOp_.release())
  , lookOp_(other.lookOp_.release())
  , choiceOp_(other.choiceOp_.release())
{}
    
ObjectFactory::Operators::~Operators() {
  /* intentionally left blank */
}

///////////////////////////////////////////////////////////////////////////////
// ObjectFactory
///////////////////////////////////////////////////////////////////////////////
ObjectFactory::ObjectFactory() 
  : opDecorator_(new OperatorDecoratorFactory())
  , owner_(true) {
}
  
ObjectFactory::ObjectFactory(OperatorDecoratorFactory* opd) 
  : opDecorator_(opd)
  , owner_(true) {
  assert(opDecorator_);
}

ObjectFactory::ObjectFactory(OperatorDecoratorFactory& opd)
  : opDecorator_(&opd)
  , owner_(false) {

}

ObjectFactory::~ObjectFactory() {
  if (owner_)
    delete opDecorator_;
}

void ObjectFactory::setOperatorDecoratorFactory(OperatorDecoratorFactory& df) {
  if (owner_)
    delete opDecorator_;
  opDecorator_ = &df;
  owner_ = false;
}

void ObjectFactory::setOperatorDecoratorFactory(OperatorDecoratorFactory* df) {
  if (owner_)
    delete opDecorator_;
  opDecorator_ = df;
  owner_ = true;
}

bool ObjectFactory::registerClass(const OpInfo& info) {
  return detOps_.insert(
    OpMap::value_type(info.name, std::make_pair(info.creator, info.description))
  ).second;
}

bool ObjectFactory::registerClass(const LookOpInfo& info) {
  return lookaheadOps_.insert(
    LookOpMap::value_type(info.name, std::make_pair(info.creator, info.description))
  ).second;
}

bool ObjectFactory::registerClass(const ChoiceOpInfo& info) {
  return choiceOps_.insert(
    ChoiceOpMap::value_type(info.name, std::make_pair(info.creator, info.description))
  ).second;
}


bool ObjectFactory::registerClass(const HeuristicInfo& info) {
  return heuristics_.insert(
    HeuristicsMap::value_type(info.name, std::make_pair(info.creator, info.description))
  ).second;
}


void ObjectFactory::registerKnownClasses() {
  detOps_s.registerClasses(*this);
  lookaheadOps_s.registerClasses(*this);
  choiceOps_s.registerClasses(*this);
  heuristics_s.registerClasses(*this);
}


std::auto_ptr<Operator> ObjectFactory::createOperator(Graph& g, const std::string& name) const {
  OpMap::const_iterator it = detOps_.find(name);
  if (it != detOps_.end())  
    return std::auto_ptr<Operator>(opDecorator_->decorate( it->second.first(g) ));
  throw UnknownObject(name, ObjectFactory::DET_OP);
}

std::auto_ptr<Operator> ObjectFactory::createLookaheadOperator(Graph& g, const std::string& name) const {
  LookOpMap::const_iterator it = lookaheadOps_.find(name);
  if (it != lookaheadOps_.end())  
    return std::auto_ptr<Operator>(opDecorator_->decorate( it->second.first(g) ));
  throw UnknownObject(name, ObjectFactory::LOOKAHEAD_OP);
}

std::auto_ptr<Operator> ObjectFactory::createChoiceOperator(Graph& g, const std::string& name, const std::string& heu) const{
  std::auto_ptr<Heuristic> h(createHeuristic(g, heu));
  ChoiceOpMap::const_iterator it = choiceOps_.find(name);
  if (it != choiceOps_.end())
    return std::auto_ptr<Operator>(opDecorator_->decorate( it->second.first(g, h.release()) ));
  throw UnknownObject(name, ObjectFactory::CHOICE_OP);
}

std::auto_ptr<Heuristic> ObjectFactory::createHeuristic(Graph& g, const std::string& name) const {
  HeuristicsMap::const_iterator it = heuristics_.find(name);
  if (it != heuristics_.end()) {   
    std::auto_ptr<Heuristic> h( it->second.first() );
    return h;
  }
  throw UnknownObject(name, ObjectFactory::HEURISTIC);
}

ObjectFactory::Operators ObjectFactory::createOperators(Graph& g, const std::string& objStr, bool validate) const {
  enum { CHOICE, DET, LOOK, HEU };
  std::vector<std::string> str = explode(":", objStr);
  const unsigned int parts = 4;
  if (str.size() != parts)
    throw FormatError("Illegal String Format!\n Expected: ChoiceOp:DetOp:LOp[DetOp]:Heuristic");
  
  for (std::vector<std::string>::size_type i = 0; i != parts; ++i) {
    if (trim(str[i]).empty())
      throw FormatError("Illegal String Format!\n Expected: ChoiceOp:DetOp:LOp[DetOp]:Heuristic");
  }
  Operators o;
  o.choiceOp_ = createChoiceOperator(g, str[CHOICE], str[HEU]);
  
  DetOpParser parser(g, 0, this);
  try {
    parser.parse(str[DET]);
  }
  catch(const std::runtime_error&) {
    parser.clearResult();
    throw;
  }
  o.detOp_ = std::auto_ptr<Operator>(parser.getResult());
  if (validate) {
    std::set<std::string>& ops = parser.getOperatorNames();
    ops.insert(o.choiceOp_->getName());
    if (!o.choiceOp_->validate(ops) || !o.detOp_->validate(ops)
      || o.detOp_->extractOperator("P") == 0) // P-Operator is required!
      throw InvalidOperatorCombination();
  }
  
  o.lookOp_ = getLookahead(g, o.detOp_, o.choiceOp_, str);

  // !!!! HACK HACK HACK !!!
  static_cast<ChoiceOperator*>(o.choiceOp_->extractOperator(str[CHOICE]))->getHeuristic().setPropagationOperator(*o.detOp_);
  return o;
}

std::auto_ptr<Operator> ObjectFactory::getLookahead(Graph& g,
                                                    const std::auto_ptr<Operator>& detOp,
                                                    const std::auto_ptr<Operator>& choice,
                                                    const std::vector<std::string>& opStrings) const {
  enum { CHOICE, DET, LOOK, HEU };
  if (opStrings[LOOK] == "None")
    return std::auto_ptr<Operator>(0);
  
  std::pair<std::string, std::string> lopPair = parseLookaheadString(opStrings[LOOK]);
  std::auto_ptr<Operator> lop = createLookaheadOperator(g, lopPair.first);
  
  // safe since createLookaheadOperator creates only Lookahead-Ops.
  // But since the operator is potentially decorated we can't downcast
  // the returned operator directly. Instead we must access the lookahead
  // operator indirectly using extractOperator.
  Lookahead* l = static_cast<Lookahead*>(lop->extractOperator(lopPair.first));
  
  // same as above
  ChoiceOperator* cp = static_cast<ChoiceOperator*>(choice->extractOperator(opStrings[CHOICE]));
  l->setConstraint(cp->getConstraint());
  l->setHeuristic(cp->getHeuristic());
  
  

  if (lopPair.second == "" || lopPair.second == opStrings[DET]) {
    l->setPropagationOperator(*detOp);    
    return lop;
  }
  else {
    DetOpParser parser(g, detOp.get(), this);
    try {
      parser.parse(lopPair.second);
    }
    catch(...) {
      parser.clearResult();
      throw;
    }
    l->setPropagationOperator(parser.getResult(), parser.ownsResult());
    return lop;
  }
}

std::vector<std::string> ObjectFactory::getRegisteredClasses(ObjectType ot) const {
  std::vector<std::string> res;
  if ( (ot & DET_OP) ) {
    appendKeys(detOps_, res);
  }
  if ( (ot & LOOKAHEAD_OP) ) {
    appendKeys(lookaheadOps_, res);
  }
  if ( (ot & CHOICE_OP) ) {
    appendKeys(choiceOps_, res);
  }
  if ( (ot & HEURISTIC) ) {
    appendKeys(heuristics_, res);
  }
  return res;
}

std::string ObjectFactory::getDescription(const std::string& name) const {
  OpMap::const_iterator op = detOps_.find(name);
  if (op != detOps_.end()) {
    return op->second.second;
  }
  
  LookOpMap::const_iterator lop = lookaheadOps_.find(name);
  if (lop != lookaheadOps_.end()) {
    return lop->second.second;
  }
  
  ChoiceOpMap::const_iterator cop = choiceOps_.find(name);
  if (cop != choiceOps_.end()) {
    return cop->second.second;
  }
  
  HeuristicsMap::const_iterator heu = heuristics_.find(name);
  if (heu != heuristics_.end()) {
    return heu->second.second;
  }
  throw UnknownObject(name, ALL_OBJECTS);
}

}

