/***************************************************************************
 *                                                                         *
 *    NoMoRe++                                                             *
 *                                                                         *
 *    Copyright (C) 2003-2005 NoMoRe Developing Group                      *
 *                                                                         * 
 *    For more information, see http://www.cs.uni-potsdam.de/nomore/       *
 *    or email to nomore-dg@cs.uni-potsdam.de                              *
 *                                                                         *
 *                                                                         *
 *    This program is free software; you can redistribute it and/or        *
 *    modify it under the terms of the GNU General Public License          *
 *    as published by the Free Software Foundation; either version 2       *
 *    of the License, or (at your option) any later version.               *
 *                                                                         *
 *    This program is distributed in the hope that it will be useful,      *
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of       *
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the        *
 *    GNU General Public License for more details.                         *
 *                                                                         *
 *    You should have received a copy of the GNU General Public License    *
 *                                                                         * 
 ***************************************************************************/

#ifndef NOMORE_MAX_SUPPORT_H
#define NOMORE_MAX_SUPPORT_H
#include <operators/default_local_propagator.h>
#include <color.h>
#include <util/hash_set.h>
#include <vector>
#include <node.h>
namespace NS_NOMORE {

//! Generates the maximal support graph and colors all nodes not in this graph to minus.
/*! 
 *  This propagator represents the Operator-combination PBU of the noMoRe++ 
 *  system. First the local propagation decides all nodes (which can be 
 *  decided by the local propagation) and runs the unfounded set operator 
 *  which works as described:
 * 
 *	The maximal support graph contains all plus colored nodes as well as all 
 *  nodes that can be reached from these nodes via zero and unlabeled arcs 
 *  and that are not colored minus.
 *
 *  This version of the U-Operator works in several steps starting with nodes 
 *  that were colored to minus between the last and the current call to the 
 *  U-Operator.
 *
 *  <b>Step 1:</b> Get all possibly unsupported nodes.<br>
 *  The operator starts with the minus colored nodes and collects all nodes 
 *  that can be reached via zero and unlabeled arcs that are not colored 
 *  Color::minus. The resulting nodes are called <em>candidates</em>.
 *
 *  <b>Step 2:</b> Determine which node is unsupported and which node is not.<br>
 *  For each node n in cs (where cs is the set of candidates from step 1):
 *  -# if n is a body node and no 0-predecessor (not colored minus) is in cs 
 *     then n is deleted from cs.
 *  -# if n is a head node and at least one predecessor node (not colored 
 *     minus) is not in cs then n is deleted from cs
 *  This algorithm repeatedly iterates over cs until no more nodes are removed.
 *
 *  <b>Step 3:</b> Color all remaining nodes in cs to minus.
 *
 *  \par preconditions: \n
 *  The partial assignment is valid. */
class UnfoundedSet : public DefaultLocalPropagator {
public:
  //! Sets all member variables and registers at graph for events.
  /*! 
   *  \param grp The graph the propagator should work on. */ 
  explicit UnfoundedSet(Graph& grp);

  //! Starts the full propagation (local propagation and unfounded set search)
	/*!
	 *  In addition to the normal local propagation this method calculates the 
   *  max support graph (MSG) and colors all nodes not in MSG to minus.
   *
	 *  \return The value true is returned if the propagator colored one node. 
   *          If no node can be colored the method returns false.
	 *  \note If the graph does not contain any non-trivial strong components 
   *        (i.e. the propgram is tight) propagateFull is equivalent to 
   *        propagateLocal since tight programs can never contain loops which 
   *        are unfounded. */
   virtual bool propagateFull();

	//! Resets all members.
  virtual void reset();

  //! Returns the name of the operator.
  /*!
   *  \return Returns the name of the propagator. */
	virtual const char* getName() const;

  //! Handles the GraphConstructionFinished event.
  /*! 
   *  Checks for tightness and enables or disables the unfounded set search.
   *
   *  \param e The raised event. */
  void handle(const GraphConstructionFinished& e);

  //! Handles a HeadNodeColored event fired.
  /*!
   *  \param e The raised event. */
  void handle(const HeadNodeColored& e);

  //! Handles a BodyNodeColored event fired.
  /*!
   *  \param e The raised event. */
  void handle(const BodyNodeColored& e);

  //! Handles a HeadNodeColoredChoicePoint event.
  /*!
   *  \param e The raised event. */
  void handle(const HeadNodeColoredChoicePoint& e);

  //! Handles a BodyNodeColoredChoicePoint event.
  /*!
   *  \param e The raised event. */
  void handle(const BodyNodeColoredChoicePoint& e);

  //! Returns the state of the unfounded set finder.
  /*!
   *  \return Returns true if the propagator has disabled itself because the 
   *          graph does not contain any non-trivial strong components. */
  bool isDisabled() const;

private:
  // type defs
  typedef util::hash_set<Node*, HashHelper, HashHelper> NodeSet;
  typedef std::vector<HeadNode*> HeadNodes;
  typedef std::vector<BodyNode*> BodyNodes;
  
  // instance variables
  HeadNodes headNodes_;
  BodyNodes bodyNodes_;
  NodeSet candidateSet_;
  bool disabled_;
    
  // instance methods
  
  //! Search all reachable node.
  /*!
   *  Adds to the candidate set all successor nodes (not colored Color::Plus) 
   *  that are reachable from the minus colored nodes.
   * 
   *  Implements step 1 of the abstract algorithm. */
  inline void addReachableNodes();
  template <class S>
  void addReachableNodes(S& s, S& t) {
    for (typename S::size_type i = 0; i != s.size(); ++i) {
      if (s[i]->getColor() != Color::plus && candidateSet_.insert(s[i]).second) {
        t.push_back(s[i]);
        addSuccessors(*s[i]);
      }
    }
    s.clear();
  }
  
	//! Removes possible supported nodes.
  /*!
   *  Removes from the candidate set those nodes that are possibly supported.
   * 
   *  Implements step 2 of the abstract algorithm. */
  inline void removePossiblySupportedNodes();  
  template <class S>
  void removePossiblySupportedNodes(S& s) {
    for (typename S::size_type i = 0; i != s.size(); ++i) {
      if (isPossiblySupported(s[i])) {
        candidateSet_.erase(s[i]);
        addSuccessors(*s[i]);
      }
    }
    s.clear();
  }
  
  //! Colors all nodes in the candidate set to minus.
  void colorCandidates();
  
  //! The possible supported state of a node.
  /*!
   *  Returns true if either one of node's predecessors <b>is not</b> in the 
   *  canidate set or <b>is not</b> colored Color::minus. */
  inline bool isPossiblySupported(HeadNode* node) const;
  
  //! The possible supported state of a node.
  /*!
   *  Returns true if none of node's zero-predecessors is in the candidate set. */
  inline bool isPossiblySupported(BodyNode* node) const;

  inline void addSuccessors(const HeadNode& n);  
  inline void addSuccessors(const BodyNode& n);
  inline void resetImpl();
};

}

#endif
