/***************************************************************************
 *                                                                         *
 *    NoMoRe++                                                             *
 *                                                                         *
 *    Copyright (C) 2003-2005 NoMoRe Developing Group                      *
 *                                                                         * 
 *    For more information, see http://www.cs.uni-potsdam.de/nomore/       *
 *    or email to nomore-dg@cs.uni-potsdam.de                              *
 *                                                                         *
 *                                                                         *
 *    This program is free software; you can redistribute it and/or        *
 *    modify it under the terms of the GNU General Public License          *
 *    as published by the Free Software Foundation; either version 2       *
 *    of the License, or (at your option) any later version.               *
 *                                                                         *
 *    This program is distributed in the hope that it will be useful,      *
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of       *
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the        *
 *    GNU General Public License for more details.                         *
 *                                                                         *
 *    You should have received a copy of the GNU General Public License    *
 *                                                                         * 
 ***************************************************************************/
#ifndef NOMORE_SOLVER_H
#define NOMORE_SOLVER_H

#include <node.h>
#include <memory>
#include <utility>
#include <propagator.h>

namespace NS_NOMORE {

class ChoiceOperator;
class Lookahead;

class Printer;
class Graph;

//! Base class for Nomore's answer set solvers.
/*! 
 *  Starts the answer set searching process using different backtracking and 
 *  selection strategies implemented in sub classes. Set operators for 
 *  propagation and printer for answer set output. */
class Solver {
public:
  //! The default constructor initializes all members to 0.
  /*!
   *  \note This constructor constructs a not yet usable object. To be usable 
   *        one first needs to set operators and a printer. */
  Solver();

  //! Destroys the object.
  virtual ~Solver();

  //! Initializes the objects within the solver instance.
  /*! 
   *  \param as      Maximal number of answer sets to compute (0 = all answer 
   *                 sets).
   *  \param prop    Propagator for the propagation process.
   *  \param look    Lookahead operator for propagation (0 to indicate no 
   *                 lookahead).
   *  \param choice  Choice operator for selecting branching nodes.
   *  \param printer Printer instance to be used for printing answer sets.
   *
   *  \pre Pointer arguments are either not null or point to objects that 
   *       were allocated using new or 0.
   *
   *  \note The Solver takes over ownership of the objects. */
  Solver(int as, Propagator* prop, Lookahead* look,  ChoiceOperator* choice, Printer* printer);

  //! Sets the number of answer sets to compute in the computation process.
  /*!
   *  \param as The number of answer sets to be compute.
   *  \pre as must be greater or equal than 0. */
  void setNrOfAnswerSetsToCompute(int as);
  
  //! Returns the number of answer sets to compute in the computation process.
  /*!
   *  \return The number of answer sets to be compute. */
  int  getNrOfAnswerSetsToCompute() const;
  
  //! Sets the propagator for the propagation to extends the partial assignment.
  /*! 
   *  \param prop The propagator object to be set.
   *  \pre Prop is not null and points to an object that was allocated 
   *       using new. */
  void setPropagator(Propagator* prop);

  //! Returns the object for the propagation.
  /*! 
   *  \return The propagator object. */
  const Propagator* getPropagator() const;

  //! Sets the lookahead operator for looking ahead operations.
  /*! 
   *  \param l The lookahead operator object to be set. */
  void setLookahead(Lookahead* l);

  //! Returns the lookahead operator for looking ahead operations.
  /*! 
   *  \return The lookahead operator object. */
  const Lookahead* getLookahead() const;
 
  //! Sets the choice operator to be used by the solver.
  /*!
   *  \param choice The choice operator object.
   *  \pre Choice is not null and points to an object that was allocated 
   *       using new. */
  void setChoiceOperator(ChoiceOperator* choice);

  //! Returns the choice operator to be used by the solver.
  /*!
   *  \return The choice operator object. */
  ChoiceOperator* getChoiceOperator() const;

  //! Sets the printer to be used by the solver.
  /*!
   *  \param printer The printer object used in the solver.
   *  \pre Printer is not null and points to an object that was allocated 
   *       using new. */
  void setAnswerSetPrinter(Printer* printer);

  //! Solves the logic program represented by the given graph using the set operators.
  /*! 
   *  Finds all answer sets (set by constructor or setNrOfAnswerSetsToCompute()) 
   *  using operators set by setPropagator() and prints the answer set to the 
   *  printer object set by setAnswerSetPrinter().
   *
   *  \pre Operators for propagation and choosing were set. 
   *  \pre A printer for printing answer sets were set.
   *
   *   The solver uses the following algorithm:
   *   \code
   *     error := false;         // - no more answer sets?
   *     while(!error) {
   *       try {
   *         propagate();      // with lookahead
   *         if(graph is totally colored) {
   *           print_answer_set();
   *           // error is true if no choice can be recolored -> no more answer 
   *           // sets
   *           error := backtrack_and_recolor();
   *         } else {
   *           // graph is not total colored -> select a choice point and color
   *           if(!select()) {
   *             // no choice point is found -> execute the post processing step
   *             post();
   *           }
   *         }
   *       } catch(ColorError) {
   *         // catches all color error of the propagator, post operator and 
   *         // selector and try the backtrack_and_recolor() method
   *         error = backtrack_and_recolor();
   *       }
   *     }
   *   \endcode
   * 
   *   \return False if all answer sets were computed. True if possibly more 
   *           answer sets could be computed. */
  bool solve(Graph& grp);

  //! Returns the number of answer sets found during computation.
  /*!
   *  \return The number of answer sets found. */
  int getNrOfFoundAS() const;
protected:
  // abstract methods
  virtual bool backtrack(Graph& g) = 0;
  virtual bool choose(Graph& g) = 0;

  // instance methods
  inline bool propagate();
  
  // instance varianles

  //! Number of answer sets to solve (0=all answer sets).
  mutable int as_;
  //! Pointer to the deterministic operator (or sequence containing several operators).
  mutable std::auto_ptr<Propagator> det_;
  //! Pointer to the lookahead operator (or sequence containing several operators).
  mutable std::auto_ptr<Lookahead> lookahead_;
  //! Pointer to the choice operator to select a choice point.
  mutable std::auto_ptr<ChoiceOperator> choice_;
  //! User defined printer object for printing the answer sets.
  mutable std::auto_ptr<Printer> printer_;
  
private:
  // deny constructors
  Solver(const Solver& other);
};

}

#endif
