#include <cppunit/TestFixture.h>
#include <cppunit/TestAssert.h>
#include <cppunit/extensions/HelperMacros.h>
#include <object_factory.h>
#include <graph.h>
#include <operators/hybrid_lookahead.h>
#include <operators/no_unfounded_check.h>
#include <choice_operator.h>
#include <heuristic.h>
#include <typeinfo>
#include <algorithm>
using namespace NS_NOMORE;

namespace NS_NOMORE_TESTS {

class TestObjectFactory : public CppUnit::TestFixture {

  CPPUNIT_TEST_SUITE(TestObjectFactory);
  CPPUNIT_TEST(testEmpty);
  CPPUNIT_TEST(testRegisterPropagator);
  CPPUNIT_TEST(testCreatePropagator);
  CPPUNIT_TEST(testCreateChoiceOperator);

  CPPUNIT_TEST(testRegisterKnownClasses);

  CPPUNIT_TEST(testCreateObjects);
  
  CPPUNIT_TEST(testCreateWithLookahead);
  CPPUNIT_TEST_SUITE_END();

  static Propagator* createProp(Graph& g) {
    return new NoUnfoundedCheck(g);
  }
  static ChoiceOperator* createDOperator(Graph& g, Heuristic* h) {
    return new SimpleChoiceOperator(new SupportedBodyConstraint, "D", g, h);
  }
  static Heuristic* createSelectFirst() {
    return new SelectFirst();
  }
  static Lookahead* createHLAOperator(Graph& g) {
    return new HybridLookahead(g);
  }
public:
  void setUp() {
    factory_ = new ObjectFactory();
  }
  void tearDown() {
    delete factory_;
  }

  void testEmpty() {
    std::vector<std::string> o = factory_->getRegisteredClasses(ObjectFactory::ALL_OBJECTS);
    CPPUNIT_ASSERT_EQUAL(true, o.empty());

    CPPUNIT_ASSERT_THROW(factory_->getDescription("Foo"), UnknownObject);
  }

  void testRegisterPropagator() {
    std::string pName = getName<NoUnfoundedCheck>(0);
    factory_->registerClass(ObjectFactory::PropInfo(pName, createProp));
		std::vector<std::string> o = factory_->getRegisteredClasses(ObjectFactory::PROPAGATOR);
    CPPUNIT_ASSERT_EQUAL(size_t(1), o.size());
    CPPUNIT_ASSERT_EQUAL(pName, o[0]);
    std::string empty = "";
    CPPUNIT_ASSERT_EQUAL(empty, factory_->getDescription(pName));
  }

  void testCreatePropagator() {
    std::string pName = getName<NoUnfoundedCheck>(0);
    factory_->registerClass(ObjectFactory::PropInfo(pName, createProp));
    Graph g;
    std::auto_ptr<Propagator> p(factory_->createPropagator(g, pName));
    CPPUNIT_ASSERT_EQUAL(pName, std::string(p->getName()));
    CPPUNIT_ASSERT( typeid(NoUnfoundedCheck) == typeid( *p ));

    CPPUNIT_ASSERT_THROW(factory_->createPropagator(g, "Foo"), UnknownObject);
  }

  void testCreateChoiceOperator() {
    std::string suppName = getName<SupportedBodyConstraint>(0);
    CPPUNIT_ASSERT_EQUAL(true, 
      factory_->registerClass(ObjectFactory::ChoiceOpInfo(suppName,  createDOperator))
    );
    CPPUNIT_ASSERT_EQUAL(true, 
      factory_->registerClass(ObjectFactory::HeuristicInfo("SF", createSelectFirst))
    );
    Graph g;
    std::auto_ptr<ChoiceOperator> d(factory_->createChoiceOperator(g, suppName, "SF"));
    CPPUNIT_ASSERT_EQUAL(suppName, std::string(d->getName()));
    CPPUNIT_ASSERT( dynamic_cast<SimpleChoiceOperator*>(d.get()) != 0 );
    CPPUNIT_ASSERT(typeid(SelectFirst) == typeid( d->getHeuristic() ));

    CPPUNIT_ASSERT_THROW(factory_->createChoiceOperator(g, suppName, "Foo"), UnknownObject);
    CPPUNIT_ASSERT_THROW(factory_->createChoiceOperator(g, "Foo", "SF"), UnknownObject);
  }
  
  void testRegisterKnownClasses() {
    factory_->registerKnownClasses();
    std::vector<std::string> classes = factory_->getRegisteredClasses(ObjectFactory::ALL_OBJECTS);
    using std::find;
    CPPUNIT_ASSERT(find(classes.begin(), classes.end(), "PB") != classes.end());
    CPPUNIT_ASSERT(find(classes.begin(), classes.end(), "PBL") != classes.end());
    CPPUNIT_ASSERT(find(classes.begin(), classes.end(), "PBU") != classes.end());

    CPPUNIT_ASSERT(find(classes.begin(), classes.end(), "D") != classes.end());
    CPPUNIT_ASSERT(find(classes.begin(), classes.end(), "C") != classes.end());

    CPPUNIT_ASSERT(find(classes.begin(), classes.end(), "HybridLA") != classes.end());
    CPPUNIT_ASSERT(find(classes.begin(), classes.end(), "BodyLA") != classes.end());
    CPPUNIT_ASSERT(find(classes.begin(), classes.end(), "SRand") != classes.end());
    CPPUNIT_ASSERT(find(classes.begin(), classes.end(), "DRand") != classes.end());
  }

  void testCreateObjects() {
    Graph g;
    factory_->registerKnownClasses();
		ObjectFactory::Operators o = factory_->createObjects(g, "PBL", "None", Propagator::local, "D", "None");
		CPPUNIT_ASSERT(o.detOp_->getName() == std::string("PBL"));
		CPPUNIT_ASSERT(o.choiceOp_->getName() == std::string("D"));
		CPPUNIT_ASSERT(o.lookOp_.get() == 0);
  }

  void testCreateWithLookahead() {
    Graph g;
    factory_->registerKnownClasses();
    ObjectFactory::Operators o = factory_->createObjects(g, "PBL", "HLA", Propagator::local, "D", "None");
		CPPUNIT_ASSERT(o.lookOp_.get() != 0);
  }
private:
  ObjectFactory* factory_;
};

CPPUNIT_TEST_SUITE_REGISTRATION(TestObjectFactory);

}	// end namespace NS_NOMORE_TESTS
