/***************************************************************************
                          cndoperator.h  -  description
                             -------------------
    begin                : Mon Aug 23 2004
    copyright            : (C) 2004 by nomore-dg
    email                : nomore-dg@cs.uni-potsdam.de
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

/*! \file cndoperator.h
		Declaration of methode ColorGraph and class CNdOperator,
    CNdOperatorC, CNdOperatorC1, CNdOperatorD and CNdOperatorDtodo
*/

#ifndef CNDOPERATOR_H
#define CNDOPERATOR_H

#include "cgraph.h"
#include "coperator.h"
#include "misc.h"

namespace NS_NOMORE{

/*! \class CNdOperator algorithm.h
		\brief Interface for defining non-deterministic operators like CNdOperatorC and CNdOperatorD.

		CNdOperator is the base class for all non-deterministic operators. All these operators
    must implement the ChooseNode() methode.
*/
class CNdOperator : virtual public COperator {
	//! Output operator
	friend std::ostream& operator<<(std::ostream& os, CNdOperator& op);

public:
  //! constructor
	CNdOperator();

  //! destructor
	virtual ~CNdOperator();

  //! start the answer set computation with deterministic operator
  /*! \param det_op Operator which describe the deterministic part
      \param graph Graph which should be colored for computing answer sets (last coloring is returned)
      \param as_count Number of answer sets to search (0 means all answer sets)
      \param sets Set of TAtomSet which represent all found answer sets
      \return returns true if all answer sets found w.r.t. as_count */
  virtual bool Call(CDetOperator* det_op, CDetOperator* end_op, CGraph* graph, int as_count, TSetOfAtomSets* sets);

  //! restores the graph to the last choice point which is colored plus and can now colored minus
  /*! \param graph Specifies the graph which nodes are recolored
      \param stack Defines the stack which holds last changes
      \return if no such choice point is found the methode returns false else true*/
  virtual bool RestoreGraph(CGraph* graph, TStack* stack);

  //! writes the answer set to stdout and put the answer set to the output set of answer sets
  /*! \param nr Number of current answer set
      \param graph Graph which contain of a total coloring
      \param sets Set of answer set for inserting the current answer set */
  virtual void WriteAnswerSet(int nr, CGraph* graph, TSetOfAtomSets* sets);

  //! writes the preferred answer set to stdout and put the answer set to the output set of answer sets
  /*! \param nr Number of current answer set
      \param graph Graph which contain of a total coloring
      \param sets Set of answer set for inserting the current answer set */
  virtual void WritePrefAnswerSet(int nr, CGraph* graph, TSetOfAtomSets* sets);
  
  //! recolores the choice point as the top of the input stack
  /*! \param graph Graph on which one choice node is colored from plus to minus
      \param stack The top element is the last plus-colored choice point for recoloring
      \return if the top element is not a plus-colored choice point the methode returns false else true */
  virtual bool RecolorChoicePoint(CGraph* graph, TStack* stack);

  //! choose a rule and return its id for choice point
  /*! \param graph Graph contains node which are used for finding a node
      \return the pointer of the chosen node, if no node is chosen the methode returns NULL */
  virtual CNode* ChooseNode(CGraph* graph) = 0;

  virtual TType GetType() = 0;
};


/*! \class CNdOperatorC algorithm.h
		\brief Implements the non-deterministic operator C of the paper.

		CNdOperatorC is the implementation of the choice operator of the paper.
    The ChooseNode() methode returns the first uncolored node of the input
    graph.
*/
class CNdOperatorC : public CNdOperator {
public:
  //! constructor
	CNdOperatorC();

  //! destructor
	virtual ~CNdOperatorC();

  //! choose the first uncolored node of the graph
  /*! \param graph Graph contains node which are used for finding a node
      \return the pointer of the chosen node, if no node is chosen the methode returns NULL */
  virtual CNode* ChooseNode(CGraph* graph);

  //! returns an operator specific id which is used for testing of equality
  virtual TType GetType() {
    return type_coperator_c;
  }
};

/*! \class CNdOperatorCtodo algorithm.h
		\brief Implements the non-deterministic operator C of the paper.

		CNdOperatorCtodo is the implementation of the choice operator of the paper.
    The ChooseNode() methode returns the first uncolored node of the input
    graph.
*/
class CNdOperatorCtodo : public CNdOperatorC {
public:
  //! constructor
	CNdOperatorCtodo();

  //! destructor
	virtual ~CNdOperatorCtodo();

  //! returns an operator specific id which is used for testing of equality
  virtual TType GetType() {
    return type_coperator_ctodo;
  }
};


/*! \class CNdOperatorC1 algorithm.h
		\brief Implements the non-deterministic slightly improved operator C of the paper.

		CNdOperatorC1 is the implementation of the choice operator of the paper.
    The ChooseNode() methode returns the first node of the uncolored list
    of the graph.
*/
class CNdOperatorC1 : public CNdOperator {
public:
  //! constructor
	CNdOperatorC1();

  //! destructor
	virtual ~CNdOperatorC1();

  //! choose the first uncolored node of the graph
  /*! \param graph Graph contains node which are used for finding a node
      \return the pointer of the chosen node, if no node is chosen the methode returns NULL */
  virtual CNode* ChooseNode(CGraph* graph);

  //! returns an operator specific id which is used for testing of equality
  virtual TType GetType() {
    return type_coperator_c1;
  }
};


/*! \class CNdOperatorC1todo algorithm.h
		\brief Implements the non-deterministic slightly improved operator C of the paper.

		CNdOperatorC1todo is the implementation of the choice operator of the paper.
    The ChooseNode() methode returns the first node of the uncolored list
    of the graph.
*/
class CNdOperatorC1todo : public CNdOperatorC1 {
public:
  //! constructor
	CNdOperatorC1todo();

  //! destructor
	virtual ~CNdOperatorC1todo();

  //! returns an operator specific id which is used for testing of equality
  virtual TType GetType() {
    return type_coperator_c1todo;
  }
};


/*! \class CNdOperatorD algorithm.h
		\brief Implements the non-deterministic operator D of the paper.

		CNdOperatorD is the implementation of the support driven choice operator of the paper.
    The ChooseNode() methode returns the first uncolored and supported node of the
    uncolored list of the input graph.
*/
class CNdOperatorD : public CNdOperator {
public:
  //! constructor
	CNdOperatorD();

  //! destructor
	virtual ~CNdOperatorD();

  //! choose the first uncolored and supported node of the graph
  /*! \param graph Graph contains node which are used for finding a node
      \return the pointer of the chosen node, if no node is chosen the methode returns NULL */
  virtual CNode* ChooseNode(CGraph* graph);

  //! returns an operator specific id which is used for testing of equality
  virtual TType GetType() {
    return type_coperator_d;
  }
};


/*! \class CNdOperatorDtodo algorithm.h
		\brief Implements the non-deterministic operator Dtodo.

		CNdOperatorDtodo is the implementation of the support driven choice operator of the paper.
    The ChooseNode() methode returns the first uncolored and supported node the uncolored list
    of the input graph. After choosing the node its successors are inserted into TODO list
    of CNdOperator. Only ColorNode() is implemented.
*/
class CNdOperatorDtodo : public CNdOperatorD{
public:
  //! constructor
	CNdOperatorDtodo();

  //! destructor
	virtual ~CNdOperatorDtodo();

  //! returns an operator specific id which is used for testing of equality
  virtual TType GetType() {
    return type_coperator_dtodo;
  }
};


/*! \class CNdOperatorD2 algorithm.h
		\brief Implements the non-deterministic operator D of the paper.

		CNdOperatorD2 is the implementation of the support driven choice operator of the paper.
    The ChooseNode() methode returns the first uncolored and supported node of the
    uncolored list of the input graph.
*/
class CNdOperatorD2 : public CNdOperator {
public:
  //! constructor
	CNdOperatorD2();

  //! destructor
	virtual ~CNdOperatorD2();

  //! returns the first node of the choice point list
  /*! \param graph Graph contains node which are used for finding a node
      \return the pointer of the chosen node, if no node is chosen the methode returns NULL */
  virtual CNode* ChooseNode(CGraph* graph);

  //! restores the graph to the last choice point which is colored plus and can now colored minus
  /*! \param graph Specifies the graph which nodes are recolored
      \param stack Defines the stack which holds last changes
      \return if no such choice point is found the methode returns false else true*/
  virtual bool RestoreGraph(CGraph* graph, TStack* stack);

  //! returns an operator specific id which is used for testing of equality
//  virtual TType GetType() {
//    return type_coperator_d2;
//  }
};


/*! \class CNdOperatorD2todo algorithm.h
		\brief Implements the non-deterministic operator D of the paper.

		CNdOperatorD2todo is the implementation of the support driven choice operator of the paper.
    The ChooseNode() methode returns the first uncolored and supported node of the
    uncolored list of the input graph.
*/
class CNdOperatorD2todo : public CNdOperatorD2 {
public:
  //! constructor
	CNdOperatorD2todo();

  //! destructor
	virtual ~CNdOperatorD2todo();

  //! returns an operator specific id which is used for testing of equality
  virtual TType GetType() {
    return type_coperator_d2todo;
  }
};


/*! \class CNdOperatorDPref algorithm.h
		\brief Interface for defining non-deterministic operators like COperatorC and COperatorD.

*/
class CNdOperatorDPref : virtual public CNdOperator {
	//! Output operator
	friend std::ostream& operator<<(std::ostream& os, CNdOperator& op);

private:
  static TChoiceLists _choicepointstack;//!< list of choice points
//	static int level;

public:
  //! constructor
	CNdOperatorDPref();

  //! destructor
	virtual ~CNdOperatorDPref();

  //! start the answer set computation with deterministic operator
  /*! \param det_op Operator which describe the deterministic part
      \param graph Graph which should be colored for computing answer sets (last coloring is returned)
      \param as_count Number of answer sets to search (0 means all answer sets)
      \param sets Set of TAtomSet which represent all found answer sets
      \return returns true if all answer sets found w.r.t. as_count */
  virtual bool Call(CDetOperator* det_op, CDetOperator* end_op, CGraph* graph, int as_count, TSetOfAtomSets* sets);

  //! restores the graph to the last choice point which is colored plus and can now colored minus
  /*! \param graph Specifies the graph which nodes are recolored
      \param stack Defines the stack which holds last changes
      \return if no such choice point is found the methode returns false else true*/
  virtual bool RestoreGraph(CGraph* graph, TStack* stack);

  //! recolores the choice point as the top of the input stack
  /*! \param graph Graph on which one choice node is colored from plus to minus
      \param stack The top element is the last plus-colored choice point for recoloring
      \return if the top element is not a plus-colored choice point the methode returns false else true */
  virtual bool RecolorChoicePoint(CGraph* graph, TStack* stack);

	virtual void GenerateChoicePointList(CGraph* graph);

  //! choose the first uncolored, maximal and supported node of the graph
  /*! \param graph Graph contains node which are used for finding a node
      \return the pointer of the chosen node, if no node is chosen the methode returns NULL */
  virtual CNode* ChooseNode(CGraph* graph);

  //! choose the first uncolored, maximal and not supported node of the graph
  /*! \param graph Graph contains node which are used for finding a node
      \return the pointer of the chosen node, if no node is chosen the methode returns NULL */
  virtual CNode* ChooseMustBeMinus(CGraph* graph);

  //! returns an operator specific id which is used for testing of equality
  virtual TType GetType() {
		return (IsTodo() ? type_coperator_dpreftodo : type_coperator_dpref);
  }

};





/*! \class CNdOperatorD2Pref algorithm.h
		\brief Interface for defining non-deterministic operators like COperatorC and COperatorD.

*/
class CNdOperatorD2Pref : virtual public CNdOperator {
	//! Output operator
	friend std::ostream& operator<<(std::ostream& os, CNdOperator& op);

private:
  static TChoiceLists _choicepointstack;//!< list of choice points
//	static int level;

public:
  //! constructor
	CNdOperatorD2Pref();

  //! destructor
	virtual ~CNdOperatorD2Pref();

  //! start the answer set computation with deterministic operator
  /*! \param det_op Operator which describe the deterministic part
      \param graph Graph which should be colored for computing answer sets (last coloring is returned)
      \param as_count Number of answer sets to search (0 means all answer sets)
      \param sets Set of TAtomSet which represent all found answer sets
      \return returns true if all answer sets found w.r.t. as_count */
  virtual bool Call(CDetOperator* det_op, CDetOperator* end_op, CGraph* graph, int as_count, TSetOfAtomSets* sets);

  //! restores the graph to the last choice point which is colored plus and can now colored minus
  /*! \param graph Specifies the graph which nodes are recolored
      \param stack Defines the stack which holds last changes
      \return if no such choice point is found the methode returns false else true*/
  virtual bool RestoreGraph(CGraph* graph, TStack* stack);

  //! recolores the choice point as the top of the input stack
  /*! \param graph Graph on which one choice node is colored from plus to minus
      \param stack The top element is the last plus-colored choice point for recoloring
      \return if the top element is not a plus-colored choice point the methode returns false else true */
  virtual bool RecolorChoicePoint(CGraph* graph, TStack* stack);

	virtual void GenerateChoicePointList(CGraph* graph);

  //! choose the first uncolored, maximal and supported node of the graph
  /*! \param graph Graph contains node which are used for finding a node
      \return the pointer of the chosen node, if no node is chosen the methode returns NULL */
  virtual CNode* ChooseNode(CGraph* graph);

  //! choose the first uncolored, maximal and not supported node of the graph
  /*! \param graph Graph contains node which are used for finding a node
      \return the pointer of the chosen node, if no node is chosen the methode returns NULL */
//  virtual CNode* ChooseMustBeMinus(CGraph* graph);

  //! returns an operator specific id which is used for testing of equality
  virtual TType GetType() {
		return (IsTodo() ? type_coperator_d2preftodo : type_coperator_d2pref);
  }

};




/*! \class CNdOperatorDHPref algorithmpref.h
		\brief Implements the non-deterministic operator Dtodo.

		COperatorDHPref is the implementation of the support driven choice operator of the paper.
    The ChooseNode() methode returns an uncolored and supported node. After choosing the node its
		successors are inserted into TODO list of COperator. Only ColorNode() is implemented.
*/
class CNdOperatorDHPref : virtual public CNdOperator {
	//! Output operator
	friend std::ostream& operator<<(std::ostream& os, CNdOperator& op);

private:
  static TChoiceLists _choicepointstack;//!< list of choice points

public:
  //! constructor
	CNdOperatorDHPref();

  //! destructor
	virtual ~CNdOperatorDHPref();

  //! start the answer set computation with deterministic operator
  /*! \param det_op Operator which describe the deterministic part
      \param graph Graph which should be colored for computing answer sets (last coloring is returned)
      \param as_count Number of answer sets to search (0 means all answer sets)
      \param sets Set of TAtomSet which represent all found answer sets
      \return returns true if all answer sets found w.r.t. as_count */
  virtual bool Call(CDetOperator* det_op, CDetOperator* end_op, CGraph* graph, int as_count, TSetOfAtomSets* sets);

  //! restores the graph to the last choice point which is colored plus and can now colored minus
  /*! \param graph Specifies the graph which nodes are recolored
      \param stack Defines the stack which holds last changes
      \return if no such choice point is found the methode returns false else true*/
//  virtual bool RestoreGraph(CGraph* graph, TStack* stack);

  //! recolors the choice point as the top of the input stack
  /*! \param graph Graph on which one choice node is colored from plus to minus
      \param stack The top element is the last plus-colored choice point for recoloring
      \return if the top element is not a plus-colored choice point the methode returns false else true */
//  virtual bool RecolorChoicePoint(CGraph* graph, TStack* stack);

	virtual bool Heightfunction(CGraph* graph);

  //! choose the first uncolored, maximal and supported node of the graph
  /*! \param graph Graph contains node which are used for finding a node
      \return the pointer of the chosen node, if no node is chosen the methode returns NULL */
  virtual CNode* ChooseNode(CGraph* graph);

  //! returns an operator specific id which is used for testing of equality
  virtual TType GetType() {
    return (IsTodo() ? type_coperator_dprefhtodo : type_coperator_dprefh);
  }
};

} // end of namespace NS_NOMORE

#endif
