/***************************************************************************
                          cnode.h  -  description
                             -------------------
    begin                : Fri Aug 20 2004
    copyright            : (C) 2004 by nomore-dg
    email                : nomore-dg@cs.uni-potsdam.de
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

/*! \file cnode.h
		Declaration of class CNode and CNodeFG
*/


#ifndef CNODE_H
#define CNODE_H

#include "misc.h"

/*! \namespace NS_NOMORE
		\brief Contains everything related to nomore++
*/
namespace NS_NOMORE {

/*! \class CNode cnode.h
  	\brief Defines node in a Graph.

		CNode is the base class for all nodes. It defines the basic behaviour that will be common to all
		future node classes, such as a color and functionality for interaction with a stack for backtracking
		purposes.
*/
class CNode {

	//! Output operator
	friend std::ostream& operator<<(std::ostream& os, CNode& node);

private:
  TId        _id; 	//!< Contains identifier of node. Used to identify node within the TNodeSet
	TColor     _color; //!< Contains color of node
  CRule     *_prule; //!< Pointer to the rule that is represented.
	int 			_blocked,_unblocked;
	TSetOfCounters _supported,_unsupported;
	int 			_status;

  TSetOfNodeSets  *_pzero_predecessors; //!< Contains all 0-preds
	TNodeSet  *_pone_predecessors; //!< Contains all 1-preds
	TNodeSet  *_pzero_successors; //!< Contains all 0-succs
	TNodeSet  *_pone_successors; //!< Contains all 1-succs

  /*! \struct TSaveState cnode.h
  		\brief Contains all attributes which should be saved for backtracking for Class CNode.
  */
  typedef struct {
    TColor color; //!< Contains color to be saved
  } TSaveState;

public:
	//! Constructor.
//  CNode();

	//! Constructor. Initalizes Node with an id.
 	CNode(CIdGenerator *ig);

  //! Constructor. Initalizes Node with an id and a rule.
 	CNode(CIdGenerator *ig, CRule *rule);

	//! Constructor. Initalizes Node with an id.
 	CNode(TId sid);

  //! Constructor. Initalizes Node with an id and a rule.
 	CNode(TId sid, CRule *rule);

	//! Destrcutor.
	virtual ~CNode();

	//! Equality operator.
	virtual bool operator==(const CNode &rhs);

  //! inequality operator
	virtual bool operator!=(const CNode &rhs);

  //! Saves dynamic state attributes for backtracking.
  /*! \param size Return value for the size of the saved buffer
      \return buffer which contains the saved node */
 	virtual char* SaveState(unsigned long& size);

  //! Recovers dynamic state attributes during backtracking.
  /*! \param rhs Buffer which contains the saved node
      \param size Size of the buffer
      \return if the state is successful loaded the methode returns true else false */
	virtual bool LoadState(char* rhs, unsigned long& size);

	virtual TId GetId() const {
    return _id;
  }

	//! Returns color.
	inline virtual TColor GetColor() const {
    return _color;
  }

	//! Returns color.
	inline virtual void SetColor(TColor color) {
    _color = color;
  }

	//! Returns type of instance
	inline virtual TType GetType() const {
    return type_cnode;
  }

	//! Returns 0-Preds
	inline virtual TSetOfNodeSets *GetZeroPredecessors() {
    return _pzero_predecessors;
  }

	//! Returns 1-Preds
	inline virtual TNodeSet *GetOnePredecessors() {
    return _pone_predecessors;
  }

	//! Returns 0-Succs
	inline virtual TNodeSet *GetZeroSuccessors() {
    return _pzero_successors;
  }

	//! Returns 1-Succs
	inline virtual TNodeSet *GetOneSuccessors() {
    return _pone_successors;
  }

	//! Inserts one 0-Pred
	virtual bool InsertZeroPredecessor(CNode*);

	//! Inserts one 1-Pred
	virtual bool InsertOnePredecessor(CNode*);

	//! Inserts one 0-Pred
	virtual bool InsertZeroSuccessor(CNode*);

	//! Inserts one 1-Pred
	virtual bool InsertOneSuccessor(CNode*);

	//! Changes Color of Node after saving the state of the node.
	virtual bool ColorNode(TColor color, TStack *stack, bool choice_point, TColor choice_color);

  //! Sets the corresponding rule
  inline virtual void SetRule(CRule *rule) {
    _prule = rule;
  }

  //! Returns the corresponding rule
  inline virtual CRule* GetRule() {
    return _prule;
  }

  //! Returns true if the node is supported
  virtual bool Supported();

  //! Returns true if the node is supported by the input set of nodes
  virtual bool Supported(TNodeSet* plus_colored);

  //! Returns true if the node is unsupported
  virtual bool Unsupported();

  //! Returns true if the node is blocked
  virtual bool Blocked();

  //! Returns true if the node is blocked by this set
	virtual bool Blocked(TNodeSet* set);

  //! Returns true if the node is unblocked
  virtual bool Unblocked();

  //! Returns the heuristic value of the node
  virtual int HeuristicValue();

  //! Returns the set of counters of supported atoms
//  virtual TSetOfCounters GetSupported() = 0;

  //! Returns the set of counters unsupported atoms
//  virtual TSetOfCounters GetUnsupported() = 0;

  //! Returns the number of blocked atoms
//  virtual int GetBlocked() = 0;

  //! Returns the number of unblocked atoms
//  virtual int GetUnblocked() = 0;

  //! decreases the set of counters of supported atoms
  inline virtual void DecSupported(TId id) {
	 	_supported[id]--;
	}

  //! increases the set of counters of supported atoms
  inline virtual void IncSupported(TId id) {
 		_supported[id]++;
	}

  //! decreases the set of counters of unsupported atoms
  inline virtual void DecUnsupported(TId id) {
		_unsupported[id]--;
	}

  //! increases the set of counters of unsupported atoms
  inline virtual void IncUnsupported(TId id) {
		_unsupported[id]++;
	}

  //! decreases the counters of blocked atoms
	inline virtual void DecBlocked() {
		_blocked--;
	}

  //! increases the counters of blocked atoms
	inline virtual void IncBlocked() {
		_blocked++;
	}

  //! decreases the counters of unblocked atoms
	inline virtual void DecUnblocked() {
		_unblocked--;
	}

  //! increases the counters of unblocked atoms
	inline virtual void IncUnblocked() {
		_unblocked++;
	}

	inline bool GetStatus() {
		return _status;
	}

	inline void SetStatus(int status) {
		_status = status;
	}

};

} // end of namespace NS_NOMORE

#endif
