/***************************************************************************
                          coperator.h  -  description
                             -------------------
    begin                : Fri Aug 20 2004
    copyright            : (C) 2004 by nomore-dg
    email                : nomore-dg@cs.uni-potsdam.de
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

/*! \file cprogram.h
		Declaration of class CProgram
*/

#ifndef CPROGRAM_H
#define CPROGRAM_H

#include "misc.h"
#include "crule.h"
#include <algorithm>
#include <iostream>

using std::pair;

namespace NS_NOMORE {

/*! \class CProgram cprogram.h
		\brief Defines internal representation of a logic program.

		CProgram consists of a set of atoms and a set of rules which are read from stdin by Read()
*/
class CProgram {
  //! output operator
	friend std::ostream& operator<<(std::ostream& os, CProgram& program);
private:
  TAtomSet *_patoms; //!< All atoms of the logic program
  TRuleSet *_prules; //!< The rules of the logic program
	TPrefList *_ppreflist; //!< a list of all preferences (as pair)
	TSetOfRuleSet *_patomsandrules; //!< a list of rules for each atom containing all rules with this atom as head
  TAtomSet *_pbminus;  

protected:
	//! Adds a number of atoms to _patoms
  /*! \param atoms Set of atoms which should be added to the internal set of atoms */
	void AddAtoms(TAtomSet *atoms);

public:
  //! constructor
	CProgram();

	//! destructor deletes all pointers in atom- and rule-map
	virtual ~CProgram();

	//! Returns the atoms of the program
	inline TAtomSet* GetAtoms(){
		return _patoms;
  };

	//! Returns the rules of the program
	inline TRuleSet* GetRules(){
		return _prules;
  };

	//! Returns the list of preferences of the program
	inline TPrefList* GetPrefs(){
		return _ppreflist;
  };

	inline TSetOfRuleSet* GetAtomsAndRules() {
		return _patomsandrules;
	};

  inline TAtomSet* GetBMinus() {
    return _pbminus;
  }

	//! Read the output of lparse
  /*! \param f Specifies the stream from which the program is read. (file or stdin)
      \return if the syntax is correct and the program is read the methode returns true else false */
	virtual bool Read(std::istream &f);

	virtual bool ReadWithPrefs(std::istream &f);  

	//! Adds a basic rule to the rule set
  /*! \param f Specifies the stream from which the basic rule is read (file or stdin)
      \return if the syntax is correct and the basic rule is read the methode returns true else false */
	bool AddBasicRule(std::istream &f);


	bool CProgram::AddRuleForAtom(CAtom *a, CRule *r);


	//! Reads the body part of an lparse decoded rule
  /*! \param f Specifies the stream from which the body is read (file or stdin)
      \param size Count of positive and negative body literals for read
      \param size_neg Count of negative body literals for read
      \param rule Rule for which the body is to set
      \return if the syntax is correct and the body is read the methode returns true else false */
	bool ReadBody(std::istream &f, long size, long size_neg, CRule* rule);

  //! Equality operator
	bool operator==(const CProgram& prg) const;

	//! Inequality operator
	bool operator!=(const CProgram& prg) const;

  //! Insert a rule into the map of rules by copying the pointer address
  /*! \note If the destructor is called the pointer is deleted */
  void InsertRule(CRule* rule);

  //! Insert an atom into the map of atoms by copying the pointer address
  /*!    \note If the destructor is called the pointer is deleted */
  void InsertAtom(CAtom* atom);

  //! Returns a rule specified by ID
  /*! \param id Specifies the id of the rule which should be returned
      \return If the rule with the specified ID does not exist the methode returns NULL. */
  CRule* GetRule(TId id);

  //! Returns an atom specified by ID
  /*! \param id Specifies the id of the atom which should be returned
      \return If the atom with the specified ID does not exist the methode returns NULL. */
  CAtom* GetAtom(TId id);

	//! Read file for preferences
  /*! \param f Specifies the stream from which the program is read. (file or stdin)
      \return if the syntax is correct and the program is read the methode returns true else false */
	virtual bool ReadPref(std::istream &f);

	virtual inline void AddPref(string pref1,string pref2) {
		_ppreflist->push_back(pair<string,string>(pref1,pref2));
	}

	virtual void CreateTransitivePrefs();

  virtual bool RemoveUnsupportedRules();

};

} // end of NS_NOMORE

#endif
