%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%                                                               %
%  Graph Tools Libary                                           %
%  Kathrin Konczak, Christian Anger                             %
%  Mar 2000                                                     %
%                                                               %
%  changed by                                                   %
%  Thomas Linke                                                 %
%  June 2000                                                    %
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%


%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%                                                                       %
% ArcList:             G=(N,A), e.g. N=[1,2,3], A=[1-2,1-3,2-3,3-1]     %
% NodeSuccessorList:   G=(N,A), e.g. N=[1,2,3], A=[1-[2,3],2-[3],3-[1]] %
% NodePredessesorList: G=(N,A), e.g. N=[1,2,3], A=[1-[3],2-[1],3-[1,2]] %
%                                                                       %
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%


%%%%%%%%%%%%%%%%%%
% Transformation %
%%%%%%%%%%%%%%%%%%

% arc2succ(+N,+A_In,-A_Out)
% N     = nodes
% A_In  = arcs in ArcList
% A_Out = arcs in NodeSuccessorList
arc2succ(N,A_In,A_Out):-adja2inzi(N,A_In,[],A_Out).

% succ2arc(+A_In, -A_Out)
% A_In  = arcs in NodeSuccessorList
% A_Out = arcs in ArcList
succ2arc(A_In,E_Out):-inzi2adja(A_In,[],E_Out).

% arc2pred(+N,+A_In,-A_Out)
% N     = nodes
% A_In  = arcs in ArcList
% A_Out = arcs in predessesor
arc2pred(N,A_In,A_Out):-adja2pred(N,A_In,[],A_Out).

% pred2arc(+A_In,-A_Out)
% A_In  = arcs in predessesor
% A_Out = arcs in ArcList
pred2arc(A_In,A_Out):-pred2adja(A_In,[],A_Out).

% succ2pred(?G1,?G2) 
% G1 = graph in NodeSuccessorList
% G2 = graph in NodePredessesorList
% either G1 or G2 has to be a variable
succ2pred(G1,G2):-
	var(G1),not(var(G2)),
	pred2arc(G2,Arcs),
	nodes(G2,Nodes),
	arc2succ(Nodes,Arcs,G1).

succ2pred(G1,G2):-
	var(G2),not(var(G1)),
	succ2arc(G1,Arcs),
	nodes(G1,Nodes),
	arc2pred(Nodes,Arcs,G2).

% nodes (+A_In,-Nodes) 
% A_In  = graph in NodeSuccessorList or NodePredessesorList
% Nodes = nodes of the graph
nodes([],[]):-!.
nodes([N-_|A0],[N|Nodes]):-
	nodes(A0,Nodes).


%%%%%%%%%%%%
% Sonstige %
%%%%%%%%%%%%

% findsucc(+N,+A_In,-Succ)
% N    = node
% A_In = arcs in ArcList
% Succ = list of succesors of N
findsucc(N,A,Succ):-findsucc(N,A,[],Succ).

% findpred(+N,+A_In,-Pred)
% N    = node
% A_In = arcs in ArcList
% Pred = list of predessors of N
findpred(N,A,Pred):-findpred(N,A,[],Pred).

% findnodes(+A_In,-Nodes)
% A_In = arcs in ArcList
findnodes(A_In,N):-findnodes(A_In,[],N).


%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%                                                               %
%  Implementation                                               %
%                                                               %
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

%Hilfe fuer adja2inzi
%Wandelt Adjazenz-Darstellung in Inzidenz-Darstellung um
adja2inzi([],_E_In,In,In).
adja2inzi([K|R],E_In,In,Out):-
	findsucc(K,E_In,[],L),
	adja2inzi(R,E_In,[K-L|In],Out).

%Hilfe fuer findsucc
%Sucht aus Adjazenz-Darstellung alle Nachfolger eines Knoten
findsucc(_K,[],In,In).
findsucc(K,[K-N|R],In,Out):-
	findsucc(K,R,[N|In],Out),!.
findsucc(K,[_|R],In,Out):-
	findsucc(K,R,In,Out).


%Hilfe fuer inzi2adja
%Wandelt Inzidenz-Darstellung in Adjazens-Darstellung um
inzi2adja([],In,In).
inzi2adja([V-L|R],In,Out):-
	make_edges_from_inzi(V,L,[],K),
	union(In,K,Out1),
	inzi2adja(R,Out1,Out).

% gives all arcs deriving from a vertice and its successors
make_edges_from_inzi(_V,[],In,In).
make_edges_from_inzi(V,[H|T],In,Out):-
	make_edges_from_inzi(V,T,[V-H|In],Out).


%Hilfe fuer adja2pred
%wandelt Adjazenz-Darstellung in Predessesor-Darstellung um
adja2pred([],_,In,In).

adja2pred([H|T],E,In,[H-L|Out]):-
	findpred(H,E,[],L),
	adja2pred(T,E,In,Out).

%Sucht aus Adjazensdarstellung alle Vorgaenger des Knoten V
findpred(_K,[],In,In).
findpred(K,[N-K|R],In,[N|Out]):-
	findpred(K,R,In,Out),!.
findpred(K,[_|R],In,Out):-
	findpred(K,R,In,Out).


%Hilfe fuer pred2adja
pred2adja([],In,In).
pred2adja([V-L|R],In,Out):-
	make_edges_from_pred(V,L,[],K),
	union(In,K,Out1),
	pred2adja(R,Out1,Out).

make_edges_from_pred(_V,[],In,In).
make_edges_from_pred(V,[H|T],In,Out):-
	make_edges_from_pred(V,T,[H-V|In],Out).


% Sucht aus ArcList die Knoten
findnodes([],In,In).
findnodes([A-B|T],In,Out):-
	union(In,[A],Out1),
	union(Out1,[B],Out2),
	findnodes(T,Out2,Out).







