%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%                                                                         %
% TheoryBase                                                              %
% Christian Anger, Katrin Konczak, Thomas Linke                           %
%                                                                         %
% last edit: Dec 22, 2000                                                 %
%                                                                         %
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%


:-ensure_loaded(graphs).

% theorybase(+T,-LP)
% theorybase is true, iff LP is the Logic Program to T, where T is one of the
% following:
% T=coloring(+T,+G,+C), rules for coloring a given graph with given colors
%                       G=(Nodes,Arcs), arcs in ArcList
%                       C is a list of colors, e.g. C=[c1,c2,c3]
%                       T=1
%                       T=2
%                       T=3
% T=hamilton(+G,+N1),   rules for finding hamilton circles in a given graph
%                       G=(Nodes,Arcs), arcs in ArcList
%                       N1=starting node
% T=independent(+G),    rules for finding independent setx
%                       G=(Nodes,Arcs), arcs in ArcList


theorybase(coloring(1,(V,E),C),LP):-col_1((V,E),C,LP).
theorybase(coloring(2,(V,E),C),LP):-col_2((V,E),C,LP).
theorybase(coloring(3,(V,E),C),LP):-col_3((V,E),C,LP).
theorybase(hamilton((V,I),V1),LP):- ham((V,I),V1,LP).
theorybase(indset(G),LP):-indsets(G,[],LP).





%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% Colorings                                                               %
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

%
% encoding of the theories
%
%	/\i
%      /__\col	for i= 1,2,3
%
%
% let be G=(V,E) an undirected graph with
% V is the set of nodes with an existing ordering "<"
% E is the set of edges [x-y] with (x,y) is an edge in the graph G
% C=[c1,...,ck] is the set of all possible colors 	
%
% col_i(+Graph,+Colors,-Rules)
%
col_1((V,E),Colors,Rules):-
	d0_rules(V,Colors,R1),
	global_allrules(E,Colors,R2),
	append(R1,R2,Rules).

col_2(Graph,Colors,Rules):-
	graphsort(Graph,SG,SN),
	d0_rules(SN,Colors,R1),
	partial_allrules(SG,SN,Colors,R2),
	append(R1,R2,Rules).

col_3((V,E),Colors,Rules):-
	d0_rules(V,Colors,R1),
	local_allrules(E,Colors,R2),
	append(R1,R2,Rules).


%
% d0_rules/2
%
d0_rules([],_,[]).
d0_rules([V|VR],C,Rules):-
	d0_nodes(V,[],C,R1),
	d0_rules(VR,C,R2),
	append(R1,R2,Rules).


%
% d0_nodes/4
%
d0_nodes(_,_,[],[]).
d0_nodes(Vi,CR1,[Cj|CR2],[rule(clr(Vi,Cj),[],L)|Rules]):-
	append(CR1,CR2,CR),
	color_node_list(Vi,CR,L),
	d0_nodes(Vi,[Cj|CR1],CR2,Rules).


%
% color_node_list/3
%
color_node_list(_,[],[]).
color_node_list(Vi,[C|CR],[clr(Vi,C)|L]):-
	color_node_list(Vi,CR,L).


%
% local_allrules/3
%
local_allrules([],_,[]).
local_allrules([K|Graph],C,Rules):-
	local_rule(K,C,R1),
	local_allrules(Graph,C,R2),
	append(R1,R2,Rules).


%
% local_rule/3
%
local_rule(_,[],[]).
local_rule(X-Y,[C|CR],[rule(f(ES,C),[clr(X,C),clr(Y,C)],[f(ES,C)])|Rules]):-
        term_string(X,SX),term_string(Y,SY),append_strings(SX,"_",S1),
        append_strings(S1,SY,ES),
	local_rule(X-Y,CR,Rules).


%
% global_allrules/3
%
global_allrules([],_,[]).
global_allrules([K|Graph],C,Rules):-
	global_rules(K,C,R1),
	global_allrules(Graph,C,R2),
	append(R1,R2,Rules).


%
% global_rules/3
%
global_rules(_,[],[]).
global_rules(X-Y,[C|CR],[rule(f,[clr(X,C),clr(Y,C)],[f])|Rules]):-
	global_rules(X-Y,CR,Rules).


%
% partial_allrules/4
%
partial_allrules([],_,_,[]).
partial_allrules([[]|SG],[_|VR],C,Rules):-
	partial_allrules(SG,VR,C,Rules).
partial_allrules([ViGraph|SG],[Vi|VR],C,Rules):-
	partial_rules(ViGraph,Vi,C,R1),
	partial_allrules(SG,VR,C,R2),
	append(R1,R2,Rules).
	

%
% partial_rules/4
%
partial_rules([],_,_,[]).
partial_rules([X-Y|ViGraph],Vi,C,Rules):-
	build_p_rules(X-Y,Vi,C,R1),
	partial_rules(ViGraph,Vi,C,R2),
	append(R1,R2,Rules).


%
% build_p_rules/4
%
build_p_rules(_,_,[],[]).
build_p_rules(X-Y,Vi,[C|Colors],
	      [rule(f(Vi),[clr(X,C),clr(Y,C)],[f(Vi)])|Rules]):-
	build_p_rules(X-Y,Vi,Colors,Rules).


%
% graphsort/3
%
graphsort((V,E),SG,SV):-
	sort(V,SV),
	sort_graph(E,SV,SG).	


%
% sort_graph/3
%
sort_graph(_,[],[]).
sort_graph(Graph,[N|Nodes],[List|SG]):-
	findall(N-X,(member(N-X,Graph),X @<N),L1),
	findall(N-Y,(member(Y-N,Graph),Y @<N),L2),
	append(L1,L2,List),
	sort_graph(Graph,Nodes,SG).

	
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% Independent Sets                                                        %
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

%
% generating rules from given graph G=(Nodes,Arcs) arcs in ArcList
% indsets(+Graph,[],-Regeln)
%
indsets(([],_),In,In).
indsets(([V|Vs],E),In,Out):-
	dv((V,E),R),
        indsets((Vs,E),[R|In],Out).


%
% dv(+Vertices,+Edges,-Rules)
%
dv((V,E),rule(in(V),[],InN)):-
	findsucc(V,E,Succ),
	findpred(V,E,Pred),
	union(Succ,Pred,Neighbors),
	in_neighbors(Neighbors,InN).


%
% in_neighbors(+neighbors,-in(neighbors))
%
in_neighbors([],[]).
in_neighbors([N|Neighbors],[in(N)|InN]):-
	in_neighbors(Neighbors,InN).


%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% Hamilton Circles                                                        %
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

%
% ham(+Graph,+V1,-R) generates the rules for a given graph G=(V,E) 
% with E is the IncList of the edges and V the set of nodes
% V1 is the visited vertex (start vertex)
%
ham((_V,E),V1,[rule(v(V1),[],[]),R3|Rules]):- 
	ham_d0_rule(E,R1),
	ham_d1_rule(E,R2),
	cycle_rule(E,V1,[R3]),
	append(R1,R2,Rules).


%
% ham_d0_rule/2
%
ham_d0_rule([],[]):-!.
ham_d0_rule([V-W|Graph],Rules):-
	move_rule(V,[],W,R1),
	ham_d0_rule(Graph,R2),
	append(R1,R2,Rules).


%
% ham_d1_rule/2
%
ham_d1_rule([],[]):-!.
ham_d1_rule([N-_|Graph],[rule(f(N),[],[v(N),f(N)])|List]):-
	ham_d1_rule(Graph,List).


%
% move_rule/4
%
move_rule(_,_,[],[]).
move_rule(V,W1,[Wj|W2],Rules):-
	append(W1,W2,W),
	build_list(V,W,List),
	move_rule(V,[Wj|W1],W2,List2),
	append([rule(v(Wj),[v(V)],List),rule(a(V,Wj),[v(V)],List)],
		List2,Rules).


%
% build_list/3
%
build_list(_,[],[]).
build_list(V,[Wj|W],[a(V,Wj)|L]):-
	build_list(V,W,L).


%
% cycle_rule/3
%
cycle_rule(Graph,V1,[rule(f,[],L)]):-
	pred_of_K(Graph,V1,VP),!,
	cycle_rule_arcs(V1,VP,L1),
	append(L1,[f],L).


%
% cycle_rule_arcs/3
%
cycle_rule_arcs(_,[],[]):-!.
cycle_rule_arcs(V1,[Ui|U],[a(Ui,V1)|L]):-
	cycle_rule_arcs(V1,U,L).


%
% pred_of_K/3
%
pred_of_K([],_,[]).
pred_of_K([N-L|R],K,[N|Pre]):-
	member(K,L),
	pred_of_K(R,K,Pre).
pred_of_K([_|R],K,Pre):-
	pred_of_K(R,K,Pre).
	














