%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% DaVinci Term Representation for block graphs                            %
%                                                                         %
% author: Christian Anger, Kathrin Konczak                                %
%                                                                         %
% last edit: Jun 2002                                                     %
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

%
% daVinci is a graph representation tool that uses a term representation
% of a graph to produces output
% a node in this term representation looks like
% l(Id,n(Class,[Attributes],[Edges]))
% we use the following attributes:
% a("OBJECT", Name_of_Node)
% a("COLOR", Color_of_Node)
% a("_GO", Form_of_Node)
%
% an edge looks like
% l(Id,e(Class,[Attributes],Goal_Id))
% we use the following attributes:
% a("EDGECOLOR",EdgeColor)
% a("EDGEPATTERN",EdgePattern)
%
% for possible colors, forms, patterns look at daVinci_db at the end of
% this file


%
% daVinci(+File)
% generates daVinci term representation of block graph in Prolog database
%
daVinci(File) :-
	(
	  get_flag(metainfo) ->
	  (
	    (
	      get_flag(names) ->
	      (
		daVinci_nodes(rule_full,DaVinci1),
		daVinci_nodes(body_full,DaVinci2),
		daVinci_nodes(head_full,DaVinci3)
	      );
	      (
		daVinci_nodes(rule,DaVinci1),
		daVinci_nodes(body,DaVinci2),
		daVinci_nodes(head,DaVinci3)
	      )
	    ),
	    append(DaVinci1,DaVinci2,DaVinci4),
	    append(DaVinci4,DaVinci3,DaVinci)
	  );
	  (
	    (
	      get_flag(names) ->
	      (
		daVinci_nodes(simple_rule_full,DaVinci)
	      );
	      (
		daVinci_nodes(simple_rule,DaVinci)
	      )
	    )
	  )
	),
	
	File1= 'bg.daVinci',
	open(File1,write,S),
	writeq(S,DaVinci),
	close(S),
	(
	    get_flag(show) -> daVinci_start(DaVinci);true
	).


%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% generic pradicates                                                      %
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

%
% daVinci_nodes(+Type,-DaVinci)
%
daVinci_nodes(Type,DaVinci):-
	daVinci_elements(Type,Ids),
	daVinci_nodes(Ids,Type,DaVinci).


%
% daVinci_nodes(+Ids,+Type,-DV_Nodes)
% gives daVinci term representation of nodes Ids of type Type
%
daVinci_nodes([],_,[]).
daVinci_nodes([Id|Ids],Type,[DV_Node|DV_Nodes]):-
	% determine Id and Name of node
	daVinci_id(Type,Id,DV_Id,DV_Name),
	daVinci_edges(Type,Id,DV_Edges),
	
	% daVinci keywords - node independent (KW = key word)
 	daVinci_keyword(object, KW_Obj),
 	daVinci_keyword(color,  KW_Color),
 	daVinci_keyword(go,     KW_Go),

	% attributes - node type dependent
	daVinci_attribute(Type,Class,Form,Color),
	daVinci_popup_menu(DV_Id,[Menu1,Menu2]),

	% daVinci term representation of node
	DV_Node = l(DV_Id,                % id
		    n(Class,              % type 
		      [a(KW_Obj,DV_Name), % list of
		       a(KW_Color,Color), % attributes
		       a(KW_Go,Form),
		       m([Menu1,Menu2])], %incl. popupMenu
		      DV_Edges
		     )
		   ),
	
	%recursive call
	daVinci_nodes(Ids,Type,DV_Nodes).			     

%
% daVinci_edges(+Ids,+Type,From,-DV_Edges)
%
daVinci_edges([],_,_,[]).
daVinci_edges([Id|Ids],Type,From,[DV_Edge|DV_Edges]):-
	% determine Id and Name of edge
	daVinci_edge_id(Type,Id,From,DV_Id,DV_Name),

	% daVinci keywords - edge independent (KW = key word)
	daVinci_keyword(edgecolor,KW_EdgeColor),
	daVinci_keyword(edgepattern,KW_EdgePattern),

	% attributes - edge type dependent
	daVinci_edge_attributes(Type,Class,Color,Pattern),

	% daVinci term representation of edge
	DV_Edge = l(DV_Id,                         % id
		    e(Class,                       % type 
		      [a(KW_EdgeColor,Color),      % list of
		       a(KW_EdgePattern,Pattern)], % attributes
		      r(DV_Name)
		     )
		   ),

	%recursive call
	daVinci_edges(Ids,Type,From,DV_Edges).


	
%
% daVinci_id(+Type,+Id,-DV_Id,-DV_Name)
%
daVinci_id(Type,Id,DV_Id,DV_Name):-
	daVinci_id2(Type,Id,Full,Id2), 
	term_string(Id,DV_Id),          %neu
	term_string(Id,DV_Name).

%
% daVinci_edge_id(+Type,+Id,+From,-DV_Id,-DV_Name)
%
daVinci_edge_id(Type,Id,From,DV_Id,DV_Name):-
	daVinci_edge_id2(Type,Id,From,Id2,From2),
	daVinci_kind(Type,Kind),
	term_string(Kind:From->Id,DV_Id),
	term_string(Id,DV_Name).

%
% daVinci_attributes(+Type,-Class,-Form,-Color)
%
daVinci_attribute(Type,Class,Form,Color):-
	daVinci_class(Type,Class),
	daVinci_form(Type,Form),
	daVinci_color(Type,Color).

daVinci_popup_menu(DV_ID,Menu):-
	append_strings(DV_ID,"show",MenID1),
	append_strings(DV_ID,"hide",MenID2),
	Menu = [menu_entry(MenID1,"show full node info"),
	        menu_entry(MenID2,"hide node info")].
	
%
% daVinci_edge_attributes(+Type,-Class,-Color,-Pattern)
%
daVinci_edge_attributes(Type,Class,Color,Pattern):-
	daVinci_class(Type,Class),
	daVinci_color(Type,Color),
	daVinci_pattern(Type,Pattern).

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% type dependent pradicates                                               %
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

%
% Type is in {body,rule,head,
%             body_full,rule_full,head_full
%             simple_rule,simple_rule_full}
% for nodes and in {b_r,r_h,h_0_b,h_1_b,r_0_r,r_1_r} for edges
%


%
% daVinci_elements(+Type,-Ids)
%
daVinci_elements(rule,Ids):-
	gr_rule_ids(Ids).
daVinci_elements(body,Ids):-
	gr_body_ids(Ids).
daVinci_elements(head,Ids):-
	gr_head_ids(Ids).
daVinci_elements(rule_full,Ids):-
	gr_rule_ids(Ids).
daVinci_elements(body_full,Ids):-
	gr_body_ids(Ids).
daVinci_elements(head_full,Ids):-
	gr_head_ids(Ids).
daVinci_elements(simple_rule,Ids):-
	gr_rule_ids(Ids).
daVinci_elements(simple_rule_full,Ids):-
	gr_rule_ids(Ids).

%
% daVinci_id2(+Type,+Id,-FullRepresentation,-Id2)
%
daVinci_id2(rule,Id,Id2,Id2):-
	id_rule(Id,_,Id2).
daVinci_id2(body,Id,Id2,Id2):-
	id_b(Id,_,Id2).
daVinci_id2(head,Id,Id2,Id2):-
	id_h(Id,_,Id2).
daVinci_id2(rule_full,Id,Full,Id2):-
	id_rule_full(Id,Full,Id2).
daVinci_id2(body_full,Id,Full,Id2):-
	id_b(Id,Full,Id2).
daVinci_id2(head_full,Id,Full,Id2):-
	id_h(Id,Full,Id2).
daVinci_id2(simple_rule,Id,Id2,Id2):-
	id_rule(Id,_,Id2).
daVinci_id2(simple_rule_full,Id,Full,Id2):-
	id_rule_full(Id,Full,Id2).

%
% daVinci_edge_id2(+Type,+Id,+From,-Id2,-From2),
%
daVinci_edge_id2(b_r,Id,From,Id2,From2):-
	id_rule(Id,_,Id2),
	id_b(From,_,From2).
daVinci_edge_id2(r_h,Id,From,Id2,From2):-
	id_h(Id,_,Id2),
	id_rule(From,_,From2).
daVinci_edge_id2(h_0_b,Id,From,Id2,From2):-
	id_b(Id,_,Id2),
	id_h(From,_,From2).
daVinci_edge_id2(h_1_b,Id,From,Id2,From2):-
	id_b(Id,_,Id2),
	id_h(From,_,From2).
daVinci_edge_id2(r_0_r,Id,From,Id2,From2):-
	id_rule(Id,_,Id2),
	id_rule(From,_,From2).
daVinci_edge_id2(r_1_r,Id,From,Id2,From2):-
	id_rule(Id,_,Id2),
	id_rule(From,_,From2).


%
% daVinci_edges(+Type,+Id,-Edges)
%
daVinci_edges(rule,Id,Edges):-
	rule_hs(Id,Succs),
	daVinci_edges(Succs,r_h,Id,Edges).
daVinci_edges(body,Id,Edges):-
	b_rules(Id,Succs),
	daVinci_edges(Succs,b_r,Id,Edges).
daVinci_edges(head,Id,Edges):-
	h_0_bs(Id,Succs1),
	daVinci_edges(Succs1,h_0_b,Id,Edges1),
	h_1_bs(Id,Succs2),
	daVinci_edges(Succs2,h_1_b,Id,Edges2),
	append(Edges1,Edges2,Edges).
daVinci_edges(rule_full,Id,Edges) :- daVinci_edges(rule,Id,Edges).
daVinci_edges(body_full,Id,Edges) :- daVinci_edges(body,Id,Edges).
daVinci_edges(head_full,Id,Edges) :- daVinci_edges(head,Id,Edges).
daVinci_edges(simple_rule,Id,Edges) :-
	rule_hs(Id,Heads),
	edges_0(Heads,E0),
	sort(E0,SE0),
	daVinci_edges(SE0,r_0_r,Id,Edges1),
	edges_1(Heads,E1),
	sort(E1,SE1),
	daVinci_edges(SE1,r_1_r,Id,Edges2),
	append(Edges1,Edges2,Edges).
daVinci_edges(simple_rule_full,Id,Edges) :-
	daVinci_edges(simple_rule,Id,Edges).

edges_0([],[]).
edges_0([H|Heads],Rules):-
	h_0_bs(H,Bodies),
	corresponding_rules(Bodies,Rules1),
	edges_0(Heads,Rules2),
	append(Rules1,Rules2,Rules).
edges_1([],[]).
edges_1([H|Heads],Rules):-
	h_1_bs(H,Bodies),
	corresponding_rules(Bodies,Rules1),
	edges_1(Heads,Rules2),
	append(Rules1,Rules2,Rules).

corresponding_rules([],[]).
corresponding_rules([Body|Bodies],Rules):-
	b_rules(Body,Rules1),
	corresponding_rules(Bodies,Rules2),
	append(Rules1,Rules2,Rules).

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% database                                                                %
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

% daVinci_class(+Type,-Class)
daVinci_class(rule,Class) :- atom_to_string(rule,Class).
daVinci_class(head,Class) :- atom_to_string(head,Class).
daVinci_class(body,Class) :- atom_to_string(body,Class).
daVinci_class(rule_full,Class) :- atom_to_string(rule,Class).
daVinci_class(head_full,Class) :- atom_to_string(head,Class).
daVinci_class(body_full,Class) :- atom_to_string(body,Class).
daVinci_class(simple_rule,Class) :- atom_to_string(rule,Class).
daVinci_class(simple_rule_full,Class) :- atom_to_string(rule,Class).
daVinci_class(b_r,Class) :- atom_to_string(b_r,Class).
daVinci_class(r_h,Class) :- atom_to_string(r_h,Class).
daVinci_class(h_0_b,Class) :- atom_to_string(h_0_b,Class).
daVinci_class(h_1_b,Class) :- atom_to_string(h_1_b,Class).
daVinci_class(r_0_r,Class) :- atom_to_string(r_0_r,Class).
daVinci_class(r_1_r,Class) :- atom_to_string(r_1_r,Class).

% daVinci_form(+Type,-Form)
daVinci_form(rule,Form) :- daVinci_db(ellipse,Form).
daVinci_form(body,Form) :- daVinci_db(box,Form).
daVinci_form(head,Form) :- daVinci_db(text,Form).
daVinci_form(rule_full,Form) :- daVinci_db(ellipse,Form).
daVinci_form(body_full,Form) :- daVinci_db(box,Form).
daVinci_form(head_full,Form) :- daVinci_db(text,Form).
daVinci_form(simple_rule,Form) :- daVinci_db(ellipse,Form).
daVinci_form(simple_rule_full,Form) :- daVinci_db(ellipse,Form).

% daVinci_color(+Type,-Color)
daVinci_color(rule,Color) :- daVinci_db(white,Color).
daVinci_color(head,Color) :- daVinci_db(white,Color).
daVinci_color(body,Color) :- daVinci_db(white,Color).
daVinci_color(rule_full,Color) :- daVinci_db(white,Color).
daVinci_color(head_full,Color) :- daVinci_db(white,Color).
daVinci_color(body_full,Color) :- daVinci_db(white,Color).
daVinci_color(simple_rule,Color) :- daVinci_db(white,Color).
daVinci_color(simple_rule_full,Color) :- daVinci_db(white,Color).
daVinci_color(b_r,Color) :- daVinci_db(black,Color).
daVinci_color(r_h,Color) :- daVinci_db(black,Color).
daVinci_color(h_0_b,Color) :- daVinci_db(green,Color).
daVinci_color(h_1_b,Color) :- daVinci_db(red,Color).
daVinci_color(r_0_r,Color) :- daVinci_db(green,Color).
daVinci_color(r_1_r,Color) :- daVinci_db(red,Color).

% daVinci_pattern(+Type,-Pattern)
daVinci_pattern(b_r,Pattern) :- daVinci_db(dotted,Pattern).
daVinci_pattern(r_h,Pattern) :- daVinci_db(dashed,Pattern).
daVinci_pattern(h_0_b,Pattern) :- daVinci_db(solid,Pattern).
daVinci_pattern(h_1_b,Pattern) :- daVinci_db(solid,Pattern).
daVinci_pattern(r_0_r,Pattern) :- daVinci_db(solid,Pattern).
daVinci_pattern(r_1_r,Pattern) :- daVinci_db(solid,Pattern).

% keywords
daVinci_keyword(object,Obj) :- daVinci_db(object,Obj).
daVinci_keyword(color,Color) :- daVinci_db(color,Color).
daVinci_keyword(node,Node) :- daVinci_db(node,Node).
daVinci_keyword(edge,Node) :- daVinci_db(edge,Node).
daVinci_keyword(go,Go) :- daVinci_db(go,Go).
daVinci_keyword(edgecolor,E) :- daVinci_db(edgecolor,E).
daVinci_keyword(edgepattern,E) :- daVinci_db(edgepattern,E).

% daVinci_kind(+Type,-Kind)
daVinci_kind(b_r,3).
daVinci_kind(r_h,3).
daVinci_kind(h_0_b,0).
daVinci_kind(h_1_b,1).
daVinci_kind(r_0_r,0).
daVinci_kind(r_1_r,1).

% show_daVinci_color(+Color,-DaVinciColor)
show_daVinci_color(0,red).
show_daVinci_color(1,green).
show_daVinci_color(2,darkgreen).
show_daVinci_color(i,gray).
show_daVinci_color(u,white).

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% daVinci_database                                                        %
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

% keywords
daVinci_db(object,Object) :- atom_to_string('OBJECT',Object).
daVinci_db(color,Color) :- atom_to_string('COLOR',Color).
daVinci_db(go,Go) :- atom_to_string('_GO',Go).
daVinci_db(node,Node) :- atom_to_string(node,Node).
daVinci_db(edge,Edge) :- atom_to_string(edge,Edge).
daVinci_db(edgecolor,EdgeColor) :- atom_to_string('EDGECOLOR',EdgeColor).
daVinci_db(edgepattern,EdgePat) :- atom_to_string('EDGEPATTERN',EdgePat).
daVinci_db(border,Border) :- atom_to_string('BORDER',Border).

% colors
daVinci_db(white,White):- atom_to_string(white,White).
daVinci_db(red,Red):- atom_to_string(red,Red).
daVinci_db(blue,Blue):- atom_to_string(blue,Blue).
daVinci_db(gray,Gray):- atom_to_string(gray,Gray).
daVinci_db(green,Green):- atom_to_string(green,Green).
daVinci_db(black,Black):- atom_to_string(black,Black).
daVinci_db(darkgreen,Black):- atom_to_string(darkgreen,Black).
% form
daVinci_db(triangle,Form) :- atom_to_string(trianlge,Form).
daVinci_db(box,Form) :- atom_to_string(box,Form).
daVinci_db(circle,Form) :- atom_to_string(circle,Form).
daVinci_db(ellipse,Form) :- atom_to_string(ellipse,Form).
daVinci_db(rhombus,Form) :- atom_to_string(rhombus,Form).
daVinci_db(text,Form) :- atom_to_string(text,Form).
% patterns
daVinci_db(solid,Pattern) :- atom_to_string(solid,Pattern).
daVinci_db(dotted,Pattern) :- atom_to_string(dotted,Pattern).
daVinci_db(dashed,Pattern) :- atom_to_string(dashed,Pattern).
daVinci_db(thick,Pattern) :- atom_to_string(thick,Pattern).
daVinci_db(double,Pattern) :- atom_to_string(double,Pattern).
daVinci_db(single,Pattern) :- atom_to_string(single,Pattern).


% help
id_rule(Id,_Rule,Id1):-
	id_rule(Id,_Rule),
	integer_atom(Id,Atom),
	concat_atoms(r,Atom,Id1).
id_rule_full(Id,Term,Id1):-
	id_rule(Id,Rule),
	integer_atom(Id,Atom),
	concat_atoms(r,Atom,Id1),
	rule2term(Rule,Term).
id_b(Id,Term,Id1):-
	id_b(Id,Body),
	integer_atom(Id,Atom),
	concat_atoms(b,Atom,Id1),
	body2term(Body,Term).
id_h(Id,Head,Id1):-
	id_h(Id,Head),
	integer_atom(Id,Atom),
	concat_atoms(h,Atom,Id1).



% pradicates for real names
rule2term(Rule,Term):-
	rule_head(Rule,Head),
	list2term(Head,THead),
	rule_body(Rule,Body),
	bodies2term(Body,TBody),
	(
	  TBody = true ->
	  (
	    Term = THead
	  );
	  (
	    Term = (THead :- TBody)
	  )
	).


list2term([],true).
list2term([Head],(Head)):-!.
list2term([Head|Heads],Term):-
	list2term(Heads,Term1),
	Term=(Head,Term1).

bodies2term([],true).
bodies2term([Body],Term):-!,body2term(Body,Term).
bodies2term([B|Bs],Term):-
	bodies2term(Bs,Term1),
	body2term(B,Term2),
	Term = ((Term2);Term1).

body2term(Body,Term):-
	neg_body(Body,Neg),
	add_nots(Neg,NNeg),
	list2term(NNeg,Term1),
	pos_body(Body,Pos),
	list2term(Pos,Term2),
	(
	  Term1 = true ->
	  (
	    Term2 = true ->
	    (
	      Term = true
	    );
	    (
	      Term = Term2
	    )
	  );
	  (
	    Term2 = true ->
	    (
	      Term = Term1
	    );
	    (
	      Term = (Term2,Term1)
	    )
	  )
	).


add_nots([],[]).
add_nots([H|T],[NH|NT]):-
%	NH = not H, % Old
	NH = not(H),
	add_nots(T,NT).


