%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% computation and compilation of program database for given program LP    %
%                                                                         %
% defined predicat: program_database(+LP)                                 %
%                                                                         %
% Thomas Linke                                                            %
%                                                                         %
% last edit:  Sep 2002                                                    %
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% interface to program database                                           %
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
h_0_b( N, B ) :- c_h_0_b(N,B).

h_1_b( N, B ) :- c_h_1_b(N,B).

rule_h( R, H ) :- c_rule_h(R,H).
rule_h( R, H ) :- nc_rule_h(R,H).

rule_b( R, B ) :- c_rule_b(R,B).

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% program_database(+LP) comutes database corresponding to program LP      %
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%                 +
program_database( LP ) :-
	initialize_db,
	indexing(LP),         % indexing
	program_db.           % prolog database for logic program (on indices) 

%
% initialize_db/0 ensures that no old predicates (from former compiled 
% logic programs) are present in the Prolog database 
%
initialize_db :-
%	write('initialize database ... '),
	All=[
             (id_rule(_,_) :- fail),       % indexing in database.pl
	     (id_b(_,_) :- fail),
	     (id_h(_,_) :- fail),

	     (c_h_0_b(_,_) :- fail),       % program database in datasbase.pl
	     (c_h_1_b(_,_) :- fail),
	     (c_rule_h(_,_) :- fail),
	     (nc_rule_h(_,_) :- fail),     % in reduction.pl
	     (c_rule_b(_,_) :- fail),

	     (c_del_b(_) :- fail),         % delete information of program
	     (nc_del_b(_) :- fail),        % in blockgraph.pl
	     (c_del_h(_) :- fail),
	     (c_del_rule(_) :- fail),

	     (h_0_bs(_,_) :- fail),        % block graph with metainformation
	     (h_1_bs(_,_) :- fail),        % in blockgraph.pl
	     (b_0_hs(_,_) :- fail),
	     (b_1_hs(_,_) :- fail),
	     (rule_hs(_,_) :- fail),
	     (rule_bs(_,_) :- fail),
	     (h_rules(_,_) :- fail),
	     (b_rules(_,_) :- fail),

	     (b_bs(_,_) :- fail),          % sitward propable bodies
	     (rule_rules(_,_) :- fail),    % sitward propable rules

	     (c_loop_1(_):-fail),          % rule classification
	     (c_fact(_):-fail),            % in blockgraph.pl
	     (c_q_fact(_):-fail)
	    ],
	compile_clauses(All,"tmp_indexing.pl"),
%	writeln('done'),nl.
	true.

%
% make indices for rules heads and bodies
% indexing:         id_rule/2, id_b/2, id_h/2 
%
indexing( LP ) :-
	indexing(LP,DB),
	compile_clauses(DB,"tmp_indexing.pl").

%
% compile databse predicates:
%   id_rule/2, 
%   id_b/2,
%   id_h/2
%         +   -
indexing( LP, DB ) :-
%	write('indexing ... '),
	set_counter(id,0),
	index_rule(LP,IndexedLP,[],Heads1,[],Bodies1),
	heuristic_sort_h(Heads1,Heads),
	heuristic_sort_b(Bodies1,Bodies),
%	set_counter(id,0),
	index_body(Bodies,ID_B),
%	set_counter(id,0),
	index_head(Heads,ID_H),
	append(IndexedLP,ID_B,All1),
	append(All1,ID_H,DB),
%	writeln('done'),nl.
	true.
%
% id_rule(+LP,-IndexedLP,+IHeads,-OHeads,+IBodies,-OBodies)
% IndexedLP is list of terms id_rule(ID,R) whrer ID is index of rule R
% Heads     is list of all heads of LP
% I/OBodies are lists of all bodies of LP
%
index_rule( [],     [],                        Hs,  Hs,  Bs,  Bs  ).
index_rule( [R|LP], IndexedLP, Hs1, Hsx, Bs1, Bsx ) :-
	rule_body(R,[]), % if R is a fact -> R is no real rule
	rule_head(R,H),  % thus no rule id is given to R
	ord_union(H,Hs1,Hs2),
	index_rule(LP,IndexedLP,Hs2,Hsx,Bs1,Bsx).
index_rule( [R|LP], [id_rule(ID,R)|IndexedLP], Hs1, Hsx, Bs1, Bsx ) :-
	inc_counter(id),
	get_counter(id,ID),
	rule_head(R,H),
	rule_body(R,B),
	ord_union(B,Bs1,Bs2),
	ord_union(H,Hs1,Hs2),
	index_rule(LP,IndexedLP,Hs2,Hsx,Bs2,Bsx).

index_body( [], [] ).
index_body( [B|Bodies], [id_b(ID,B)|Indexed] ) :-
	inc_counter(id),
	get_counter(id,ID),
	integer_atom(ID,IDAtom),
	concat_atoms(b,IDAtom,Name),
	index_body(Bodies,Indexed).

index_head( [], [] ).
index_head( [H|Heads], [id_h(ID,H)|Indexed] ) :-
	inc_counter(id),
	get_counter(id,ID),
	index_head(Heads,Indexed).


%
% create prolog database for logic program containing relations:
% (c = compiled)
%                   c_rule_b/2, c_rule_h/2  (rule_db)
%                   c_h_0_b/2,  c_h_1_b/2   (head_db)
%                   

program_db :-
%	write('computing program database ...'),
	rule_body_db,
	rule_head_db,
	head_0_succsessors,
	head_1_succsessors,
%	writeln('done'),nl.
	true.

rule_body_db :- 
	findall(c_rule_b(R,B),
	       ( id_rule(R,Rule),
	         rule_body(Rule,Bs),
	         member(Body,Bs),
		 id_b(B,Body)  
	       ),DB),
	compile_clauses(DB,"tmp_rule_body_db.pl").

rule_head_db :-
	findall(c_rule_h(R,B),
	       ( id_rule(R,Rule),
	         rule_head(Rule,Hs),
	         member(Head,Hs),
		 id_h(B,Head)  
	       ),DB),
	compile_clauses(DB,"tmp_rule_head_db.pl").
	
head_0_succsessors :-
	findall(c_h_0_b(H,B),
	       ( id_h(H,Head),
	         id_b(B,Body),
		 body_pos(Body,Pos),
		 member(Head,Pos)
	       ),DB),
	compile_clauses(DB,"tmp_head_0_body_db.pl").

head_1_succsessors :-
	findall(c_h_1_b(H,B),
	       ( id_h(H,Head),
	         id_b(B,Body),
		 body_neg(Body,Neg),
		 member(Head,Neg)
	       ),DB),
	compile_clauses(DB,"tmp_head_1_body_db.pl").
	
