%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% Flags for system configuration                                          %
%                                                                         %
% Thomas Linke, Christian Anger                                           %
%                                                                         %
% last edit:  Aug 2003                                                    %
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

:- ensure_loaded(lists).

:- dynamic(flag/1).
:- assert(flag(noflag)).

flag_file('flags.ini').

%
% init_flag/0
% initializes the flag setting with file 'flags.ini'
%
init_flags :-
	flag_file(File),
	(
	  exists(File) ->
	  (
	    open(File,read,S),
	    read_flags(S,Flags),
	    set_flags(Flags)
	  );true
	).

%
% read_flags(+Stream,-Flags)
% Flags is a list of flags found in Stream
%
read_flags(S,Flags):-
	read(S,Term),
	(
	  Term = end_of_file ->
	  (
	    close(S),
	    Flags=[]
	  );
	  (
	    read_flags(S,Flags2),
	    Flags=[Term|Flags2]
	  )
	).

%
% write_flags(+Stream,+Flags)
% Flags is a list of flags to be written in Stream
%
write_flags(S,[]):-
	close(S).
write_flags(S,[Head|Tail]):-
	write(S,Head),write(S,'.'),nl(S),
	write_flags(S,Tail).


%
% clear_flag(Flag)
% takes a single flag or a list of flags and clears them
% also calls clear_flag_file, for handeling 'flags.ini'
%
clear_Flag( X ) :- clear_flag(X).

clear_flag( [] ).
clear_flag( [Head|Tail] ) :-
	clear_flag(Head),clear_flag(Tail).
clear_flag( degree(_X) ) :- writeln('Sorry, this flag can only be changed in nomore.pl'). 
clear_flag( Flag ) :-
	(
	  valid_flag(Flag) ->
	  (
	    get_flag(Flag) ->
	    (
	      (
		depends_on(Flag2,Flag) ->
		(
		  clear_flag(Flag2)
		);true
	      ),
	      retract(flag(Flag)),
	      write('Flag '),write(Flag),write(' cleared'),nl,
	      clear_flag_file(Flag)
	    );true
	  );
	  (
	    not_a_valid_flag(Flag)
	  )
	).

%
% clear_flag_file(Flag)
%
clear_flag_file(Flag):-
	flag_file(File),
	(
	  exists(File) ->
	  (
	    open(File,read,S),
	    read_flags(S,Flags),
	    (
	      choose(Flag,Flags,New) ->
	      (
		open(File,write,S2),
		write_flags(S2,New)
	      );true
	    )
	  );
	  true
	).

%
% set_flag(Flag)
% takes a single flag or a list of flags and sets them
% also calls set_flag_file, for handeling 'flags.ini'
%

set_Flag( X ) :- set_flag(X).

set_flag( [] ).
set_flag( [Head|Tail] ) :-
	set_flag(Head),set_flag(Tail).
set_flag( degree(_X) ) :- 
	writeln('sorry, this flag can only be changed in flags.ini'). 
set_flag( Flag ) :- 
	(
	  valid_flag(Flag) ->
	  (
	    get_flag(Flag) ->
	    (
	      true
	    );
	    (
	      (
		depends_on(Flag,Flag2) ->
		(
		  set_flag(Flag2)
		);true
	      ),
	      asserta(flag(Flag)),
	      write('Flag '),write(Flag),write(' set'),nl,
	      set_flag_file(Flag)
	    )
	  );
	  (
	    not_a_valid_flag(Flag)
	  )
	).

%
% set_flags(Flags)
% takes a list of flags and sets them without prior checking
%
set_flags( [] ).
set_flags( [Head|Tail] ) :-
	(
	  valid_flag(Head) ->
	  (
	    asserta(flag(Head))
	  );true
	),
	set_flags(Tail).


%
% set_flag_file(Flag).
% 
set_flag_file(Flag):-
	flag_file(File),
	(
	  exists(File) ->
	  (
	    open(File,read,S),
	    read_flags(S,Flags)
	  );
	  (
	    Flags=[]
	  )
	),
	(
	  choose(Flag,Flags,_New) ->
	  (
	    true
	  );
	  (
	    open(File,write,S2),
	    write_flags(S2,[Flag|Flags])
	  )
	).


%
% get_flag( Flag )
% is true iff Flag is set
%
get_flag( X ) :-
	call(flag(X)).

%
% flags/0 prints the current nomore user-flags settings
% iflags/0 prints the current internal-flags settings
% allflags/0 prints all current nomore flags settings
%
flags :- 
	write('----------- User Flags --------------'),nl,
	findall(X,(valid_flag(X),show_flag(X)),Flags),
	get_max_length(Flags,Lenght),
	print_flag(Flags,Lenght),
	nl.

iflags :- 
	write('-------- Internal Flags -------------'),nl,
	findall(X,(valid_flag(X),showall_flag(X)),Flags),
	get_max_length(Flags,Lenght),
	print_flag(Flags,Lenght),
	nl.

aflags :- 
	flags,
	iflags.
	       

print_flag( X ) :-
	(
	    valid_flag(X) ->
	    (
		get_flag(X) ->
		(
		    write(X),write(' = '),write(on),nl
		);
		(
		    write(X),write(' = '),write(off),nl
		)
	    );
	    (
		not_a_valid_flag(X)
	    )
	).

print_flag([],_).
print_flag( [X|Flags], Length ) :-
	(
	  valid_flag(X) ->
	  (
	    laenge(X,L),
	    (
	      get_flag(X) ->
	      (
		my_write(X,L,Length,on)
	      );
	      (
		my_write(X,L,Length,off)
	      )
	    )
	  );
	  (
	    not_a_valid_flag(X)
	  )
	),
	print_flag(Flags,Length).

%get_max_length(_,18).
get_max_length([],0).
get_max_length([X|Flags],L):-
	laenge(X,L1),
	get_max_length(Flags,L2),
	(
	  L1>L2 ->
	  (
	    L=L1
	  );
	  (
	    L=L2
	  )
	).

laenge(X,L):-
	atom(X),
	atom_length(X,L).
laenge(X,L):-
	not(atom(X)),
	X=..[A,B],
	atom_length(A,L1),
	atom_length(B,L2),
	L is L1 + L2 + 2.

my_write(X,L,Max,Value):-
	write(X),
	L1 is Max - L,
	write_times(L1,' '),
	write(' = '),
	write(Value),
	nl.

write_times(X,_):-
	X<1,!.
write_times(X,C):-
	write(C),
	X1 is X - 1,
	write_times(X1,C).

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

not_a_valid_flag(X):-
	write(X),write(' is not a valid flag. For a listing of all valid'),nl,
	write('flags and their settings try flags/0.'),nl.

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% Flag Database                                                           %
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
valid_flag(output).
% transformations
valid_flag(capture).
valid_flag(transformation).
valid_flag(reduction).
valid_flag(transformationSH).
valid_flag(transformationSB).
% propagation 
valid_flag(backprop).
%valid_flag(jumping).
valid_flag(ignore).
% statistics
valid_flag(profile).
% prolof system
valid_flag(prolog(eclipse)).
valid_flag(prolog(swi)).
valid_flag(prolog(sicstus)).
% daVinci flags
valid_flag(show).
valid_flag(names).
valid_flag(metainfo).
% variables
valid_flag(grounder).
% misc 
valid_flag(lookahead).
valid_flag(compile_clauses).
valid_flag(log).


depends_on(reduction,backprop).


show_flag(transformation).
show_flag(backprop).
show_flag(show).
show_flag(names).
show_flag(grounder).



showall_flag(output).
showall_flag(capture).
%showall_flag(jumping).
showall_flag(ignore).

showall_flag(transformationSH).
showall_flag(transformationSB).
showall_flag(reduction).

showall_flag(lookahead).
showall_flag(metainfo).
showall_flag(profile).
showall_flag(compile_clauses).
showall_flag(log).



:- init_flags.

