%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% program transformations                                                 %
%                                                                         %
% Thomas Linke, Christian Anger, Andreas Boesel                           %
%                                                                         %
% last edit:  July, 2003                                                  %
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

:-ensure_loaded(library(ordset)).
:-ensure_loaded(rules).

:-import ordset.

:-op(3,fx,not).

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% Program transformations work on list of rules (output parser)           %
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

capture(LP1,LP2) :- 
	hs(LP1,[],Hs),
	capture_all(LP1,Hs,LP2).

hs( [], In, In ).
hs( [R|Rs], HsIn, HsOut ) :-
	rule_head(R,Heads),
	ord_union(Heads,HsIn,HsIn1),
	hs( Rs, HsIn1, HsOut ).

capture_all( [], _, [] ).
capture_all( [R|Rs], Hs, [R1|Out] ) :-
	capture_rule(R,Hs,R1),!,
	capture_all(Rs,Hs,Out).     
capture_all( [R|Rs], Hs, Out ) :-       % rule cannot be captured -> delete
	%writeln(capture:deletes:R),
	capture_all(Rs,Hs,Out).

capture_rule( R, Hs, R1 ) :- 
	rule_body_head_name(R,Body,Head,Name),
	capture_all_bodies(Body,Hs,Body1),!,
	Body1 = [_|_],                      % some normal body has to be captured
	rule_body_head_name(R1,Body1,Head,Name).

capture_all_bodies( [], _Hs, [] ).
capture_all_bodies( [B|Body], Hs, [B1|Body1] ) :-
	capture_body(B,Hs,B1),!,
	capture_all_bodies(Body,Hs,Body1).
capture_all_bodies( [_|Body], Hs, Body1 ) :-
	capture_all_bodies(Body,Hs,Body1).

capture_body( B, Hs, B1 ) :- 
	pos_body(B,P),
	neg_body(B,N),
	(
	    P == [] -> true;
	    (
		ord_subset(P,Hs)
	    )
	),
	(
	    N == [] -> N1 = [];
	    (
		ord_intersect(N,Hs,N1),!
	    )
	),
	pos_body(B1,P),
	neg_body(B1,N1).


%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% Program transformations same head and same body                         %
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

program_transformation( LP1,LPx ):-
	( 
	    get_flag(transformationSH) -> 
	    (
		same_head(LP1,LP2)
	    );
	    (
		LP1 = LP2
	    )
	),
	(
	    get_flag(transformationSB) -> 
	    (
		same_body(LP2,LPx)
	    );
	    (
		LP2 = LPx
	    )
	).
%	same_head(LP2,LP3), -> errors check with ./Examples/NotWorkingFullTrans
%	redundant_bodies(LP3,LP4),
%	non_application(LP4,LP5),
%	fact_reduction(LP4,LPx).

%
% same_heads(+LP1,-LP2)
% transforms LP1 into LP2 where all rules with the same head are
% transformed into a single rule
%

same_head( LP1, LP2 ) :-
       	same_head(LP1,[],LP2).

same_head( [], Out, Out ).
same_head( [Rule1|Rest1], In, Out ) :-
	rule_head(Rule1,Head1),
	rule_name(Rule1,Name),

	get_bodies(Head1,[Rule1|Rest1],Rest2,Bs),
	rule_head(NewRule,Head1),
	rule_body(NewRule,Bs),
	rule_name(NewRule,Name),

	same_head(Rest2,[NewRule|In],Out).



get_bodies( _Head, [], [], [] ).
get_bodies( Head, [R|Rest1], Rest2, Bs ) :-
	(
	    rule_head(R,Head) ->
	    (
		rule_body(R,B),
		get_bodies(Head,Rest1,Rest2,Bs1),
		ord_union(B,Bs1,Bs)
	    );
	    (
		Rest2 = [R|Rest1],Bs=[]
	    )
	).

%
% same_body(+LP1,-LP2)
% transforms LP1 into LP2 where all rules with the same body are
% transformed into a single rule
%
same_body(LP1,LP2):-
       	same_body(LP1,[],LP2).

same_body([],[],[]).
same_body([Single],Others,LP_Out):-
	same_body(Others,[],LP),
	ord_insert(LP,Single,LP_Out).

same_body([Rule1,Rule2|Rest],Others,CLP):-
	rule_body(Rule1,Body1),
	(
	  rule_body(Rule2,Body1) ->
	  (
	    rule_head(Rule1,Head1),
	    rule_head(Rule2,Head2),
	    rule_name(Rule1,Name1),
	    rule_name(Rule2,Name2),
	    ord_union(Head1,Head2,Head3),
	    
	    rule_head(Rule3,Head3), 
	    rule_body(Rule3,Body1),
	    Name3 = Name1,
	    rule_name(Rule3,Name3), 	    
	    same_body([Rule3|Rest],Others,CLP)
	  );
	  (
	    same_body([Rule1|Rest],[Rule2|Others],CLP)
	  )
	).

%%%% not used right now!!!!!!!!!!!!!!!!!!!!!!!!!!!
%
% redundant_bodies(+LP1,-LP2)
% removes redundant bodies
%
redundant_bodies([],[]).
redundant_bodies([Rule|Rules],[Check|NewRules]):-
	rule_name(Rule,Name),
	rule_head(Rule,Head),
	rule_body(Rule,Body),
	redundant_bodies_check(Body,[],Checked),
	rule_name(Check,Name),
	rule_head(Check,Head),
	rule_body(Check,Checked),
	redundant_bodies(Rules,NewRules).


redundant_bodies_check([],[],[]).
redundant_bodies_check([Single],Others,Bodies_Out):-
	redundant_bodies_check(Others,[],Bodies),
	ord_insert(Bodies,Single,Bodies_Out).
redundant_bodies_check([Body1,Body2|Bodies],Others,Checked):-
	(
	  contained(Body1,Body2,Body) ->
	  (
	    redundant_bodies_check([Body|Bodies],Others,Checked)
	  );
	    redundant_bodies_check([Body1|Bodies],[Body2|Others],Checked)
	).
	
contained(Body1,Body2,Body1):-
	body_neg(Body1,Neg1),
	body_neg(Body2,Neg2),
	ord_subset(Neg1,Neg2),
	body_pos(Body1,Pos1),
	body_pos(Body2,Pos2),
	ord_subset(Pos1,Pos2).
contained(Body1,Body2,Body2):-
	body_neg(Body1,Neg1),
	body_neg(Body2,Neg2),
	ord_subset(Neg2,Neg1),
	body_pos(Body1,Pos1),
	body_pos(Body2,Pos2),
	ord_subset(Pos2,Pos1).



%
% non_application(+LP1,-LP2)
% transforms LP1 into LP2 where all rules that are non-applicable
% (e.g. x :- a, not a ) are deleted
%
non_application([],[]).
non_application([Rule|Rules],[NewRule|NewRules]):-

	rule_body(Rule,Body),
	non_application_body(Body,NewBody),
	\+(NewBody == []),!,           %rules with empty body will be deleted!!
		                       %(facts should have body (body([],[]))    
	rule_head(Rule,Head),
	rule_name(Rule,Name),
	rule_head(NewRule,Head),
	rule_body(NewRule,NewBody),
	rule_name(NewRule,Name),
	
	non_application(Rules,NewRules).

non_application([_Rule|Rules],NewRules):-
	non_application(Rule,NewRules).


%
% non_application_body(+Body,-NewBody),
%
non_application_body([],[]).
non_application_body([Body|Bodies],[Body|NewBodies]):-
	pos_body(Body,Pos),
	neg_body(Body,Neg),
        ord_disjoint(Pos,Neg),!,	
	non_application_body(Bodies,NewBodies).

non_application_body([_|RB],NRB):-
	non_application_body(RB,NRB).



%
% fact_reduction
% merges all facts into one rule
%
fact_reduction(LP,LPx):-
	collect_facts(LP,LP2,Fact_Heads),
	(
	  Fact_Heads = [] ->
	  ( % fact_free program
	    LP = LPx
	  );
	  (
	    pos_body(Body,[]),
	    neg_body(Body,[]),
	    rule_body(Fact,[Body]),
	    rule_name(Fact,the_fact),
	    rule_head(Fact,Fact_Heads),
	    ord_insert(LP2,Fact,LPx)
	  )
	).

% removes all facts from LP and collects their heads
collect_facts([],[],[]).
collect_facts([Rule|Rules],Rules_Out,Facts):-
	fact_rule(Rule),!,
	rule_head(Rule,Heads),
	collect_facts(Rules,Rules_Out,Fact_Heads),
	ord_union(Fact_Heads,Heads,Facts).
collect_facts([Rule|Rules],[Rule|Rules_Out],Facts):-
	collect_facts(Rules,Rules_Out,Facts).
	






