#include "mpi.h"
#include <iostream>
#include <fstream>
#include <cassert>
#include <memory>
#include <vector>
#include <queue>
#include <algorithm>
#include <iterator>
#include <ctime>
#include <sys/time.h>//for jean's newest timing
#include <interfaces/smodels_program_impl.h>
#include <interfaces/expander_chooser_impl.h>
#include <interfaces/choicestodo_impl.h>
#include <interfaces/chooser_impl.h>
#include <interfaces/program_factory.h>
#include <interfaces/communicator.h>
#include <interfaces/ticks.h>
#include <interfaces2/command_line_parser.h>

using namespace MPI;
using namespace std;
using namespace Platypus;

//typedefs for passing partial models around
//typedef ChoicesToDo<DelegatableChoiceType, std::vector<DelegatableChoiceType> > ChoiceToDoType;
//typedef Expander<DelegatableExpanderInterfaceType> ExpanderType;
//typedef std::auto_ptr<ExpanderType> SafeExpander;
//typedef Chronological<std::vector<DelegatableChoiceType> > DelegatablePolicyType;

//typedefs for passing internal choice points around
typedef Expander<LocalExpanderInterfaceType> InternalExpanderType;
typedef std::auto_ptr<InternalExpanderType> InternalSafeExpander;
typedef ChoicesToDo<ChoiceType, std::vector<ChoiceType> > InternalChoicesToDoType;
typedef DelegatablePolicy<std::vector<ChoiceType> > InternalDelegatablePolicyType;
typedef InternalChoicesToDoType::CollectionType DelegatableCollectionType;

//Time class for jean's newest timing code
namespace
{
  class Time
  {
  public:
    Time()
      :	start_(0)
	,	end_(0)
    {
      assert(frequency_ && "This platform does not support high performance clocks!");
    }
    inline uint64 frequency() const
    {
      return frequency_;
    }
    inline void start()
    {
      start_ = end_ = stamp();
    }
    inline void stop()
    {
      end_ = stamp();
    }
    inline uint64 difference() const
    {
      return end_ - start_;
    }
    uint64 stamp() const;
  private:
    static uint64 calculateFrequency();
  private:
    uint64 start_, end_;
    static const uint64 frequency_;
  };
  uint64 Time::stamp() const
  {
    timeval tv;
    gettimeofday(&tv, 0);
    
    uint64 t = tv.tv_usec;
    t += tv.tv_sec*1000000;
    return t;
  }
  
  uint64 Time::calculateFrequency()
  {
    return 1000000;
  }
  
  const uint64 Time::frequency_ = Time::calculateFrequency();
  
}


//debug ostream overloads
std::ostream & operator<<(std::ostream & os, const Platypus::ChoiceType & choice){                   
                
  os << "name: " << choice.atom_.name() << " positive: " << choice.positive_ << " expired: " << choice.expired_ << endl; cout.flush();          
  return os;  
                
}             
std::ostream & operator<<(ostream & os, const DelegatableCollectionType & dct){                      
                
  DelegatableCollectionType::const_iterator it = dct.begin();           
  for(;it != dct.end(); it++){             
                
    os << (*it);                           
  }           
                
  return os;  
}             
std::ostream & operator<<(std::ostream & os, const std::vector<int> & vi){                      
                
  std::vector<int>::const_iterator it = vi.begin();           
  for(;it != vi.end(); it++){             
                
    os << (*it) << " ";                           
  }           
                
  return os;  
}     

namespace{

  //process(or) ranks and ids                 
  int GATE_KEEPER = 0;                        
  int MASTER = 1;
               
  //status table flags                        
  const int WORKING = -1;                     
  const int REQUESTING = -2;                  
  const int DENIED = -3;                      
  const int INITIAL_VALUE = -7;               
                
  //MESSAGE TAGS 
  //tags for messages from GATE_KEEPER to WORKERS                          
  const int WORK_REQUEST_FROM_GK = 0;         
  const int TERMINATE = 1;                    
  const int BOOT_STRAP = 2;                  
  const int REISSUE_REQUEST = 3;              
  //tags for messages from WORKERS TO GATEKEEPER                           
  const int WORK_REQUEST_TO_GK = 4;           
  const int DENY_CONFIRMATION = 5;            
  const int SUCCESS_CONFIRMATION = 6;         
  const int TERMINATE_CONFIRMATION = 7;       
  const int WORKING_CONFIRMATION = 8;         
  const int MESSAGE_COUNT = 9;                
  //tags for messages from WORKER to WORKER   
  const int MODEL = 10;
  const int MODEL_RECEIPT = 11;
  //receipts for various messages received by GATE_KEEPER                  
  const int WORKING_CONFIRMATION_RECEIPT = 12;
  const int BOOT_STRAP_RECEIPT = 13;          
  const int SUCCESS_CONFIRMATION_RECEIPT = 14;
  const int DENY_CONFIRMATION_RECEIPT = 15;   
  const int WORK_REQUEST_TO_GK_RECEIPT = 16;                    
  
  const char* const PROGRAM_NAME = "demo";

  //print only the positive part of the partial model
  void printPositiveAtoms(const PartialModel& pm){
    copy(pm.positiveAtoms().begin(), pm.positiveAtoms().end(), ostream_iterator<AtomType>(cout, " "));
    cout << endl; cout.flush();
  }
  
  //print the positive atoms within a given partial model and the number of models generated so far
  void printAnswerSet(const PartialModel& pm, unsigned noModels, int myid){
    cout << "Answer " << myid << "." << noModels << ": ";cout.flush();
    printPositiveAtoms(pm);		
  }

  //class MessageConverter converts platypus data types into a vector for communication among agents
  const int PARTIAL_ASSIGNMENT_MESSAGE_DELIMITER = -15;  //the delimiter for the choice info within the message
  const int CHOICE_MESSAGE_DELIMITER = -15;
  const int PID_LOCATION = 1;
  const int SIZE_LOCATION = 2;

  class MessageConverter{

  private:
    
    //definitions specific to the MODEL message    
    typedef std::vector<int> MessageType;
    MessageType message_;
    
  public:
    
    MessageConverter(){}
    
    ~MessageConverter(){}
    
    const MessageType & getMessage(){ return message_; } 

    void createMessage(const DelegatableCollectionType & choiceType, int pid){

      //cout << MPI::COMM_WORLD.Get_rank() << " is in createMessage." << endl; cout.flush();
      //cout.flush();

      message_.clear();

      // the first two spaces will be for header and size of the overall message
      message_.push_back(CHOICE_MESSAGE_DELIMITER);
      message_.push_back(pid);

      //cout << "choice type size: " << choiceType.size() << endl; cout.flush();

      message_.push_back((choiceType.size() * 3) + 3);//push the total size of the message
      
      DelegatableCollectionType::const_iterator it = choiceType.begin();
      for(; it != choiceType.end(); it++){

	message_.push_back(it->atom_.id());
	message_.push_back(it->positive_);
	message_.push_back(it->expired_);
      }

      //cout << MPI::COMM_WORLD.Get_rank() << " is leaving createMessage." << endl; cout.flush();
      //cout.flush();

    }

    void createMessage(const PartialModel & m, int pid){
      
      message_.clear();

      //add the first delimiter   
      message_.push_back(PARTIAL_ASSIGNMENT_MESSAGE_DELIMITER);
      
      //push on the sizes for the three collections
      message_.push_back(m.positiveAtoms().size());
      message_.push_back(m.negativeAtoms().size());
      message_.push_back(m.unknownAtoms().size());
      
      //push on the id for all the positive atoms in the partial model in the choice object
      for(PartialModelType::CollectionType::const_iterator it = m.positiveAtoms().begin(); 
	  it != m.positiveAtoms().end();
	  ++it){
	message_.push_back(((Platypus::Atom)*it).id());
      }
      
      //do the same for tha negative atoms
      for(PartialModelType::CollectionType::const_iterator it = m.negativeAtoms().begin();
	  it != m.negativeAtoms().end();
	  ++it){
	message_.push_back(((Platypus::Atom)*it).id());
      }
      
      //finally do the same for all the unknown atoms
      for(PartialModelType::CollectionType::const_iterator it = m.unknownAtoms().begin();
	  it != m.unknownAtoms().end();
	  ++it){
	message_.push_back(((Platypus::Atom)*it).id());
      }
      //push on the last delimiter
      message_.push_back(PARTIAL_ASSIGNMENT_MESSAGE_DELIMITER);

      vector<int>::iterator it = message_.begin();
      //add the id of the worker that sent the message
      message_.insert(++it, pid);
      //add two to the size because you are adding two more pieces of data to the vector
      message_.insert(++it, message_.size() + 2);  
      
    }
    
    DelegatableCollectionType convertToCollection(const int message[], const SmodelsEnhancedProgram & program){

      //cout << MPI::COMM_WORLD.Get_rank() << " is in convertMessage." << endl; cout.flush();
      //cout.flush();

      assert(message[0] == CHOICE_MESSAGE_DELIMITER);

      DelegatableCollectionType choices;

      int messageSize = message[2];  //2 is where the size of the entire array is stored;
      //magic number 3 is start of actual choice info
      for(int i = 3; i < messageSize; i+=3){

	AtomType atom(message[i], program);
	ChoiceType choice(atom, message[i+1], message[i+2]);
	choices.push_back(choice);

      }

      return choices;
    }

    //create a DelegatableChoice using an array as passed through an MPI message
    Platypus::PartialModel convertToPartialAssignment(const int message[], const SmodelsEnhancedProgram & program){
      
      assert(message[0] == PARTIAL_ASSIGNMENT_MESSAGE_DELIMITER);

      const unsigned CHOICE_INFO_START     = 3; //this is the index where the actual info. starts
      
      const unsigned posSizeIndex          = CHOICE_INFO_START;
      const unsigned negSizeIndex          = CHOICE_INFO_START + 1;
      const unsigned unknownSizeIndex      = CHOICE_INFO_START + 2;
      const unsigned posSetStartIndex      = CHOICE_INFO_START + 3;
      const unsigned posSetEndIndex        = posSetStartIndex + (message[posSizeIndex] - 1);
      const unsigned negSetStartIndex      = posSetEndIndex + 1;
      const unsigned negSetEndIndex        = negSetStartIndex + (message[negSizeIndex] - 1);
      const unsigned unknownSetStartIndex  = negSetEndIndex + 1;
      const unsigned unknownSetEndIndex    = unknownSetStartIndex + (message[unknownSizeIndex] - 1); 
      
      list<Platypus::Atom> posSet, negSet, unknownSet;
      
      for(unsigned i=posSetStartIndex;i<=posSetEndIndex;i++){
	
	Platypus::Atom posTemp(message[i], program);
	posSet.push_back(posTemp);
      }
      
      for(unsigned i=negSetStartIndex;i<=negSetEndIndex;i++){
	
	Platypus::Atom negTemp(message[i], program);
	negSet.push_back(negTemp);
      }  
      
      for(unsigned i=unknownSetStartIndex;i<=unknownSetEndIndex;i++){
	
	Platypus::Atom unknownTemp(message[i], program);
	unknownSet.push_back(unknownTemp);
      }
      
      Platypus::PartialModelType pm(posSet.begin(), posSet.end(),
				   negSet.begin(), negSet.end(),
				   unknownSet.begin(), unknownSet.end(),
				   program);
      
      return pm;
      
    }
    
  };

  //global counters
  int myid = -1, np = -1;
  
  //counters local to a process
  int localNodes = 0;
  int localInconsist = 0;
  unsigned localAnswers = 0;
  unsigned localExpands = 0;
  unsigned localDelegations = 0;
  int localMessageCount = 0;

  unsigned conflicts = 0;
  unsigned expanderInits = 0;
  unsigned backtracks = 0;
  
}

namespace Platypus{

  //another debug ostrean overload
  std::ostream & operator<<(std::ostream & os, const std::vector<int> & vi){                      
    
    std::vector<int>::const_iterator it = vi.begin();           
    for(;it != vi.end(); it++){             
      
      os << (*it) << " ";                           
    }           
    
    return os;  
  }     
  
  void internalLocalExpand(InternalSafeExpander & expander, 
			   InternalChoicesToDoType & choicesToDo, 
			   InternalChoicesToDoType::BacktrackLevel level,
			   const SmodelsEnhancedProgram & program,
			   bool & extraRun,
			   bool suppressed,
			   Communicator & comm){

    extraRun = false;

    if(!expander->done()){

      ChoiceType c(expander->makeChoice());

      //cout << "ChoiceType before expand: " << endl << c << endl;
      expander->expand(c);
      localExpands++;

      choicesToDo.add(c);

      //if there was request for work
      if(comm.probe(GATE_KEEPER, WORK_REQUEST_FROM_GK)){
    
	//Receive the node request
        int requestingID = comm.receive<int>(MPI::INT, GATE_KEEPER, WORK_REQUEST_FROM_GK); 

	MessageConverter converter;
	converter.createMessage(choicesToDo.nextDelegatableChoice(level), MPI::COMM_WORLD.Get_rank());
	vector<int> messageVec = converter.getMessage();

	//Send the info to the requesting processor and confirm with gatekeeper
	comm.send<int>(messageVec, MPI::INT, requestingID, MODEL);
	//cout << myid << " sent " << requestingID << " MODEL." << endl; cout.flush();

	comm.send<int>(requestingID, MPI::INT, GATE_KEEPER, SUCCESS_CONFIRMATION);
	//cout << myid << " sent GATE_KEEPER SUCCESS_CONFIRMATION." << endl; cout.flush();

	//make sure you go through the loop one more time
	extraRun = true;
      }
    }
    else{
      if(!(expander->state() & InternalExpanderType::HAS_CONFLICT)){
	++localAnswers;
	if(!suppressed)
	  printAnswerSet(expander->partialModel(), localAnswers, myid);
      }
      else
	++conflicts;
      
      if(choicesToDo.hasChoice()){
	ChoiceType c(choicesToDo.nextChoice());
	
	++backtracks;
	expander->backtrackTo(c.atom_);
	extraRun = true;
      }
    }
  }
  
  //control method for non-chronological back-tracking
  void control(const SmodelsEnhancedProgram & program, 
	       const DelegatableCollectionType & dct,
	       InternalChoicesToDoType::BacktrackLevel level,
	       bool suppressed,
	       Communicator & comm){


    PartialModel pm(program);
    InternalDelegatablePolicyType policy;
    InternalChoicesToDoType choicesToDo(&policy);;

    assert(choicesToDo.choicesToDo().empty());

    DelegatableCollectionType::const_iterator it = dct.begin();
    for(;it != dct.end(); ++it){
      if(it->positive_ == true)
	pm.setTrue(it->atom_);
      else
	pm.setFalse(it->atom_);
      
      choicesToDo.add(*it);
    }

    bool extraRun = true;
      
    InternalSafeExpander expander(new InternalExpanderType(pm, program));

    comm.send(GATE_KEEPER, WORKING_CONFIRMATION);
    //cout << myid << " sent GATE_KEEPER WORKING_CONFIRMATION in control." << endl; cout.flush();

    while(choicesToDo.hasChoice() || extraRun){
      
      extraRun = false;
      internalLocalExpand(expander, choicesToDo, level, program, extraRun, suppressed, comm);
    }
    
    return;
    
  }

  void bootstrap(const SmodelsEnhancedProgram & program, 
		 const DelegatableCollectionType & dct,  
		 InternalChoicesToDoType::BacktrackLevel level,
		 bool suppressed,                        
		 Communicator & comm){                   
      
    //int dummy = -1;
    //MPI::COMM_WORLD.Send(&dummy, 1, MPI::INT, GATE_KEEPER, BOOT_STRAP); 
    comm.send(GATE_KEEPER, BOOT_STRAP); 
    //cout << myid << " sent GATE_KEEPER BOOT_STRAP." << endl; cout.flush();

    PartialModel pm(program);                          
    InternalDelegatablePolicyType policy;                      
    InternalChoicesToDoType choicesToDo(&policy);
                          
    assert(choicesToDo.choicesToDo().empty());         
                        
    DelegatableCollectionType::const_iterator it = dct.begin();                     
    for(;it != dct.end(); ++it){                       
      if(it->positive_ == true)                        
        pm.setTrue(it->atom_);                         
      else                
        pm.setFalse(it->atom_);                        
                          
      choicesToDo.add(*it);                            
    }                     
                          
    bool extraRun = true; 
                          
    InternalSafeExpander expander(new InternalExpanderType(pm, program));                           
                          
    //comm.send(GATE_KEEPER, WORKING_CONFIRMATION);              
    //cout << myid << " sent GATE_KEEPER WORKING_CONFIRMATION in bootstrap." << endl; cout.flush();

    while(choicesToDo.hasChoice() || extraRun){ 

      //extraRun = false;   
      internalLocalExpand(expander, choicesToDo, level, program, extraRun, suppressed, comm);      
    }                     

    return;               
                          
  }         

  void dequeueRequest(vector<int> & statusTable, 
		    queue<int> & requests, 
		    Communicator & comm,
		    int source){
      int queuedRequest = requests.front();
      requests.pop();
      statusTable[source] = queuedRequest;
      comm.send(queuedRequest, MPI::INT, source, WORK_REQUEST_FROM_GK);
      //cout << "GATE_KEEPER sent " << source << " WORK_REQUEST_FROM_GK for " << queuedRequest << "." << endl; cout.flush();
  }

}


int main(int argc, char * argv[]){

  double tv1 = 0, tv2 = 0, tv3 = 0;
  unsigned totalMessages = 0;
  unsigned totalAnswers = 0;
  unsigned totalExpands = 0;

  MPI::Init(argc,argv);  
  myid = MPI::COMM_WORLD.Get_rank();
  np = MPI::COMM_WORLD.Get_size();

  //newest Jean timing
  Time ct;
  ct.start();
  
  //jean's new timing
  //uint64 freq = frequency();
  //uint64 t1 = ticks();
  
  //old timing
  //tv1 = MPI::Wtime();

  DistributedCommandLineParser parser;                          
      
  //gather and process command line arguments                   
  if(!parser.processArgs(argc, argv)){                          
    parser.printUsage();           
    return 0;                      
  }   

    InternalChoicesToDoType::BacktrackLevel BTlevel = InternalChoicesToDoType::DEEP;
    string runLevel = "";
    if(parser.checkLevel()){
      if(parser.getLevel() == "shallow"){
	BTlevel = InternalChoicesToDoType::SHALLOW;
	runLevel = "SHALLOW";
      }
      else if(parser.getLevel() == "mid"){
	BTlevel = InternalChoicesToDoType::MID;
	runLevel = "MID";
      }
      else if(parser.getLevel() == "deep"){
	BTlevel = InternalChoicesToDoType::DEEP;
	runLevel = "DEEP";
      }
      else{
	parser.printUsage();
	MPI::COMM_WORLD.Abort(MPI_ERR_ARG);  
      }
    }
    else{
      BTlevel = InternalChoicesToDoType::DEEP;
      runLevel = "DEEP";
    }

  SmodelsEnhancedProgram& program = ProgramFactory::instance().create();                     

  //read program from either stdin or a file                    
  if(!parser.checkFile()){         
    try{                           
      program.setup(cin);          
    }catch(std::runtime_error & e){
      cout << endl << "There was an error with your program." << endl; cout.flush();                       
      MPI::COMM_WORLD.Abort(MPI_ERR_ARG);                       
    } 
  }   
  else{                            
    string theFile;                
    parser.getFile(theFile);       
    ifstream file(theFile.c_str());
    try{                           
      program.setup(file);         
    }catch(std::runtime_error & e){
      cout << endl << "There was an error reading your file." << endl; cout.flush();                       
      MPI::COMM_WORLD.Abort(MPI_ERR_ARG);                     
    } 
  }                 

  //wait until all agents get to this point 
  //MPI::COMM_WORLD.Barrier();

  try{

    int messageSource = MPI::ANY_SOURCE;
    int messageTag = MPI::ANY_TAG;
    
    vector<int> statusTable(np);
    queue<int> qRequests;
 
    switch(myid){
      /***************************
       *  GATE_KEEPER CODE
       ***************************/
    case 0:
      {

	//tv3 = MPI::Wtime();
	//cout << "GATE_KEEPER: start time is " << (tv3 - tv1) << endl;cout.flush();
	
	//initialize the status table      
	//make sure the values in here are initialized to a negative value (other than -1 or -2)
	//initialize the status table
	for(int i = 0;i < (int)statusTable.size();i++){
	  statusTable[i]= INITIAL_VALUE;
	}
	
	//intialize gate keeper specific counters and buffers
	int denyCount = 0;
	int messageBuffer = -1;
	int message = 0;
	
	MPI::Status probeStatus;

	Communicator gkComm;
	
	//receive for the bootstrap message from MASTER worker
	gkComm.receive<int>(MPI::INT, MASTER, BOOT_STRAP);

	statusTable[MASTER] = WORKING;
	
	int loopExitCondition = np - 1;
	
	//exit when the queue is full
	while(/*(denyCount < loopExitCondition)){ && */((int)qRequests.size() < loopExitCondition)){

	  //non-blocking probe for the first available message
	  if(gkComm.probe(MPI::ANY_SOURCE, MPI::ANY_TAG, probeStatus)){

	    messageSource = probeStatus.Get_source();
	    messageTag = probeStatus.Get_tag();
 
	    //receive the message probed for
	    message = gkComm.receive<int>(MPI::INT, messageSource, messageTag);

	    //a WORKER has successfully received work and started working
	    if(messageTag == WORKING_CONFIRMATION){
	      //set the node to working
	      if(!qRequests.empty())
		dequeueRequest(statusTable, qRequests, gkComm, messageSource);
	      else
		statusTable[messageSource] = WORKING;
	    }
	    //a WORKER has successfully shared work with another WORKER
	    else if(messageTag == SUCCESS_CONFIRMATION){
	      
	      if(!qRequests.empty())
		dequeueRequest(statusTable, qRequests, gkComm, messageSource);
	      else
		statusTable[messageSource] = WORKING;
	    }
	    //a WORKER could not complete a work request because it completed its own work
	    //before the request could be fulfilled
	    else if(messageTag == DENY_CONFIRMATION){
	      
	      int denyingNode = messageSource;
	      int requestingNode = message;
	      
	      //set denying processor from in progress to requesting
	      statusTable[denyingNode] = REQUESTING;
	      qRequests.push(requestingNode);
	    }
	    //Message received is a work request from a worker
	    else if(messageTag == WORK_REQUEST_TO_GK){
	      
	      int requestingNode = messageSource;
	      //gkComm.send(messageSource, WORK_REQUEST_TO_GK_RECEIPT); 
	      //cout << "GATE_KEEPER sent " << messageSource << " WORK_REQUEST_TO_GK_RECEIPT." << endl; cout.flush();

	      if (statusTable[requestingNode] == DENIED){
		denyCount--;
	      }
	      
	      //set the requesting node to requesting
	      statusTable[requestingNode] = REQUESTING;
	      
	      //look for working node
	      bool workingNodeFound = false;
	      int workingNodeIndex = 0;
	      for(int i = 1;i < (int)statusTable.size();i++){
		if(statusTable[i] == WORKING){
		  workingNodeIndex = i;
		  workingNodeFound = true;
		  break;
		}
	      }
	      //if working node found
	      //place requesting rank in working node index, indicates in progress
	      //  (IN PROGRESS nodes cannot be asked again)
	      //SEND MESSAGE TO WORKING NODE WITH RANK OF REQUESTING NODE AS THE MESSAGE CONTENT
	      if(workingNodeFound){
		
		statusTable[workingNodeIndex] = requestingNode;
		gkComm.send(requestingNode, MPI::INT, workingNodeIndex, WORK_REQUEST_FROM_GK);
		//cout << "GATE_KEEPER sent " << workingNodeIndex << " WORK_REQUEST_FROM_GK for " << requestingNode << "." << endl; cout.flush();

	      }
	      //if no working node was found look for an in progress node
	      else{
		bool inProgressNodeFound = false;
		if(!workingNodeFound){
		  for(int i = 1;i < (int)statusTable.size();i++){
		    if(statusTable[i] > 0){
		      inProgressNodeFound = true;
		      break;
		    }
		  }
		}
		//if no working node was found but there is a node that is in progress than do not change the
		//status of the requesting node om the table. Simply let it reissue the request.
		if(inProgressNodeFound){
		  
		  statusTable[requestingNode] = REQUESTING;
		  qRequests.push(requestingNode);
		}
		// if working node not found and there is nothing in progress deny request for work 
		// (status of requesting node is set to denied),
		// reset counts to old values, and send reissue request 
		// message to requesting node 
		else{
		  statusTable[requestingNode] = DENIED;
		  denyCount++;
		  qRequests.push(requestingNode);
		}
	      }
	    }
	  }
	}
	
	//int i = 1;
	//all workers have been denied, send the terminate message
	for(int i = 1; i < (int)statusTable.size();i++){
	  int messageDest = i;
	  gkComm.send<int>(messageBuffer, MPI::INT, messageDest,TERMINATE); 
	  //cout << "GATE_KEEPER sent " << messageDest << " TERMINATE." << endl; cout.flush();
	  
	}
	
	//wait for the confirmation of the termination message from the workers before exiting
	const unsigned messageSize = 3;
	int stats[messageSize] = {0,0,0}; 
	int terminationsReceived = 0;
	MPI::Status pStatus;
	int endCondition = (int)statusTable.size()-1;
	
	while(terminationsReceived < endCondition){
	  //non-blocking probe for the first available message
	  if(gkComm.probe(MPI::ANY_SOURCE, TERMINATE_CONFIRMATION, pStatus)){	    
	    int source = pStatus.Get_source();
	    
	    gkComm.receive<int>(stats, messageSize, MPI::INT, source, TERMINATE_CONFIRMATION);
	    totalMessages += stats[0]; 
	    totalAnswers += stats[1];
	    totalExpands += stats[2];
	    terminationsReceived++;
	  }
	}  
	
	totalMessages += gkComm.totalSent();
	
	break;
      }
    default:
      //MASTER and SLAVE code
      {


	//tv3 = MPI::Wtime();
	//cout << "WORKER << " << myid << ": elapsed time after start is " << (tv3 - tv1) << endl;cout.flush();
	//usleep(10000);
	
	int requestingRank = -1;

	Communicator workerComm;
	
	//master starts off the show by calling control first
	if(myid == MASTER){
	  DelegatableCollectionType collection;
	  bootstrap(program, collection, BTlevel, parser.checkSuppressed(), workerComm);
	}
       
	workerComm.send(GATE_KEEPER, WORK_REQUEST_TO_GK);

	MPI::Status wStatus;
	int mSource = MPI::ANY_SOURCE;
	int mTag = MPI::ANY_TAG;

	//tv3 = MPI::Wtime();
	//cout << "WORKER << " << myid << ": elapsed time before while is " << (tv3 - tv1) << endl;cout.flush();

	while(1){

	  if(workerComm.probe(MPI::ANY_SOURCE, MPI::ANY_TAG, wStatus)){

	    mSource = wStatus.Get_source();
	    mTag = wStatus.Get_tag();
	    
	    if(mTag == MODEL){
	      int intBuffer[10000];
	      int intBufferSize = 10000;
	      
	      workerComm.receive<int>(intBuffer, intBufferSize, MPI::INT, mSource, MODEL);
	      
	      MessageConverter converter;
	      
	      DelegatableCollectionType dct = converter.convertToCollection(intBuffer, program);

	      //tv3 = MPI::Wtime();
	      //cout << "WORKER << " << myid << ": elapsed time before control is " << (tv3 - tv1) << endl;cout.flush();

	      control(program, dct, BTlevel, parser.checkSuppressed(), workerComm);

	      //tv3 = MPI::Wtime();
	      //cout << "WORKER << " << myid << ": elapsed time after control is " << (tv3 - tv1) << endl;cout.flush();
	      
	      workerComm.send(GATE_KEEPER, WORK_REQUEST_TO_GK);
	    }
	    else if(mTag == WORK_REQUEST_FROM_GK){   
	      
	      requestingRank = workerComm.receive<int>(MPI::INT, mSource, WORK_REQUEST_FROM_GK);
	      workerComm.send<int>(requestingRank, MPI::INT, mSource, DENY_CONFIRMATION);

	      //tv3 = MPI::Wtime();
	      //cout << "WORKER << " << myid << ": elapsed time after sending DENY_CONFIRMATION is " << (tv3 - tv1) << endl;cout.flush();
	      //cout << myid << " sent GATE_KEEPER DENY_CONFIRMATION for " << requestingRank << "." << endl; cout.flush();
	    }
	    else if(mTag == TERMINATE){ 
	      workerComm.receive<int>(MPI::INT, mSource, TERMINATE); 

	      //tv3 = MPI::Wtime();
	      //cout << "WORKER << " << myid << ": elapsed time after receiving TERMINATE is " << (tv3 - tv1) << endl;cout.flush();
	      break;
	    }
	  }
	}
	
	vector<int> stats;
	int messageCountBuffer = workerComm.totalSent() + 1;
	int answerCountBuffer = localAnswers;
	int expandCountBuffer = localExpands;
	stats.push_back(messageCountBuffer);
	stats.push_back(answerCountBuffer);
	stats.push_back(expandCountBuffer);
	workerComm.send<int>(stats, MPI::INT, GATE_KEEPER, TERMINATE_CONFIRMATION);
	//cout << myid << "sent GATE_KEEPER TERMINATE_CONFIRMATION." << endl; cout.flush();

      }   
    }


    MPI::Status lStatus;
    Communicator lComm;
    if(myid == GATE_KEEPER){
      while(lComm.probe(MPI::ANY_SOURCE, MPI::ANY_TAG, lStatus)){	    
	lComm.receive<int>(MPI::INT, MPI::ANY_SOURCE, MPI::ANY_TAG);
      }
    }
    else{
      while(lComm.probe(MPI::ANY_SOURCE, MPI::ANY_TAG, lStatus)){
	int tag = lStatus.Get_tag();
	if(tag == MODEL){
	  int intBuffer[10000];
	  int intBufferSize = 10000;
	  lComm.receive<int>(intBuffer, intBufferSize, MPI::INT, MPI::ANY_SOURCE, MODEL);
	}
	else{
	  lComm.receive<int>(MPI::INT, MPI::ANY_SOURCE, MPI::ANY_TAG);
	}
      }
    }
      
    //MPI::COMM_WORLD.Barrier();
    if(myid == GATE_KEEPER){
      
      //jean's timing
      //uint64 t2 = ticks();
      //uint64 t3 = t2 - t1;
      //double t_time = (t3 * 1.0)/(freq * 1.0);

      //newest Jean timing
      ct.stop();
      
      //tv2 = MPI::Wtime();
      cout << "solutions: " << totalAnswers << endl; cout.flush();
      cout << "messages: " << (totalMessages) << endl; cout.flush();
      cout << "expansions: " << (totalExpands) << endl; cout.flush();
      //cout << "total time: " << /*t_time*/tv2-tv1 << endl; cout.flush();
      cout << "total time: " << (double)(ct.difference()*1.0)/(ct.frequency()*1.0) << endl;cout.flush();
  

    }
 
    //tv3 = MPI::Wtime();
    //cout << myid << ": elapsed time before final barrier is " << (tv3 - tv1) << endl;cout.flush();
    
    MPI::COMM_WORLD.Barrier();

    //tv3 = MPI::Wtime();
    //cout << myid << ": elapsed time after final barrier is " << (tv3 - tv1) << endl;cout.flush();

    MPI::Finalize();        
    return 0; 

  }catch(const PlatypusException& e){
    
    cerr << e.what() << endl; cout.flush();
    MPI::COMM_WORLD.Abort(MPI_ERR_ARG);
    
  }

  MPI::COMM_WORLD.Abort(MPI_ERR_ARG);    
  return 0; 
  
}







