// $Id: choicestodo_impl.h,v 1.10 2005/01/17 20:30:30 rtichy Exp $

#ifndef PLATYPUS_CHOICESTODO_IMPL_H
#define PLATYPUS_CHOICESTODO_IMPL_H

#include <list>
#include <cassert>
#include <algorithm>
#include <vector>
#include <memory>
#include <interfaces/from_idl/exception.h>
#include <interfaces/from_idl/choicestodo.h>
//#include <interfaces/choice.h>
#include <interfaces/utility.h>

namespace Platypus
{
	template<class CollectionT>
	class ChoiceToDoPolicy
	{
	protected:
		ChoiceToDoPolicy() {}
		virtual ~ChoiceToDoPolicy() {}
	public:
		typedef CollectionT CollectionType;
		typedef typename CollectionType::iterator iterator;
		typedef typename CollectionType::const_iterator const_iterator;
		typedef typename CollectionType::value_type value_type;

		virtual void add(const value_type& element) = 0;
		virtual void remove(const value_type& element) = 0;
		virtual bool hasChoice() = 0;
		virtual value_type makeChoice() = 0;
		virtual value_type & getChoice() = 0;
		virtual const CollectionType& get() const = 0;
		virtual CollectionType& get() = 0;
	};

	template<class CollectionT>
	class Chronological : public ChoiceToDoPolicy<CollectionT>
	{
		typedef typename ChoiceToDoPolicy<CollectionT>::CollectionType CollectionType;
		typedef typename CollectionType::value_type value_type;
		typedef typename CollectionType::iterator iterator;
		CollectionType collection_;
	public:
		void add(const value_type& element)
		{
			iterator res = std::find(collection_.begin(), collection_.end(), element);
			if(res != collection_.end())
				throw InvalidChoice();
			
			collection_.push_back(element);
		}
		void remove(const value_type& element)
		{
			iterator res = std::find(collection_.begin(), collection_.end(), element);
			if(res == collection_.end())
				throw InvalidChoice();
			collection_.erase(res);			
		}

		bool hasChoice(){

		  return (!collection_.empty());
		}

		value_type makeChoice()
		{
			value_type ret(collection_.back());
			collection_.pop_back();
			return ret;
		}
		value_type & getChoice()
		  {
		    return collection_.back();
		  }

		const CollectionType& get() const { return collection_; }

		CollectionType& get() { return collection_; }

	};

	template<class CollectionT>
	class DelegatablePolicy : public ChoiceToDoPolicy<CollectionT>
	{
		typedef typename ChoiceToDoPolicy<CollectionT>::CollectionType CollectionType;
		typedef typename CollectionType::value_type value_type;
		typedef typename CollectionType::iterator iterator;
		typedef typename CollectionType::reverse_iterator reverse_iterator;
		CollectionType collection_;
		
	public:
		unsigned choices_;

		DelegatablePolicy(): choices_(0){}
		~DelegatablePolicy(){}

		void add(const value_type& element)
		  {
		    if(!element.expired_)
		      choices_++;
		    
		    iterator res = std::find(collection_.begin(), collection_.end(), element);
		    if(res != collection_.end())
		      throw InvalidChoice();
		    
		    collection_.push_back(element);
		  }

		void remove(const value_type& element)
		{
			iterator res = std::find(collection_.begin(), collection_.end(), element);
			if(res == collection_.end())
				throw InvalidChoice();
			if(!res->expired_)
			  --choices_;
			collection_.erase(res);			
		}

		bool hasChoice(){

		  return choices_ > 0 ;
		}

		value_type makeChoice()
		{

		  bool found = false;
		  while(!found){
		    if(collection_.back().expired_){
		      this->remove(collection_.back());
		    }
		    else{
		      --choices_;
		      
		      //modify the choice on the local stack to reflect the fact that it is no longer a valid choice
		      //it must however be retained so delegation can occur properly
		      collection_.back().expired_ = true;
		      
		      if(collection_.back().positive_)
			collection_.back().positive_ = false;
		      else
			collection_.back().positive_ = true;
		      
		      found = true;
		    }
		  }

		  return collection_.back();
		}
		
		value_type & getChoice()
		  {
		    reverse_iterator rev(collection_.end());
		    for(;rev != reverse_iterator(collection_.begin()); rev++){
		      if(!rev->expired_){
			break;
		      }
		    }
		    return *rev;
		  }
		
		const CollectionType& get() const { return collection_; }

		CollectionType& get() { return collection_; }

	};
	
	
	template<class ChoiceT, class CollectionT>	
	  class ChoicesToDo : 
	  public ChoicesToDoInterface<ChoiceT, CollectionT>, 
	  private NoDefaultConstruction 
	  {
	  public:
	    //typedef typename ChoicesToDo<ChoiceT, CollectionT>::CollectionType CollectionType;
	    //typedef typename CollectionType::reverse_iterator reverse_iterator;
	    typedef CollectionT CollectionType;
	    typedef typename CollectionType::reverse_iterator reverse_iterator;
	    typedef ChoiceT ChoiceType;
	    typedef ChoiceToDoPolicy<CollectionType> PolicyType;
	    typedef typename PolicyType::iterator iterator;
	    //enum for the different choice type
	    enum BacktrackLevel { SHALLOW, MID, DEEP };

	  private:
	    PolicyType* policy_;
	    
	  public:
	    ChoicesToDo(PolicyType* policy)
	      :	policy_(policy)
	      {
		assert(policy);
	      }
	    
	    void add(const ChoiceType& choice)
	      {
		policy_->add(choice);			
	      }
	    
	    // returns the collection of choices todo allowing
	    // for iterating over it
	    // accessor method
	    const CollectionType& choicesToDo() const
	      {
		return policy_->get();
	      }
	    
	    // returns the collection of choices todo allowing
	    // for iterating over it
	    // accessor method
	    CollectionType& choicesToDo()
	      {
		return policy_->get();
	      }
	    
	    // removes a choice from the collection
	    void remove(const ChoiceType& choice)
	      {
		policy_->remove(choice);
	      }
	    
	    // returns true whenever there are choices stored, false otherwise
	    bool hasChoice() const
	      {
		return (policy_->hasChoice());
	      }
	    
	    // returns the next choice to be change using the 
	    // builtin policy
	    // A stack for instance would allow for chronological backtracking,
	    // however, one could apply other policies e.g. random choice, 
	    // only choose visible atoms (those that have a meaning in the
	    // original logic program, ...)
	    ChoiceType nextChoice(){
	      if(policy_->get().empty() || !policy_->hasChoice())
		throw NoChoiceLeft();
	      
	      return policy_->makeChoice();
	    }
	    
	    CollectionType nextDelegatableChoice(BacktrackLevel bl){
	      
	      //cout << "in next delegatable choice" << endl;
	      
	      if(policy_->get().empty() || !policy_->hasChoice()){
		cout << "OOOOPS" << endl;
		throw NoChoiceLeft();
	      }
	      
	      DelegatablePolicy<CollectionType> * dp = dynamic_cast<DelegatablePolicy<CollectionType> *>(policy_);
	      
	      int activeChoices = dp->choices_;

	      int choiceLevel = 1; //intialize it with the shallowest choice.
	      if(bl == MID){
		choiceLevel = activeChoices/2;
	      }
	      else if(bl == DEEP){
		choiceLevel = activeChoices;
	      }
		
	      //cout << "past exception throw in next delegatable choice" << endl;
	      
	      CollectionType ct;
	      
	      bool found = false;
	      bool choiceFound=false;
	      int activeChoicesFound=0;
	      iterator it = policy_->get().begin();
	      for(; !found && it != policy_->get().end(); ++it){
		choiceFound=false;
		if(!it->expired_){
		  choiceFound=true;
		  //activeChoicesFound++;
		  if(++activeChoicesFound == choiceLevel){
		    found = true;
		    it->expired_ = true;
		    dp->choices_--;
		  }
		}
		ct.push_back(*it);
		if(!ct.back().expired_)
		  ct.back().expired_ = true;
	      }
	      	      
	      reverse_iterator rit(ct.end());
	      if(rit->positive_)
		rit->positive_ = false;
	      else
		rit->positive_ = true;
	      return ct;
	    }
	    
	    void clear(){
	      
	      policy_->get().clear();
	      choices_ = 0;
	      
	    }
	  };
	
}

#endif
