// $Id: parser.cpp,v 1.4 2004/08/01 11:37:47 jean Exp $

#include <sstream>
#include <iostream>
#include <algorithm>
#include <interfaces/parser.h>

using namespace std;


LParser::~LParser()
{
	delete data_;
}
LParser::LParser()
	:	input_(0)
	,	data_(0)
	,	callback_(0)
{}
LParser::LParser(const std::string& input, LParserCallback* callback)
	:	input_(0)
	,	data_(0)
	,	callback_(callback)
{
	setInput(input);
}
LParser::LParser(std::istream& input, LParserCallback* callback)
	:	input_(&input)
	,	data_(0)
	,	callback_(callback)
{}
void LParser::setInput(const std::string& input)
{
	std::istream* temp = 0;
	std::swap(temp, data_);
	delete temp;
	
	data_ = new stringstream(input);
	setInput(*data_);
}
void LParser::setInput(std::istream& input)
{
	input_ = &input;
}
void LParser::setCallback(LParserCallback* callback)
{
	callback_ = callback;
}
void LParser::parse()
{
	if(!callback_)
		throw ::ParseError("No callback object set!");
	if(!input_)
		throw ::ParseError("No input set!");

	AtomSet atoms;
	int lineCount = 0;
	string line;
	
	bool readingRules = true;
	bool readingAtoms = false;
	bool readingBplus = false;
	bool readingBminus = false;
	bool readingNoModels = false;
	bool readingDone = false;

	// call callback, tell it we are about to start parsing
	callback_->setup();

	while(getline(*input_, line))
	{
		++lineCount;
		if(!line.size())
			continue;
		if(!input_->good())
			throw ::ParseError("Input stream is bad!", lineCount);
		

		if(readingRules)
		{
			if(line[0] == '0')
			{
				readingAtoms = true;
				readingRules = false;
				continue;
			}
			callback_->addRuleLine(line);
			if(!readRule(line, atoms))
				throw ::ParseError("Rule malformed", lineCount);
		}
			
		if(readingAtoms)
		{
			if(line[0] == '0')
			{
				readingAtoms = false;
				readingBplus = true;
				continue;
			}
			if(!readAtomName(line, atoms))
				throw ::ParseError("Atom name malformed", lineCount);
		}
		
		if(readingBplus)
		{
			if(line == "B+")
				continue;
			
			if(line[0] == '0')
			{
				readingBplus = false;
				readingBminus = true;				
				continue;
			}
			callback_->addBodyPlusLine(line);
		}

		if(readingBminus)
		{
			if(line == "B-")
				continue;

			if(line[0] == '0')
			{
				readingBminus = false;
				readingNoModels = true;				
				continue;
			}
			callback_->addBodyMinusLine(line);
		}

		if(readingNoModels)
		{
			if(!readNoModels(line))
				throw ::ParseError("Could not read number of models!", lineCount);

			readingDone = true;
			readingNoModels = false;
			continue;
		}

		if(readingDone)
			throw ::ParseError("There should be no further lines after the number of models line!", lineCount);
	}
	if(!readingDone)
		throw ::ParseError("Missing number models line!", lineCount);
	callback_->done();
}

bool LParser::readNoModels(std::string& line)
{
	stringstream s(line);
	int models = -1;
	s >> models;
	if(models == -1 || (!s.good() && !s.eof()))
		return false;
	callback_->addNoModels(models);
	return true;
}

bool LParser::addAtom(int atom, AtomSet& atoms)
{
	// must be a valid atom hence have a positive atom id
	if(atom <= 0)
		return false;

	if(atoms.insert(atom).second)
		callback_->addAtom(atom);

	return true;
}

bool LParser::readBodyAtoms(stringstream& s, AtomSet& atoms, int noNegative, int noPositive)
{
	for(int i = 0; i < noNegative; ++i)
	{
		int negative = -1;
		s >> negative;
		if(!s.good())
			return false;
		if(!addAtom(negative, atoms))
			return false;
	}

	for(int j = 0; j < noPositive; ++j)
	{
		int positive = -1;
		s >> positive;
		if(!s.good() && !s.eof())
			return false;
		if(!addAtom(positive, atoms))
			return false;
	}
	return true;
}

bool LParser::readBasicRule(std::string& line, AtomSet& atoms)
{
	stringstream s(line);
	
	int ruleid = -1, head = -1, noLiterals = -1, noNegative = -1;
	s >> ruleid >> head >> noLiterals >> noNegative;
	if(!s.good())
		return false;

	if(!addAtom(head, atoms))
		return false;

	if(noLiterals < noNegative)
		return false;

	return readBodyAtoms(s, atoms, noNegative, noLiterals - noNegative);	
}

bool LParser::readConstraintRule(std::string& line, AtomSet& atoms)
{
	stringstream s(line);
	
	int ruleid = -1, head = -1, noLiterals = -1, noNegative = -1, bound = -1;
	s >> ruleid >> head >> noLiterals >> noNegative >> bound;
	if(!s.good())
		return false;
	
	if(!addAtom(head, atoms))
		return false;

	if(noLiterals < noNegative)
		return false;

	if(noLiterals < noNegative)
		return false;
	
	if(bound > noLiterals || bound <= 0)
		return false;
	
	return readBodyAtoms(s, atoms, noNegative, noLiterals - noNegative);
}

bool LParser::readChoiceRule(std::string& line, AtomSet& atoms)
{
	stringstream s(line);
	
	int ruleid = -1, noHead = -1, noLiterals = -1, noNegative = -1;
	s >> ruleid >> noHead;
	if(!s.good())
		return false;
	
	if(noHead <= 0)
		return false;

	// First accumulate all heads and bodies into lists
	for(int k = 0; k < noHead; ++k)
	{
		int head = -1;
		s >> head;
		
		if(!s.good())
			return false;		
		
		if(!addAtom(head, atoms))
			return false;
	}

	s >> noLiterals >> noNegative;
	if(!s.good())
		return false;

	if(noLiterals < noNegative)
		return false;
	
	return readBodyAtoms(s, atoms, noNegative, noLiterals - noNegative);
}

bool LParser::readWeightRule(std::string& line, AtomSet& atoms)
{
	stringstream s(line);
	
	int ruleid = -1, head = -1, bound = -1, noLiterals = -1, noNegative = -1;
	s >> ruleid >> head >> bound >> noLiterals >> noNegative;
	if(!s.good())
		return false;

	if(!addAtom(head, atoms))
		return false;

	if(noLiterals < noNegative)
		return false;

	return readBodyAtoms(s, atoms, noNegative, noLiterals - noNegative);

	// a real parser would read the weights that follow the body atoms as well 
}

bool LParser::readMinimizeRule(std::string& line, AtomSet& atoms)
{
	stringstream s(line);
	
	int ruleid = -1, zero = -1, noLiterals = -1, noNegative = -1;	
	s >> ruleid >> zero >> noLiterals >> noNegative;
	if(!s.good())
		return false;

	if(zero != 0)
		return false;

	if(noLiterals < noNegative)
		return false;

	return readBodyAtoms(s, atoms, noNegative, noLiterals - noNegative);

	// a real parser would read the weights that follow the body atoms as well 
}

bool LParser::readRule(std::string& line, AtomSet& atoms)
{
	switch(line[0])
	{
	case '1' :
		return readBasicRule(line, atoms);
	case '2' :
		return readConstraintRule(line, atoms);
	case '3' :
		return readChoiceRule(line, atoms);		
	case '5' :
		return readWeightRule(line, atoms);
	case '6' :
		return readMinimizeRule(line, atoms);
	default :
		return false;
	}
}


bool LParser::readAtomName(std::string& line, AtomSet& atoms)
{
	if(line.size() < 3) // no atom name
		return false;

	stringstream s(line);
	int lparseid = -1;
	
	s >> lparseid >> line;
	if(!s.good() && !s.eof())
		return false;
	
	AtomSet::iterator res = atoms.find(lparseid);
	if(res == atoms.end())
		return false;

	callback_->setName(lparseid, line);
	atoms.erase(res);
    	
	return true;
}



