#ifndef __PROCESS_CONTROL__H__
#define __PROCESS_CONTROL__H__

/*===============================================================
 * ProcessControl.h
 * Richard Tichy
 *
 * This interface contains methods to control process forking
 * and zombie control on Linux machines.
 *
 * Modifications:
 *     May 25th - doxygen style commenting complete
 *================================================================*/

#include "semaphoreInterface.h"
#include "SharedMemory.h"
#include <cassert>
#include <vector>
#include <string>
#include <iterator>
#include <iostream>
#include <sys/types.h> 
#include <unistd.h>

using std::string;
using std::vector;
using std::iterator;

namespace Platypus{
  
  const int CHILD=0;
  const int FORK_ERROR=-1;

  const int MUTEX_EXPANDFN=0;
  const int MUTEX_SHAREDMEM=1;
  
  /**
   * Method controls the creation of new processes using
   * the Unix/Linux call to fork(). Returns FORK_ERROR
   * if the fork failed, CHILD on fork success and the returning process is the created
   * child or the new process pid if the process is the parent. Accesses
   * shared memory and thus needs the handle (pointer) to the 
   * shared memory segment and the integer id of the semaphore.
   *
   *@param int semid - the id of the semap
   *@param SharedMemory * shm - the pointer to the shared memory segment
   *@return int - FORK_ERROR, CHILD or the pid of the new process
   */
  int forkProcess(int semid, SharedMemory * shm);
  
  /**
   * Allows created (child) processes to exit without being orphaned to the
   * init process. Accesses shared memory so the pointer to the shared memory
   * segment and the semaphore id need to be passes to the method.
   *@param int semid - the id of the semap
   *@param SharedMemory * shm - the pointer to the shared memory segment
   */
  void childExiting(int semid, SharedMemory * shm);
  
  /**
   * Allows parent processes to reap multiple children using waitpid
   * before the parent itself exits. It thus prevents the orphaning
   * of processes to init.
   *
   *@param int semid - the id of the semap
   *@param SharedMemory * shm - the pointer to the shared memory segment
   */
  void parentReturning(int semid, SharedMemory * shm, vector<pid_t> & cids);
   
}

#endif
