#include "sharedMemoryInterface.h"

using std::cout;
using std::endl;

namespace Platypus{
  const int SUCCESS = 0;
  const int FAILURE = 1;
}

/*=========================================================================== 
 * sharedImplementation.cc
 * Christine Bailey and Richard Tichy
 *
 * Modifications:
 *     May 25th - code level documentation done
 *==========================================================================*/

/*   
 * This function creates a key that can be used to create shared memory
 * segments or semaphores.
 *
 * @param char c - unique id
 * @param key_t - key value
 */
key_t Platypus::getSharedMemoryKey(char c){
  
  //this variable is used to hold the key being created
  key_t key = ftok("./sharedMemoryImplementation.cc", c);
  
  return key;
}

/*    
 * This function creates a shared memory segment from the given key and 
 * returns the id of the segment.
 * 
 * @param key_t key - key used to access the segment
 * @return int - id of the shared memory accessed
 */
int Platypus::createSegment(key_t key){
  
  //this is used to hold the shared memory id of the shared memory just created
  int shmid=shmget(key, _POSIX_PIPE_BUF, 0644|IPC_CREAT);
  
  return shmid;
}

/*  
 * This function accesses and already existing shared memory segment using 
 * the given key and returns the id of the segment.
 *
 * @param key_t key - the key to be used to access the segment
 * @return int - the id of the shared memory just accessed
 */
int Platypus::getSegment(key_t key){
  
  //this is used to hold the shared memory id of the shared memory just accessed
  int shmid=shmget(key, _POSIX_PIPE_BUF, 0644);
  
  return shmid;
}

/* 
 * This function attaches a pointer to the shared memory associated with
 * the shared memory id shmid for access; will return
 * (void*)(-1) if there was an error.
 *
 * @param void* data - the pointer we want to attach to the shared memory
 * @param int shmid - the id of the shared memory we want to attach to
 * @return void* - the data that's now attached to the shared memory. 
 */
void * Platypus::attachSegment(void* data, int shmid){
  
  if ((data=shmat(shmid, (void *)0, 0)) == (void *)(-1)){
    cout << "shmat error in attachSegment" << endl;
  }
  
  return data;
}

/* 
 * This function detaches a pointer from shared memory.
 *
 * @param void* data - the pointer we want to detach from shared memory
 * @return bool - returns SUCCESS if detach succeeds, FAILURE otherwise.
 */
bool Platypus::detachSegment(void* data){
  
  if (shmdt(data)==-1) {
    cout << "shmdt error in detachSegment" << endl;
    return FAILURE;
  }
  
  return SUCCESS;
  
}

/*
 * This function deallocates the shared memory associated with the
 * given shared memory id.
 *
 * @param int shmid - the id of the shared memory we want to remove.
 * @return bool - returns SUCCESS if detach succeeds, else FAILURE
 */
bool Platypus::removeSegment(int shmid){
  
  if(shmctl(shmid, IPC_RMID, NULL)==-1){
    cout << "shmctl error in removeSegment" << endl;
    return FAILURE;
  }
  
  return SUCCESS;
}
