// $Id: test_expander_impl.cpp,v 1.4 2004/08/02 18:34:46 jean Exp $

#include <iostream>
#include <string>
#include <algorithm>
#include <fstream>
#include <functional>
#include <vector>
#include <cppunit/TestCase.h>
#include <cppunit/TestCaller.h>
#include <cppunit/TestResult.h>
#include <cppunit/TestSuite.h>
#include <cppunit/extensions/HelperMacros.h>
#include <interfaces/partial_model_impl.h>
#include <interfaces/atom.h>
#include <interfaces/choice.h>
#include <interfaces/program_impl.h>
#include <interfaces/smodels_program_impl.h>
#include <interfaces/program_factory.h>
#include <interfaces/expander_impl.h>
#include <interfaces/chooser_impl.h>
#include <test/program_metadata.h>

using namespace std;
using namespace Platypus;

namespace
{
	struct FindByName : public std::unary_function<Platypus::Atom, bool>
	{
		typedef std::unary_function<Platypus::Atom, bool> Base;
		const std::string& name_;
		FindByName(const std::string& name)
			:	name_(name)
		{}
		inline bool operator()(const Platypus::Atom& atom) const
		{
			return atom.name() == name_;
		}
	};
}

class ExpanderTest : public CppUnit::TestFixture
{
	ProgramMetaData* metaData_;
	typedef Expander<LocalExpanderInterfaceType> ExpanderType;
	Platypus::Atom getAtom(PartialModel& pm, const std::string& name)
	{
		for(PartialModel::CollectionType::const_iterator it = pm.unknownAtoms().begin();
			it != pm.unknownAtoms().end(); ++it)
		{
			if(it->name() == name)
				return *it;
		}

		for(PartialModel::CollectionType::const_iterator it = pm.positiveAtoms().begin();
			it != pm.positiveAtoms().end(); ++it)
		{
			if(it->name() == name)
				return *it;
		}

		for(PartialModel::CollectionType::const_iterator it = pm.negativeAtoms().begin();
			it != pm.negativeAtoms().end(); ++it)
		{
			if(it->name() == name)
				return *it;
		}
		throw 1;
	}

	inline bool contains(const PartialModel::CollectionType& collection, const AtomType& atom)
	{
		return std::count(collection.begin(), collection.end(), atom) == 1;
	}
	bool isSubset(const PartialModel& smaller, const PartialModel& larger)
	{
		typedef PartialModel::CollectionType::const_iterator const_iterator;
		for(const_iterator it = smaller.positiveAtoms().begin();
			it != smaller.positiveAtoms().end(); ++it)
		{
			if(!contains(larger.positiveAtoms(), *it))
				return false;
		}
		for(const_iterator it = smaller.negativeAtoms().begin();
			it != smaller.negativeAtoms().end(); ++it)
		{
			if(!contains(larger.negativeAtoms(), *it))
				return false;
		}
		for(const_iterator it = larger.unknownAtoms().begin();
			it != larger.unknownAtoms().end(); ++it)
		{
			if(!contains(smaller.unknownAtoms(), *it))
				return false;
		}
		return true;
	}
public:
	void setUp()
	{
		ifstream in("metadata.txt");
		CPPUNIT_ASSERT(in);
		metaData_ = new ProgramMetaData(in);
	}
	void tearDown()
	{
		delete metaData_;
	}

	void testEmpty()
	{
		Program& p = ProgramFactory::instance().create();
		p.setup(EMPTY_PROGRAM);
		PartialModel pm(p);
		ExpanderType expander(pm, p);

		CPPUNIT_ASSERT_EQUAL(expander.done(), true);
		CPPUNIT_ASSERT(!(expander.state() & ExpanderType::HAS_UNKNOWN));
		CPPUNIT_ASSERT(!(expander.state() & ExpanderType::HAS_CONFLICT));
        
		PartialModel pm2(expander.partialModel());
		CPPUNIT_ASSERT_EQUAL((unsigned)pm2.unknownAtoms().size(), (unsigned)0);
		CPPUNIT_ASSERT_EQUAL((unsigned)pm2.positiveAtoms().size(), (unsigned)0);
		CPPUNIT_ASSERT_EQUAL((unsigned)pm2.negativeAtoms().size(), (unsigned)0);
	}
	void testUsualcases()
	{
		const MetaData& md = metaData_->get("usualcases.lparse");
		ifstream in(md.filename_.c_str());
		CPPUNIT_ASSERT(in);


		ProgramType& p = ProgramFactory::instance().create();
		p.setup(in);
		PartialModel pm(p);
		ExpanderType expander(pm, p);

		CPPUNIT_ASSERT_EQUAL(expander.done(), false);
		CPPUNIT_ASSERT((expander.state() & ExpanderType::HAS_UNKNOWN));
		CPPUNIT_ASSERT(!(expander.state() & ExpanderType::HAS_CONFLICT));
        
		PartialModel pm2(expander.partialModel());
		CPPUNIT_ASSERT_EQUAL((unsigned)pm2.unknownAtoms().size(), (unsigned)2);
		CPPUNIT_ASSERT_EQUAL((unsigned)pm2.positiveAtoms().size(), (unsigned)2);
		CPPUNIT_ASSERT_EQUAL((unsigned)pm2.negativeAtoms().size(), (unsigned)md.atoms_ - 4);

		// get the remaining atom i
		AtomType i(*pm2.unknownAtoms().begin());
		Choice c(i, true);

		// expand with i set to true
		expander.expand(c);
		pm2 = expander.partialModel();
		CPPUNIT_ASSERT_EQUAL(expander.done(), true);
		CPPUNIT_ASSERT(!(expander.state() & ExpanderType::HAS_UNKNOWN));
		CPPUNIT_ASSERT(!(expander.state() & ExpanderType::HAS_CONFLICT));
		CPPUNIT_ASSERT_EQUAL((unsigned)pm2.unknownAtoms().size(), (unsigned)0);
		CPPUNIT_ASSERT_EQUAL((unsigned)pm2.positiveAtoms().size(), (unsigned)3);
		CPPUNIT_ASSERT_EQUAL((unsigned)pm2.negativeAtoms().size(), (unsigned)md.atoms_ - 3);

		// try to expand again using i which now has a truth value
		try
		{
			expander.expand(c);
			CPPUNIT_ASSERT(false);
		}
		catch(InvalidChoice&)
		{}

		// Try with another atom ("a") that already has a truth value
		try
		{
			AtomType a(getAtom(pm2, "a"));
			
			Choice c(a, false);
			expander.expand(c);
			CPPUNIT_ASSERT(false);
		}
		catch(InvalidChoice&)
		{}
	
		expander.backtrackTo(c.atom_);
		CPPUNIT_ASSERT_EQUAL(expander.done(), true);
		CPPUNIT_ASSERT(!(expander.state() & ExpanderType::HAS_UNKNOWN));
		CPPUNIT_ASSERT(!(expander.state() & ExpanderType::HAS_CONFLICT));
        
		pm2 = expander.partialModel();
		CPPUNIT_ASSERT_EQUAL((unsigned)pm2.unknownAtoms().size(), (unsigned)0);
		CPPUNIT_ASSERT_EQUAL((unsigned)pm2.positiveAtoms().size(), (unsigned)3);
		CPPUNIT_ASSERT_EQUAL((unsigned)pm2.negativeAtoms().size(), (unsigned)md.atoms_ - 3);
	}
	void testDoneAfterInit()
	{
		const MetaData& md = metaData_->get("usualcases-i-both.lparse");
		ifstream in(md.filename_.c_str());
		CPPUNIT_ASSERT(in);

		ProgramType& p = ProgramFactory::instance().create();
		p.setup(in);
		PartialModel pm(p);
		ExpanderType expander(pm, p);

		CPPUNIT_ASSERT_EQUAL(expander.done(), true);
		CPPUNIT_ASSERT(!(expander.state() & ExpanderType::HAS_UNKNOWN));
		CPPUNIT_ASSERT((expander.state() & ExpanderType::HAS_CONFLICT));
        
	}
	void testFalseBacktracking()
	{
		const MetaData& md = metaData_->get("usualcases.lparse");
		ifstream in(md.filename_.c_str());
		CPPUNIT_ASSERT(in);


		ProgramType& p = ProgramFactory::instance().create();
		p.setup(in);
		PartialModel pm(p);
		ExpanderType expander(pm, p);
		pm = expander.partialModel();
		// now we have the wellfounded model in our hands
		AtomType i(getAtom(pm, "i"));

		try
		{
			expander.backtrackTo(i);
			CPPUNIT_ASSERT(false);
		}
		catch(const NoChoiceMade&)
		{}

		AtomType h(getAtom(pm, "h"));
		try
		{
			expander.backtrackTo(h);
			CPPUNIT_ASSERT(false);
		}
		catch(const NoChoiceMade&)
		{}

		AtomType a(getAtom(pm, "a"));
		try
		{
			expander.backtrackTo(a);
			CPPUNIT_ASSERT(false);
		}
		catch(const NoChoiceMade&)
		{}
	}
	void testBacktracking()
	{
		typedef list<Choice> ChoiceList;

		const MetaData& md = metaData_->get("color-3-4.lparse");
		ifstream in(md.filename_.c_str());
		CPPUNIT_ASSERT(in);


		ProgramType& p = ProgramFactory::instance().create();
		p.setup(in);
		PartialModel pmInitial(p);
		ExpanderType expander(pmInitial, p);
		pmInitial = expander.partialModel();

		ChoiceList choicesMade;
		while(!expander.done())
		{
			PartialModel pm(expander.partialModel());
			ChooseFirstUnknown chooser(pm, p);
			Choice c(chooser.makeChoice());
			choicesMade.push_back(c);
			expander.expand(c);
			pm = expander.partialModel();
			// choosen atom must be positive in new partial model
			CPPUNIT_ASSERT_EQUAL(contains(pm.positiveAtoms(), c.atom_), true);
			// new partial model must be a superset of the initial partial model
			CPPUNIT_ASSERT_EQUAL(isSubset(pmInitial, pm), true);
		}

		// at this point we have an answer set, let's assert
		// it shows
		CPPUNIT_ASSERT(!(expander.state() & ExpanderType::HAS_CONFLICT));
		CPPUNIT_ASSERT(!(expander.state() & ExpanderType::HAS_UNKNOWN));

		// for color-3-4 there were 3 choices made, pop first and last
		choicesMade.pop_front();
		choicesMade.pop_back();
		while(!choicesMade.empty())
		{
			Choice c(choicesMade.back());
			choicesMade.pop_back();
			expander.backtrackTo(c.atom_);
			PartialModel pm(expander.partialModel());
			// choosen atom must be negative in new partial model
			CPPUNIT_ASSERT_EQUAL(contains(pm.negativeAtoms(), c.atom_), true);
			// new partial model must be a superset of the initial partial model
			CPPUNIT_ASSERT_EQUAL(isSubset(pmInitial, pm), true);

		}
		
		CPPUNIT_ASSERT((expander.state() & ExpanderType::HAS_UNKNOWN));
		CPPUNIT_ASSERT(!(expander.state() & ExpanderType::HAS_CONFLICT));

	}
	CPPUNIT_TEST_SUITE( ExpanderTest );
		CPPUNIT_TEST( testEmpty );
		CPPUNIT_TEST( testUsualcases );
		CPPUNIT_TEST( testDoneAfterInit );
		CPPUNIT_TEST( testFalseBacktracking );
		CPPUNIT_TEST( testBacktracking );
	CPPUNIT_TEST_SUITE_END();
};


CPPUNIT_TEST_SUITE_REGISTRATION( ExpanderTest );


