// $Id: test_parser.cpp,v 1.3 2004/08/01 11:37:51 jean Exp $

#include <iostream>
#include <string>
#include <fstream>
#include <map>
#include <cppunit/TestCase.h>
#include <cppunit/TestCaller.h>
#include <cppunit/TestResult.h>
#include <cppunit/TestSuite.h>
#include <cppunit/extensions/HelperMacros.h>
#include <interfaces/parser.h>
#include <test/program_metadata.h>

using namespace std;

namespace 
{
	struct CountVerifyParser : public LParserCallback
	{
		
		int setupCount_;
		int doneCount_;
		int ruleLines_;
		int bPlusLines_;
		int bMinusLines_;
		int noModelsCount_;
		int atomCount_;
		int nameCount_;
		CountVerifyParser()
			:	setupCount_(0)
			,	doneCount_(0)
			,	ruleLines_(0)
			,	bPlusLines_(0)
			,	bMinusLines_(0)
			,	noModelsCount_(0)
			,	atomCount_(0)
			,	nameCount_(0)
		{}
		void setup() { ++setupCount_; }
		void addRuleLine(const std::string& line) { ++ruleLines_; }
		void addAtom(int atom) { ++atomCount_; }
		void setName(int atom, const std::string& name) { ++nameCount_; }
		void addBodyPlusLine(const std::string& line) { ++bPlusLines_; }
		void addBodyMinusLine(const std::string& line) { ++bMinusLines_; }
		void addNoModels(unsigned no) { ++noModelsCount_; }
		void done() { ++doneCount_; }
	};

	struct VerifyContent : public LParserCallback
	{
		stringstream errors_;
		typedef std::map<int,std::string> Atom2Name;
		Atom2Name atoms_;
		const MetaData& md_;
		VerifyContent(const MetaData& md)
			:	md_(md)
		{}
		void setup() { }
		void addRuleLine(const std::string& line) { }
		void addAtom(int atom) 
		{
			if(atom < 1 || atom > md_.atoms_)
			{
				errors_ << "[addAtom] atom " << atom << " out of bounds!\n";
			}
			if(!atoms_.insert(make_pair(atom, string())).second)
			{
				errors_ << "[addAtom] atom " << atom << " exists!\n";
			}
		}
		void setName(int atom, const std::string& name) 
		{
			Atom2Name::iterator res = atoms_.find(atom);
			if(res == atoms_.end())
			{
				errors_ << "[setName] atom " << atom << " does not exist!\n";
			}
			else
			{
				if(res->second.size())
				{
					errors_ << "[setName] atom " << atom << " already has name " << res->second << ", new name is " << name << endl;					
				}
				else
				{
					res->second = name;
				}
			}
  		}
		void addBodyPlusLine(const std::string& line) {  }
		void addBodyMinusLine(const std::string& line) {  }
		void addNoModels(unsigned no) { }
		void done() {  }
		
	};

};

class LParserTest : public CppUnit::TestFixture
{
	ProgramMetaData* metaData_;
	
	void testCounts(const char* filename)
	{
		CPPUNIT_ASSERT(filename);
		ifstream in(filename);
		CPPUNIT_ASSERT(in);
		CountVerifyParser verifier;
		LParser p(in, &verifier);
		p.parse();
		

		const MetaData& md(metaData_->get(filename));

		CPPUNIT_ASSERT_EQUAL(verifier.setupCount_, 1);
		CPPUNIT_ASSERT_EQUAL(verifier.doneCount_, 1);
		CPPUNIT_ASSERT_EQUAL(verifier.ruleLines_, md.ruleLines_);
		CPPUNIT_ASSERT_EQUAL(verifier.bPlusLines_, md.bPlusLines_);
		CPPUNIT_ASSERT_EQUAL(verifier.bMinusLines_, md.bMinusLines_);
		CPPUNIT_ASSERT_EQUAL(verifier.noModelsCount_, md.models_);
		CPPUNIT_ASSERT_EQUAL(verifier.atomCount_, md.atoms_);
		CPPUNIT_ASSERT_EQUAL(verifier.nameCount_, md.atomNames_);
	}
	void testContent(const char* filename)
	{
		CPPUNIT_ASSERT(filename);
		ifstream in(filename);
		CPPUNIT_ASSERT(in);
		VerifyContent verifier(metaData_->get(filename));

		LParser p(in, &verifier);
		p.parse();

		string empty;
		CPPUNIT_ASSERT_EQUAL(verifier.errors_.str(), empty);
	}
public:
	void setUp()
	{
		ifstream in("metadata.txt");
		CPPUNIT_ASSERT(in);
		metaData_ = new ProgramMetaData(in);
	}
	void tearDown()
	{
		delete metaData_;
	}
	void testCtor()
	{
		try
		{
			// need to set callback interface first!
			LParser p;
			p.parse();
			CPPUNIT_ASSERT(false);
		}
		catch(ParseError&)
		{}

		ifstream in("usualcases.lparse");
		CPPUNIT_ASSERT(in);
		try
		{
			// need to set callback interface first!
			LParser p(in);
			p.parse();
			CPPUNIT_ASSERT(false);
		}
		catch(ParseError&)
		{}

	}
	void testSetInput()
	{
		LParser p;
		VerifyContent verifier1(metaData_->get("empty.lparse"));
		p.setCallback(&verifier1);
		try
		{
			p.parse();
			CPPUNIT_ASSERT(false);
		}
		catch(ParseError&)
		{
		}
		

		const string EMPTY("0\n0\nB+\n0\nB-\n0\n1\n");
		p.setInput(EMPTY);
		p.parse();


		ifstream in("usualcases.lparse");
		CPPUNIT_ASSERT(in);
		p.setInput(in);
		VerifyContent verifier2(metaData_->get("usualcases.lparse"));
		p.setCallback(&verifier2);
		p.parse();
	}
	void testEmpty()
	{
		testCounts("empty.lparse");
	}
	void testUsualCases()
	{
		testCounts("usualcases.lparse");
		testContent("usualcases.lparse");
	}
	void testHamilton()
	{
		testCounts("ham_comp_3.lparse");
		testContent("ham_comp_3.lparse");
	}
	void testColor()
	{
		testCounts("color-3-4.lparse");
		testContent("color-3-4.lparse");
	}
	void testPigeon()
	{
		testCounts("pigeon-3-4.lparse");
		testContent("pigeon-3-4.lparse");
	}
	void testSchur()
	{
		testCounts("schur-3-3.lparse");
		testContent("schur-3-3.lparse");
	}
	void testErrorneous(const char* filename)
	{
		try
		{
			testCounts(filename);
			CPPUNIT_ASSERT(false);
		}
		catch(const ParseError&)
		{}
	}
	void testErrorneous()
	{	
		testErrorneous("no_name.lparse");
		testErrorneous("broken_rule.lparse");
		testErrorneous("no_models.lparse");
		testErrorneous("no_compute.lparse");
		testErrorneous("no_final_zero.lparse");
	}
	
	CPPUNIT_TEST_SUITE( LParserTest );
		CPPUNIT_TEST( testCtor );	
		CPPUNIT_TEST( testSetInput );	
		CPPUNIT_TEST( testEmpty );		
		CPPUNIT_TEST( testUsualCases );	
		CPPUNIT_TEST( testHamilton );	
		CPPUNIT_TEST( testColor );	
		CPPUNIT_TEST( testPigeon );	
		CPPUNIT_TEST( testSchur );
		CPPUNIT_TEST( testErrorneous );
	CPPUNIT_TEST_SUITE_END();
};


CPPUNIT_TEST_SUITE_REGISTRATION( LParserTest );


