#include <iostream>
#include <fstream>
#include <cassert>
#include <memory>
#include <vector>
#include <algorithm>
#include <iterator>
#include <ctime>
#include <interfaces2/sharedMemoryInterface.h>
#include <interfaces2/semaphoreInterface.h>
#include <interfaces2/ProcessControl.h>
#include <interfaces2/SharedMemory.h>
#include <interfaces2/command_line_parser.h>
#include <interfaces/program_impl.h>
#include <interfaces/smodels_program_impl.h>
#include <interfaces/expander_chooser_impl.h>
#include <interfaces/choicestodo_impl.h>
#include <interfaces/partial_model_impl.h>
#include <interfaces/chooser_impl.h>
#include <interfaces/program_factory.h>

using namespace std;
using namespace Platypus;

typedef ChoicesToDo<ChoiceType, std::vector<ChoiceType> > ChoiceToDoType;
typedef Expander<LocalExpanderInterfaceType> ExpanderType;
typedef std::auto_ptr<ExpanderType> SafeExpander;
typedef DelegatablePolicy<std::vector<ChoiceType> > DelegatablePolicyType;
typedef ChoiceToDoType::CollectionType DelegatableCollectionType;

namespace{

  const int DEFAULT_PROCESSES = 5;

  //place a time stamp into the string parameterx
  void timeStamp(string & theTime){
    
    time_t rawTime;
    struct tm * timeInfo;
    
    time(&rawTime);
    timeInfo = localtime(&rawTime);
    theTime = asctime(timeInfo);
  }
  
  const char* const PROGRAM_NAME = "demo";
  int usage(){
    cerr << "usage: " << PROGRAM_NAME << " <filename>" << endl;
    return 1;
  }
  
  void printPositiveAtoms(const PartialModel& pm){
    copy(pm.positiveAtoms().begin(), pm.positiveAtoms().end(), ostream_iterator<AtomType>(cout, " "));
    cout << endl;
  }
  
  void printAnswerSet(const PartialModel& pm, unsigned id, unsigned answer){
    cout << "Answer " << id << "." << answer << ": ";
    printPositiveAtoms(pm);		
  }
  
  const int MUTEX_INIT=1;
  
  //global shared memory and semaphore stuff
  int semid;
  Platypus::SharedMemory * shm;
  
    
  class Statistics{
    
  public:
    
    unsigned localForks_;
    unsigned localAnswers_;
    unsigned expanderInits_;
    unsigned conflicts_;
    unsigned backtracks_;
    
    Statistics() :
      localForks_(0),
      localAnswers_(0),
      expanderInits_(0),
      conflicts_(0),
      backtracks_(0)
    {}

    ~Statistics(){}
    
    void reset(){

      localForks_ = 0;
      localAnswers_ = 0;
      expanderInits_ = 0;
      conflicts_ = 0;
      backtracks_ = 0;

    }

  };

  Statistics localStats;

  std::ostream & operator<<(std::ostream & os, const Platypus::ChoiceType & choice){
    
    os << "name: " << choice.atom_.name() << " positive: " << choice.positive_ << " expired: " << choice.expired_ << endl;  
    return os;
    
  }
  
  ostream & operator<<(ostream & os, const DelegatableCollectionType & dct){
    
    DelegatableCollectionType::const_iterator it = dct.begin();
    for(;it != dct.end(); it++){
      
      os << (*it);
    }
    
    return os;
  }
  
  ostream & operator<<(ostream & os, const ChoiceToDoType & choices){
    
    ChoiceToDoType::CollectionType coll = choices.choicesToDo();
    
    ChoiceToDoType::CollectionType::const_iterator it = coll.begin();
    for(;it != coll.end(); it++){
      
      os << (*it);
    }
    
    return os;
  }
  
  //forward declaration of control
  //void control(const SmodelsEnhancedProgram & program, const PartialModel & pm, const bool suppressed);
  void control(const SmodelsEnhancedProgram & program, const DelegatableCollectionType & dct, bool suppressed);
        
  void localExpand(SafeExpander & expander, 
		   ChoiceToDoType & choicesToDo, 
		   const SmodelsEnhancedProgram & program,
		   bool & extraRun,
		   vector<pid_t> & cids,
		   bool suppressed){

    extraRun = false;

    if(!expander->done()){

      PartialModel pm(expander->partialModel());
      ChooseFirstUnknown chooser(pm, program);

      //choose the first unknown atom in the undefined collection of atoms
      ChoiceType c(chooser.makeChoice());
      expander->expand(c);
      choicesToDo.add(c);

      //DELEGATE
      int status = forkProcess(semid, shm);
      if(!(status == Platypus::FORK_ERROR)){
	
	DelegatableCollectionType dct = choicesToDo.nextDelegatableChoice();

	if(status == Platypus::CHILD){
	  if(dct.back().positive_ = true){
	    dct.back().positive_ = false;
	  }
	  else
	    dct.back().positive_ = true;

	  localStats.reset();
	  control(program, dct, suppressed);
	  
	}
	else{

	  cids.push_back(status);
	  
	  extraRun = true;
	  ++(localStats.localForks_);
	}
      }
    }
    else{
      if(!(expander->state() & ExpanderType::HAS_CONFLICT)){
	if(!suppressed)
	  printAnswerSet(expander->partialModel(), getpid(), ++(localStats.localAnswers_));
      }
      else{
	++(localStats.conflicts_);
      }
      if(choicesToDo.hasChoice()){
	
	ChoiceType c(choicesToDo.nextChoice());
	++(localStats.backtracks_);
	expander->backtrackTo(c.atom_);
	
	extraRun = true;
      }
    }
  }
  
  void control(const SmodelsEnhancedProgram & program, const DelegatableCollectionType & dct, bool suppressed){

    //cout << getpid() << " dct top of CONTROL: " << endl << dct << endl;
       
    vector<pid_t> cids;
    
    PartialModel pm(program);
    DelegatablePolicyType policy;
    ChoiceToDoType choicesToDo(&policy);;

    assert(choicesToDo.choicesToDo().empty());

    DelegatableCollectionType::const_iterator it = dct.begin();
    for(;it != dct.end(); ++it){
      if(it->positive_ == true)
	pm.setTrue(it->atom_);
      else
	pm.setFalse(it->atom_);
      
      choicesToDo.add(*it);
    }
      
    SafeExpander expander(new ExpanderType(pm, program));

    ++(localStats.expanderInits_);
    bool extraRun = true;

    while(choicesToDo.hasChoice() || extraRun){
      
      localExpand(expander, choicesToDo, program, extraRun, cids, suppressed);

    }

    /*******************************
     *      LOCK
     *******************************/
    waitSem(semid, MUTEX_SHAREDMEM);
    
    shm->forks+=localStats.localForks_;
    shm->numAnswerSets+=localStats.localAnswers_;
    shm->maxNumberOfForks++; //increment the number of forks to reflect that this is doing no useful work

    int firstPid = shm->first_pid; 

    signalSem(semid, MUTEX_SHAREDMEM);
    /*******************************************
     *        RELEASE
     *******************************************/
    
    parentReturning(semid, shm, cids);
    if(firstPid != getpid())
      childExiting(semid,shm);
    return;
    
  }//end of control
  
}//end of namespace

int main(int argc, char * argv[]){
  
  int numberOfProcesses = DEFAULT_PROCESSES;
  const int FAILURE = 1;
  
  const clock_t tv1 = clock();
  
  cout << "FIRST PID: " << getpid() << endl;
  
  try{
    
    //SETUP MUTUAL EXCLUSION AND SHARED MEMORY
    key_t semkey=getSemKey('a');
    if((semid=createSem(semkey, 2))==-1){
      exit(EXIT_FAILURE);
    }
    if((initSem(semid, MUTEX_SHAREDMEM, MUTEX_INIT))==FAILURE){
      exit(EXIT_FAILURE);
    }
    key_t shmkey=getSharedMemoryKey('m');
    int shmid=createSegment(shmkey);
    shm=(SharedMemory*)attachSegment(shm, shmid);

    DistributedCommandLineParser parser;

    //gather and process command line arguments
    if(!parser.processArgs(argc, argv)){
      parser.printUsage();
      return 0;
    }
    
    //set the number of forks
    if(parser.checkProcesses()){
      numberOfProcesses = parser.getProcesses();
    }

    SmodelsEnhancedProgram& program = ProgramFactory::instance().create();
    //read program from either stdin or a file
    if(!parser.checkFile()){
      try{
	program.setup(cin);
      }catch(std::runtime_error & e){
	cout << endl << "There was an error with your program." << endl;
	exit(0);
      }
    }
    else{
      string theFile;
      parser.getFile(theFile);
      ifstream file(theFile.c_str());
      try{
	program.setup(file);
      }catch(std::runtime_error & e){
	cout << endl << "There was an error reading your file." << endl;
	exit(0);
      }
    }
    
    initSharedMemory(shm, numberOfProcesses);
    
    /*******************************************
     *          LOCK
     *******************************************/
    waitSem(semid, MUTEX_SHAREDMEM);
    
    shm->ppids[shm->pidIndex] = getpid();
    shm->ppidWaiting[shm->pidIndex++] = 0;
    
    signalSem(semid, MUTEX_SHAREDMEM);
    /*******************************************
     *          RELEASE
     *******************************************/
   
    //print some run specific stats
    cout << "version: Platypus Bitset Forked Iterative Linux Vector" << endl;
    cout << "maximum processes: " << numberOfProcesses << endl;
    if(parser.checkSuppressed())
      cout << "output is suppressed." << endl;
    string theTime;
    timeStamp(theTime);
    cout << "time: " << theTime;
    if(parser.checkData()){
      string theData;
      parser.getData(theData);
      cout << "data set: " << theData << endl;
    } 
    
    //entry point for platypus
    DelegatableCollectionType dct;
    control(program, dct, parser.checkSuppressed());
    
    const clock_t tv2 = clock();
    double time = tv2/(CLOCKS_PER_SEC*1.0) - tv1/(CLOCKS_PER_SEC*1.0);
    
    /*******************************
     *      LOCK
     *******************************/
    waitSem(semid, MUTEX_SHAREDMEM);
    
    cout << "solutions: " << shm->numAnswerSets << endl;
    cout << "forks: " << shm->forks << endl;
    cout << "total time: " << time << endl;
    signalSem(semid, MUTEX_SHAREDMEM);
    /*******************************
     *      RELEASE
     *******************************/
    
    detachSegment(shm);
    removeSegment(shmid);
    removeSem(semid);
    
    return 0;
  }
  catch(PlatypusException & e){
    
    cout << e.what() << endl;
    exit(1);
  }
}







