#include "mpi.h"
#include <vector>
#include <iostream>
#include <stdexcept>
#include <interfaces/from_idl/exception.h>

const int TIME_LIMIT = 1;
const int BUFFER_SIZE = 10000;

namespace Platypus{
  
  
  // Indicates indicates an invalid choice
  // (e.g. the atom already has a truth value)
  class ReceiveFailure : public PlatypusException{
    
  public:                                                                 
    ReceiveFailure(const std::string& reason = "Message receive failed.") : PlatypusException(reason){}
  }; 
  
  
  //the template functions within the class at this point only built in types
  //as template arguments.
  class Communicator{
    
  private:
    
    unsigned sent_;
    
    inline bool timedOut(double check, double start){ 
      /*std::cout << "time: " << check - start << std::endl;*/ return ((check - start) > TIME_LIMIT); }
    
  public:
    
    Communicator() : sent_(0) {}
    ~Communicator(){};
    
    inline bool probe(const int  from, const int messageType){ 
      return MPI::COMM_WORLD.Iprobe(from, messageType); 
    }  
    inline bool probe(const int from, const int messageType, MPI::Status & status){
      return MPI::COMM_WORLD.Iprobe(from, messageType, status);
    } 
    
    //send the message type to the receiver with dummy info in the buffer
    inline void send(const int to, const int messageType){ 
      int dummy = -1; sent_++; MPI::COMM_WORLD.Send(&dummy, 1, MPI::INT, to, messageType); 
    }
    
    //method will send one item of T as a message. A general warning. The parameter type
    //is the mpi type and must conincide with the built in T type. If this is not the case
    // then behaviour is UNDEFINED.
    template <class T> 
      void send(T message, const MPI::Datatype type, const int to, const int messageType){ 
      sent_++; MPI::COMM_WORLD.Send(&message, 1, type, to, messageType); 
    }
    
    //template function will send the vector within the message buffer, the type parameter
    //specifies the MPI type of the class. Caution: at this point this does not test type
    //correctness. It assumes the client does not specify T to be char and then type to
    //be MPI::INT. Behaviour in this case is UNDEFINED.
    template <class T> 
      void send(std::vector<T> & message, const MPI::Datatype type, const int to, const int messageType){ 
      sent_++; MPI::COMM_WORLD.Send(&message[0], message.size(), type, to, messageType); 
    }
    
    //receive the message. Assumes the message buffer consisted of only one item. See above warning 
    //mpi type specifier does not match the T type. 
    template <class T> 
      T timedReceive(const MPI::Datatype type, 
		     const int from, 
		     const int messageType){
      
      double start = MPI::Wtime();
      T buffer;
      
      while(1){
	
	if(!MPI::COMM_WORLD.Iprobe(from, messageType)){
	  
	  double check = MPI::Wtime();
	  //give the message n number of seconds to be received
	  if(timedOut(check, start)){
	    throw ReceiveFailure();
	  }
	}
	else{
	  
	  MPI::COMM_WORLD.Recv(&buffer, 1, type, from, messageType);
	  return buffer;
	  
	}
      }
    }
    
    //receive the message. The message will be placed in the array of type T
    //passed as the parameter and the number of items sent will be placed in size.
    template <class T> 
      void timedReceive(T arr[], 
			const unsigned size, 
			const MPI::Datatype type, 
			const int from, 
			const int messageType){
      
      double start = MPI::WTime(); 
      
      while(1){
	
	if(!MPI::COMM_WORLD.Iprobe(from, messageType)){
	  
	  double check = MPI::WTime();
	  //give the message n number of seconds to be received
	  if(timedOut(check, start)){
	    throw MessageFailure();
	  }
	}
	else{
	  
	  MPI::COMM_WORLD.Recv(arr, size, type, from, messageType);
	  break;
	}
      }
      
      return;
      
    }
    
    //receive the message. Assumes the message buffer consisted of only one item. See above warning            
    //mpi type specifier does not match the T type.                                                            
    template <class T>                                                                                         
      T receive(const MPI::Datatype type, 
		const int from, 
		const int messageType){                         
      
      T buffer;                                                                                               
      
      MPI::COMM_WORLD.Recv(&buffer, 1, type, from, messageType);                                           
      return buffer;                                                                                       
      
    }
    
    //receive the message. The message will be placed in the array of type T                                    
    //passed as the parameter and the number of items sent will be placed in size.                              
    template <class T>                                                                                         
      void receive(T arr[], 
		   const unsigned size, 
		   const MPI::Datatype type, 
		   const int from, 
		   const int messageType){                                                                     
      
      MPI::COMM_WORLD.Recv(arr, size, type, from, messageType);                                            
      return;                                                                                                  
      
    }

    inline unsigned totalSent(){ return sent_; }

  };
}  

