// $Id: partial_model_impl.cpp,v 1.6 2004/12/02 21:46:35 rtichy Exp $

#include <algorithm>
#include <cassert>
#include <functional>
//#include <iostream>
#include <interfaces/partial_model_impl.h>
#include <interfaces/atom.h>
#include <interfaces/program_impl.h>

using namespace std;

namespace Platypus
{
	namespace
	{
		struct AtomSort : public binary_function<Atom, Atom, bool>
		{
			inline bool operator()(const Atom& lhs, const Atom& rhs) const
			{
				return lhs.id() < rhs.id();
			}
		};
	}

	PartialModel::PartialModel(const Program& program)
	{
		for(ProgramInterfaceType::CollectionType::const_iterator it = program.atoms().begin();
			it != program.atoms().end();
			++it)
		{
			unknown_.push_back(Atom(*it, program));
		}
		unknown_.sort(AtomSort());
	}

  void PartialModel::moveAtom(const Atom& atom, CollectionType& to)
  {
    //holds a pair of iterators
    typedef pair<CollectionType::iterator, CollectionType::iterator> ItPair;
    
    //find the position in the Collection into which you can insert the atom while preserving the ordering
    const CollectionType::iterator insert = lower_bound(to.begin(), to.end(), atom, AtomSort());

    //find the range of values into which atom can be inserted into unknown_
    //if first==second then the atom was not in unknown_
    ItPair res = equal_range(unknown_.begin(), unknown_.end(), atom, AtomSort());
    
    
    if(res.first != res.second) // atom is in unknown
      {
	to.splice(insert, unknown_, res.first);
      }
    else 
      {
	res = equal_range(positive_.begin(), positive_.end(), atom, AtomSort());
	if(res.first != res.second)
	  {
	    to.splice(insert, positive_, res.first);
	  }
	else
	  {
	    to.splice(insert, negative_, lower_bound(negative_.begin(), negative_.end(), atom, AtomSort()));
	  }
      }
  }

	void PartialModel::setTrue(const Atom& atom)
	{
		moveAtom(atom, positive_);
	}
	void PartialModel::setFalse(const Atom& atom)
	{
		moveAtom(atom, negative_);
	}
	void PartialModel::setUnknown(const Atom& atom)
	{
		moveAtom(atom, unknown_);
	}
	bool PartialModel::hasUnknown() const
	{
		return !unknown_.empty();
	}
			
	const PartialModel::CollectionType& PartialModel::positiveAtoms() const
	{
		return positive_;
	}
	const PartialModel::CollectionType& PartialModel::negativeAtoms() const
	{
		return negative_;
	}
	const PartialModel::CollectionType& PartialModel::unknownAtoms() const
	{
		return unknown_;
	}
	void PartialModel::reset()
	{
		unknown_.splice(unknown_.end(), positive_);
		unknown_.splice(unknown_.end(), negative_);
		unknown_.sort(AtomSort());
	}

  ostream & Platypus::operator<<(ostream & os, const PartialModel & model){
    
    os << "T: ";
    
    for(PartialModel::CollectionType::const_iterator it = model.positiveAtoms().begin();it != model.positiveAtoms().end(); it++){
      os << ((Atom)*it).name();
    }

    os << endl << "F: ";
    
    for(PartialModel::CollectionType::const_iterator it = model.negativeAtoms().begin();it != model.negativeAtoms().end(); it++){
      os << ((Atom)*it).name();
    }
    
    os << endl << "U: ";
    
    for(PartialModel::CollectionType::const_iterator it = model.unknownAtoms().begin();it != model.unknownAtoms().end(); it++){
      os << ((Atom)*it).name();
    }
    
    os << std::endl;
    return os;
  }
   
	bool operator==(const PartialModelType& lhs, const PartialModelType& rhs)
	{
		if(&lhs == &rhs)
			return true;
		return	lhs.positiveAtoms() == rhs.positiveAtoms() &&
				lhs.negativeAtoms() == rhs.negativeAtoms() &&
				lhs.unknownAtoms() == rhs.unknownAtoms();
	}	
};
