#ifndef __COMMAND_LINE_PARSER__
#define __COMMAND_LINE_PARSER__

/*================================================================================
 * Richard Tichy
 * CommandLineParser.h
 *
 * Modifications: 
 *               - doxygen commenting completed
 *================================================================================*/

#include <iostream>
#include <vector>
#include <string>
#include <cstdlib>

using std::cout;
using std::endl;
using std::string;
using std::vector;

namespace Platypus{
  
  /**
   * Takes care of parsing of the command-line and the various command line options. These are
   * -p n      --the number of processes allowed to run for the multi-process versions.
   * -d <data> --additional, user specified information printed at the top of the 
   *             program run.
   */ 
  class CommandLineParser{
    
  protected:
    
    vector<string> _args;
    //int _processes;
    string _data;
    string _file;
    bool _fileSet;
    bool _suppressed;
    void _addArg(const char * arg);
    
  public:
    
    ///Constructor
      CommandLineParser() :
	_data(""),
	_file(""),
	_fileSet(false),
	_suppressed(false)
	{}
      
      ///Default Destructor
	virtual ~CommandLineParser(){};
	
	/**
	 * Process the command line arguments entered by the user. Method 
	 * returns true if no syntax error on the command line was found,
	 * false oterwise.
	 *
	 *@param int argc - argc from main method
	 *@param char **argv - argv from main method
	 *@return bool
	 */
	virtual bool processArgs(int argc, char **argv);
	
	/**
	 * Check to see if a data field was entered. True if there is
	 * a data field, false otherwise.
	 *
	 *@return bool
	 */
	virtual bool checkData() const{ return(_data.size() != 0); }
	
	/**
	 * Check to see whether there was a data file specified. Method
	 * returns true is a data file was specified, false otherwise.
	 *
	 *@return bool
	 */
	virtual bool checkFile() const{ return _fileSet; }
	
	virtual bool checkSuppressed() const{ return _suppressed; }
	
	/**
	 * Return the data entered on the command line in the parameter.
	 *
	 *@param string & theData
	 */
	virtual void getData(string & theData) const{ theData = _data; }
	
	/** 
	 * Return the name of the file entered on the command line
	 * in the parameter.
	 *
	 *@param string & theFile
	 */
	virtual void getFile(string & theFile) const{ theFile = _file; }

	/**
	 * Print the syntax guide to stdout.
	 */
	virtual void printUsage() const;
	
	/**
	 * Print the data set entered on the command line by the user.
	 */
	virtual void printDataSet() const;
	
  };
  
  class DistributedCommandLineParser : public CommandLineParser{
    
    const static int _DEFAULT = 7; 
    
    int _processes;
    int _internal;
    string _level;
    bool _levelSet;

  public:
    
    ///Constructor
      DistributedCommandLineParser() :
	CommandLineParser(),
	_processes(-1),
	_internal(false),
	_level(""),
	_levelSet(false)
	{}
      
      /**
       * Process the command line arguments entered by the user. Method 
       * returns true if no syntax error on the command line was found,
       * false oterwise.
       *
       *@param int argc - argc from main method
       *@param char **argv - argv from main method
       *@return bool
       */
      bool processArgs(int argc, char **argv);
      
      int & getProcesses(){ return _processes; }

      /**
       * Check to see whether the number of processes was set to
       * a valid number. Return true if the parameter was set to
       * a valid number, return false otherwise.
       *
       *@return bool
       */
      bool checkProcesses() const{ return(_processes > -1); }

      bool checkLevel() const{ return _levelSet; }
      
      /**
       * Return the level string.
       *
       */
      virtual const string & getLevel() const{ return _level; };
      
      void setInternal(bool internal){ _internal = internal; }

      /**
       * Check to see which version is running.
       */
      bool checkInternal() const{ return _internal; }

      //print the syntax guideline for the program
      void printUsage() const;


      
  };
  
}
#endif
