#include "semaphoreInterface.h"

using std::cout;
using std::cerr;
using std::endl;

const int SUCCESS = 0;
const int FAILURE = 1;


/*================================================================================ 
 * semaphoreImplementation.cc
 * Christine Bailey and Richard Tichy
 *
 * This file holds functions specific to creating, using and destroying semaphores.
 *
 * Modifications:
 *     May 25th - code level documentation done
 *===============================================================================*/

/*   
 * This function creates a key that can be used to create shared memory
 * segments or semaphores.
 *
 * @param char c - unique id
 * @param key_t - key value
 */
key_t Platypus::getSemKey(char c){
  
  key_t key=ftok("./semaphoreImplementation.cc", c);
  
  return key;
  
}

/*
 * This function creates a semaphore set of size numSemaphores with
 * protection level 644 and returns the id of the semaphore created.
 *
 * @param key_t - a key to be used to create the semaphore set
 * @param unsigned int numSemaphores - the size of the semaphore set to be created
 * @return int - the semaphore id that was created on sucess, else if failure occurred, returns -1
 */
int Platypus::createSem(key_t key, unsigned int numSemaphores){
  
  //this will be the id of the semaphore set to be created
  int semid;
  
  if((semid=semget(key, numSemaphores, 0644|IPC_CREAT))==-1){
    cerr << "semget error in createSem" << endl;
    return -1;
  }
  
  return semid;
  
}

/*
 * Returns a handle to a semaphore set of size numSemaphores with
 * protection level 644 and returns the id of the semaphore created.
 *
 * @param key_t - a key to the semaphore set
 * @param unsigned int numSemaphores - the size of the semaphore set 
 * @return int - the semaphore id that was created on sucess, else if failure occurred, returns -1
 */
int Platypus::getSem(key_t key, unsigned int numSemaphores){
  
  //this will be the id of the semaphore set to be created
  int semid;
  
  if((semid=semget(key, numSemaphores, 0644))==-1){
    cerr << "semget error in getSem" << endl;
    return -1;
  }
  
  return semid;
  
}

/*
 * This function initializes the semaphore specified to the given value.
 *
 * @param int semaphore - id of the semaphore set being accessed
 * @param unsigned int semNumber - the number in the set of the semaphore to be initialized
 * @param int value - the number that the semaphore is to be set to
 * @return bool - returns SUCCESS if call succeeds, else FAILURE		
 */
bool Platypus::initSem(int semaphore, unsigned int semNumber, int value){
  
  /*this union semun will hold the value that the semaphore is to be set to
   *and will pass the value to the semaphore through this structure*/
  union semun arg;
  arg.val=value;
  
  if(semctl(semaphore, semNumber, SETVAL, arg)==-1){
    cerr << "semctl error in initSem" << endl;
    return FAILURE;
  }
  return SUCCESS;
}

/*
 * This function executes a wait command on the semaphore; that is, 
 * if the current semaphore's value is greater than 1, subtract 1 from the
 * semaphore's value.  Otherwise, block the running process until it reaches
 * 1, then subtract 1 and continue.
 *
 * @param int semaphore - id of the semaphore set being accessed
 * @param unsigned int semNumber - the number in the set of the semaphore to be waited on
 * @return void
 */
void Platypus::waitSem(int semaphore, unsigned int semNumber){

  
  unsigned int nsops=1;
  struct sembuf wait_buf;
  wait_buf.sem_num=semNumber;
  wait_buf.sem_op=-1;
  wait_buf.sem_flg=0;
  
  if (semop(semaphore, &wait_buf, nsops) == -1){
    cout << "Error in wait semaphore" << endl;
    if (errno == E2BIG) cout << "E2BIG" << endl;
    if (errno == EACCES) cout << "EACCES" << endl;
    if (errno == EAGAIN) cout << "EAGAIN" << endl;
    if (errno == EFAULT) cout << "EFAULT" << endl;
    if (errno == EFBIG) cout << "EFBIG" << endl;
    if (errno == EIDRM) cout << "EIDRM" << endl;
    if (errno == EINTR) cout << "EINTR" << endl;
    if (errno == EINVAL) cout << "EINVAL" << endl;
    if (errno == ENOMEM) cout << "ENOMEM" << endl;
    if (errno == ERANGE) cout << "ERANGE" << endl;
    exit (1);
  }
}

/*
 * Description: this function executes a signal command on the semaphore given; that is, it
 * simply adds 1 to the semaphore.
 *
 * @param int semaphore - id of the semaphore set being accessed
 * @param unsigned int semNumber - the number in the set of the semaphore to be signalled
 * @return void
 */
void Platypus::signalSem(int semaphore, unsigned int semNumber){
  
  unsigned int nsops=1;
  struct sembuf signal_buf;
  signal_buf.sem_num=semNumber;
  signal_buf.sem_op=1;
  signal_buf.sem_flg=0;

  if (semop(semaphore, &signal_buf, nsops) == -1){
    cerr << "semop error in signal semaphore" << endl;
    exit (1);
  }
  
}

/*
 * Description: this function deallocates the given semaphore.
 *
 * @param int semaphore - id of the semaphore set being accessed
 * @return bool - returns SUCCESS if the call succeeds, else FAILURE	
 */
bool Platypus::removeSem(int semaphore){
  
  // this is passed into the function used to remove the semaphore; note: it doesn't 
  // really matter what arg is since we're just removing the semaphore
  union semun arg;
  
  if(semctl(semaphore, 0, IPC_RMID, arg)==-1){
    cerr << "semctl error in removeSem" << endl;
    return FAILURE;
  }
  return SUCCESS;
}
