#ifndef __SEMAPHORE__INTERFACE__H__
#define __SEMAPHORE__INTERFACE__H__

/*====================================================================
 * semaphoreInterface.h
 * Programmer: C. Bailey and R. Tichy
 *
 * Contains semaphore control code.
 *
 * Modifications:
 *     May 25th - doxygen style commenting done
 *====================================================================*/

//#include "Types.h"
#include <iostream>
#include <errno.h>
#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <sys/types.h>
#include <sys/ipc.h>
#include <sys/sem.h>
#include <sys/wait.h>

union semun{
  //used for SETVAL when initializing
  int val;
  //used for IPC_STAT & IPC_SET
  struct semid_ds *bus;
  //used for GETALL & SETALL
  ushort *array;
};

namespace Platypus{
      
  /**   
   * This function creates a key that can be used to create shared memory
   * segments or semaphores.
   *
   * @param char c - unique id
   * @param key_t - key value
   */
  key_t getSemKey(char c);

  /**
   * This function creates a semaphore set of size numSemaphores with
   * protection level 644 and returns the id of the semaphore created.
   *
   * @param key_t - a key to be used to create the semaphore set
   * @param unsigned int numSemaphores - the size of the semaphore set to be created
   * @return int - the semaphore id that was created on sucess, else if failure occurred, returns -1
   */
  int createSem(key_t key, unsigned int numSemaphores);

  /**
   * Returns a handle to a semaphore set of size numSemaphores with
   * protection level 644 and returns the id of the semaphore created.
   *
   * @param key_t - a key to the semaphore set
   * @param unsigned int numSemaphores - the size of the semaphore set 
   * @return int - the semaphore id that was created on sucess, else if failure occurred, returns -1
   */
  int getSem(key_t key, unsigned int numSemaphores);
  
  /**
   * This function initializes the semaphore specified to the given value.
   *
   * @param int semaphore - id of the semaphore set being accessed
   * @param unsigned int semNumber - the number in the set of the semaphore to be initialized
   * @param int value - the number that the semaphore is to be set to
   * @return bool - returns SUCCESS if call succeeds, else FAILURE		
   */
  bool initSem(int semaphore, unsigned int semNumber, int value);

  /**
   * This function executes a wait command on the semaphore; that is, 
   * if the current semaphore's value is greater than 1, subtract 1 from the
   * semaphore's value.  Otherwise, block the running process until it reaches
   * 1, then subtract 1 and continue.
   *
   * @param int semaphore - id of the semaphore set being accessed
   * @param unsigned int semNumber - the number in the set of the semaphore to be waited on
   * @return void
   */
  void waitSem(int semaphore, unsigned int semNumber);

  /**
   * Description: this function executes a signal command on the semaphore given; that is, it
   * simply adds 1 to the semaphore.
   *
   * @param int semaphore - id of the semaphore set being accessed
   * @param unsigned int semNumber - the number in the set of the semaphore to be signalled
   * @return void
   */
  void signalSem(int semaphore, unsigned int semNumber);

  /*
   * Description: this function deallocates the given semaphore.
   *
   * @param int semaphore - id of the semaphore set being accessed
   * @return bool - returns SUCCESS if the call succeeds, else FAILURE	
   */
  bool removeSem(int semaphore);
  
}

#endif
