#include <iostream>
#include <fstream>
#include <cassert>
#include <memory>
#include <vector>
#include <algorithm>
#include <iterator>
#include <ctime>
#include <sys/time.h>
//#include <interfaces/program_impl.h>
#include <interfaces/smodels_program_impl.h>
#include <interfaces/expander_chooser_impl.h>
#include <interfaces/choicestodo_impl.h>
#include <interfaces/chooser_impl.h>
//#include <interfaces/partial_model_impl.h>
#include <interfaces/program_factory.h>
#include <interfaces/ticks.h>
#include <interfaces2/sharedMemoryInterface.h>
#include <interfaces2/semaphoreInterface.h>
#include <interfaces2/ProcessControl.h>
#include <interfaces2/SharedMemory.h>
#include <interfaces2/command_line_parser.h>

using namespace std;
using namespace Platypus;

//typedefs for passing partial models from agent to agent
//typedef ChoicesToDo<DelegatableChoiceType, std::vector<DelegatableChoiceType> > ChoiceToDoType;
//typedef Expander<DelegatableExpanderInterfaceType> ExpanderType;
//typedef std::auto_ptr<ExpanderType> SafeExpander;
//typedef Chronological<std::vector<DelegatableChoiceType> > DelegatablePolicyType;

//typdefs for passing around choices internal to choicesToDo
typedef Expander<LocalExpanderInterfaceType> InternalExpanderType;
typedef std::auto_ptr<InternalExpanderType> InternalSafeExpander;
typedef ChoicesToDo<ChoiceType, std::vector<ChoiceType> > InternalChoicesToDoType;
typedef DelegatablePolicy<std::vector<ChoiceType> > InternalDelegatablePolicyType;
typedef InternalChoicesToDoType::CollectionType DelegatableCollectionType;

namespace
{
  class Time
  {
  public:
    Time()
      :	start_(0)
      ,	end_(0)
    {
      assert(frequency_ && "This platform does not support high performance clocks!");
    }
    inline uint64 frequency() const
    {
      return frequency_;
    }
    inline void start()
    {
      start_ = end_ = stamp();
    }
    inline void stop()
    {
      end_ = stamp();
    }
    inline uint64 difference() const
    {
      return end_ - start_;
    }
    uint64 stamp() const;
  private:
    static uint64 calculateFrequency();
  private:
    uint64 start_, end_;
    static const uint64 frequency_;
  };
  uint64 Time::stamp() const
  {
    timeval tv;
    gettimeofday(&tv, 0);
    
    uint64 t = tv.tv_usec;
    t += tv.tv_sec*1000000;
    return t;
  }
  
  uint64 Time::calculateFrequency()
  {
    return 1000000;
  }

  const uint64 Time::frequency_ = Time::calculateFrequency();

}

namespace{

  //global variable definitions
  const int DEFAULT_PROCESSES = 5;
  const int MUTEX_INIT=1;

  //GLOBAL shared memory and semaphore stuff
  int semid;
  Platypus::SharedMemory * shm;

  //place a time stamp into the string parameterx
  void timeStamp(string & theTime){
    
    time_t rawTime;
    struct tm * timeInfo;
    
    time(&rawTime);
    timeInfo = localtime(&rawTime);
    theTime = asctime(timeInfo);
  }
  

  
  void printPositiveAtoms(const PartialModel& pm){
    copy(pm.positiveAtoms().begin(), pm.positiveAtoms().end(), ostream_iterator<AtomType>(cout, " "));
    cout << endl;
  }
  
  void printAnswerSet(const PartialModel& pm, unsigned id, unsigned answer){
    cout << "Answer " << id << "." << answer << ": ";
    printPositiveAtoms(pm);		
  }
    
  //helper class gathers statistics relevant to each individual agent
  class Statistics{
    
  public:
    
    unsigned localForks_;
    unsigned localAnswers_;
    unsigned localExpands_;
    unsigned expanderInits_;
    unsigned conflicts_;
    unsigned backtracks_;
    
    Statistics() :
      localForks_(0),
      localAnswers_(0),
      localExpands_(0),
      expanderInits_(0),
      conflicts_(0),
      backtracks_(0)
    {}

    ~Statistics(){}
    
    void reset(){

      localForks_ = 0;
      localAnswers_ = 0;
      localExpands_ = 0;
      expanderInits_ = 0;
      conflicts_ = 0;
      backtracks_ = 0;

    }

  };

  //global statistics gathering for each agent
  Statistics localStats;

  //debug ostream overloading 
  std::ostream & operator<<(std::ostream & os, const Platypus::ChoiceType & choice){
    
    os << "name: " << choice.atom_.name() << " positive: " << choice.positive_ << " expired: " << choice.expired_ << endl;  
    return os;
    
  }
  ostream & operator<<(ostream & os, const DelegatableCollectionType & dct){
    
    DelegatableCollectionType::const_iterator it = dct.begin();
    for(;it != dct.end(); it++){
      
      os << (*it);
    }
    
    return os;
  } 
  
  //forward declaration of of overloaded control functions
  void control(const SmodelsEnhancedProgram & program, 
	       const DelegatableCollectionType & dct,
	       InternalChoicesToDoType::BacktrackLevel level,
	       bool suppressed);
  
  void internalLocalExpand(InternalSafeExpander & expander, 
			   InternalChoicesToDoType & choicesToDo,
			   InternalChoicesToDoType::BacktrackLevel level,
			   const SmodelsEnhancedProgram & program,
			   bool & extraRun,
			   vector<pid_t> & cids,
			   bool suppressed){
    
    extraRun = false;

    if(!expander->done()){

      //PartialModel pm(expander->partialModel());
      //ChooseFirstUnknown chooser(pm, program);

      //choose the first unknown atom in the undefined collection of atoms
      ChoiceType c(expander->makeChoice());//faster than expander make choice

      //cout << "PM before expand: " << endl << pm << endl; 
      //cout << "ChoiceType before expand: " << endl << c << endl;
      expander->expand(c);
      ++(localStats.localExpands_);

      choicesToDo.add(c);

      //DELEGATE
      int status = forkProcess(semid, shm);
      if(!(status == Platypus::FORK_ERROR)){
	
	DelegatableCollectionType dct = choicesToDo.nextDelegatableChoice(level);

	if(status == Platypus::CHILD){
	  
	  localStats.reset();
	  control(program, dct, level, suppressed);
	  
	}
	else{

	  cids.push_back(status);
	  
	  extraRun = true;
	  ++(localStats.localForks_);
	}
      }
    }
    else{
      if(!(expander->state() & InternalExpanderType::HAS_CONFLICT)){
	if(!suppressed)
	  printAnswerSet(expander->partialModel(), getpid(), (localStats.localAnswers_ + 1));
	++(localStats.localAnswers_);
      }
      else{
	++(localStats.conflicts_);
      }
      if(choicesToDo.hasChoice()){
	
	ChoiceType c(choicesToDo.nextChoice());
	++(localStats.backtracks_);
	expander->backtrackTo(c.atom_);
	
	extraRun = true;
      }
    }
  }
  
    void control(const SmodelsEnhancedProgram & program, 
		 const DelegatableCollectionType & dct,
		 InternalChoicesToDoType::BacktrackLevel level,
		 bool suppressed){

    vector<pid_t> cids;
    
    PartialModel pm(program);
    InternalDelegatablePolicyType policy;
    InternalChoicesToDoType choicesToDo(&policy);;
    
    assert(choicesToDo.choicesToDo().empty());

    DelegatableCollectionType::const_iterator it = dct.begin();
    for(;it != dct.end(); ++it){
      if(it->positive_ == true)
	pm.setTrue(it->atom_);
      else
	pm.setFalse(it->atom_);
      
      choicesToDo.add(*it);
    }
      
    InternalSafeExpander expander(new InternalExpanderType(pm, program));

    ++(localStats.expanderInits_);
    bool extraRun = true;

    while(choicesToDo.hasChoice() || extraRun){
      
      internalLocalExpand(expander, choicesToDo, level, program, extraRun, cids, suppressed);

    }

    /*******************************
     *      LOCK
     *******************************/
    waitSem(semid, MUTEX_SHAREDMEM);
    
    shm->forks += localStats.localForks_;
    shm->numAnswerSets += localStats.localAnswers_;
    shm->maxNumberOfForks++; //increment the number of forks to reflect that this is doing no useful work
    shm->expands += localStats.localExpands_;

    int firstPid = shm->first_pid; 

    signalSem(semid, MUTEX_SHAREDMEM);
    /*******************************************
     *        RELEASE
     *******************************************/
    
    parentReturning(semid, shm, cids);
    if(firstPid != getpid())
      childExiting(semid,shm);
    return;
    
  }//end of control
  
}//end of namespace


int main(int argc, char * argv[]){
  
  int numberOfProcesses = DEFAULT_PROCESSES;
  const int FAILURE = 1;
  
  //jean's new timing
  //uint64 freq = frequency();
  //uint64 t1 = ticks();
  
  //newest Jean timing
  Time ct;
  ct.start();
  
  //low_res timing
  //time_t time_start = time(0);
  
  //old timing
  //const clock_t tv1 = clock();
  
  //cout << "FIRST PID: " << getpid() << endl;
  
  try{
    
    //SETUP MUTUAL EXCLUSION AND SHARED MEMORY
    key_t semkey=getSemKey('a');
    if((semid=createSem(semkey, 2))==-1){
      exit(EXIT_FAILURE);
    }
    if((initSem(semid, MUTEX_SHAREDMEM, MUTEX_INIT))==FAILURE){
      exit(EXIT_FAILURE);
    }
    key_t shmkey=getSharedMemoryKey('m');
    int shmid=createSegment(shmkey);
    shm=(SharedMemory*)attachSegment(shm, shmid);

    DistributedCommandLineParser parser;

    //gather and process command line arguments
    if(!parser.processArgs(argc, argv)){
      parser.printUsage();
      return 0;
    }

    InternalChoicesToDoType::BacktrackLevel BTlevel;
    string runLevel = "";
    if(parser.checkLevel()){
      if(parser.getLevel() == "shallow"){
	BTlevel = InternalChoicesToDoType::SHALLOW;
	runLevel = "SHALLOW";
      }
      else if(parser.getLevel() == "mid"){
	BTlevel = InternalChoicesToDoType::MID;
	runLevel = "MID";
      }
      else if(parser.getLevel() == "deep"){
	BTlevel = InternalChoicesToDoType::DEEP;
	runLevel = "DEEP";
      }
      else{
	parser.printUsage();
	return 0;
      }
    }
    else{
      BTlevel = InternalChoicesToDoType::DEEP;
      runLevel = "DEEP";
    }
      
    //set the number of forks
    if(parser.checkProcesses()){
      numberOfProcesses = parser.getProcesses();
    }

    SmodelsEnhancedProgram& program = ProgramFactory::instance().create();
    //read program from either stdin or a file
    if(!parser.checkFile()){
      try{
	program.setup(cin);
      }catch(std::runtime_error & e){
	cout << endl << "There was an error with your program." << endl;
	exit(0);
      }
    }
    else{
      string theFile;
      parser.getFile(theFile);
      ifstream file(theFile.c_str());
      try{
	program.setup(file);
      }catch(std::runtime_error & e){
	cout << endl << "There was an error reading your file." << endl;
	exit(0);
      }
    }
    
    initSharedMemory(shm, numberOfProcesses);
    
    /*******************************************
     *          LOCK
     *******************************************/
    waitSem(semid, MUTEX_SHAREDMEM);
    
    shm->ppids[shm->pidIndex] = getpid();
    shm->ppidWaiting[shm->pidIndex++] = 0;
    
    signalSem(semid, MUTEX_SHAREDMEM);
    /*******************************************
     *          RELEASE
     *******************************************/
   
    //print some run specific stats
    cout << "Maximum processes: " << numberOfProcesses << endl;
    if(parser.checkSuppressed())
      cout << "Answer set output is suppressed." << endl;
    cout << "Choice level: " << runLevel << endl;
    string theTime;
    timeStamp(theTime);
    cout << "Start time: " << theTime;
    if(parser.checkData()){
      string theData;
      parser.getData(theData);
      cout << "Data set: " << theData << endl;
    } 
    
    //entry point
    //if(!parser.checkInternal()){
    //DelegatableCollectionType collection;
    //cout << "Running DEEP version." << endl;
    //control(program, collection, InternalChoicesToDoType::DEEP, parser.checkSuppressed());
    //}
    //else{
    DelegatableCollectionType collection;
    //cout << "Running SHALLOW version." << endl;
    //control(program, collection, InternalChoicesToDoType::SHALLOW, parser.checkSuppressed());
    //}
    control(program, collection, BTlevel, parser.checkSuppressed());
    
    //jean's timing
    //uint64 t2 = ticks();
    //uint64 t3 = t2 -t1;
    //double t_time = (t3*1.0)/(freq*1.0);

    //newest Jean timing
    ct.stop();

    //low_res timing
    //time_t time_stop = time(0);
    
    //old timing
    //const clock_t tv2 = clock();
    //double time = tv2/(CLOCKS_PER_SEC*1.0) - tv1/(CLOCKS_PER_SEC*1.0);
    
    /*******************************
     *      LOCK
     *******************************/
    waitSem(semid, MUTEX_SHAREDMEM);
    
    cout << "solutions: " << shm->numAnswerSets << endl;
    cout << "forks: " << shm->forks << endl;
    cout << "expands: " << shm->expands << endl;
    //cout << "total time: " << time << endl; //old timing
    //cout << "total time: " << t_time << endl; //jean's timing
    //cout << "total time (ticks): " << t_time << endl; //jean's timing
    //cout << "total time (clock): " << time << endl; //jean's timing
    cout << "total time: " << (double)(ct.difference()*1.0)/(ct.frequency()*1.0) << ", frequency: " <<  ct.frequency() << endl;
    //cout << "total time (time): " << (time_stop-time_start) << endl;
    signalSem(semid, MUTEX_SHAREDMEM);
    /*******************************
     *      RELEASE
     *******************************/
    
    detachSegment(shm);
    removeSegment(shmid);
    removeSem(semid);
    
    return 0;
  }
  catch(PlatypusException & e){
    
    cout << e.what() << endl;
    exit(1);
  }
}







