// $Id: expander_impl.h,v 1.8 2004/08/05 07:51:13 jean Exp $

#ifndef PLATYPUS_EXPANDER_IMPL_H
#define PLATYPUS_EXPANDER_IMPL_H

#include <memory>
#include <string>
#include <sstream>
#include <interfaces/from_idl/exception.h>
#include <interfaces/from_idl/expander.h>
#include <interfaces/choice.h>
#include <interfaces/partial_model_impl.h>
#include <interfaces/smodels_expander_impl.h>

namespace Platypus
{
	class Program;
	class SmodelsEnhancedProgram;
	class InjectConstraints
	{
		std::stringstream stream_;
		static std::string formatLparse(const PartialModelType& partialModel, const Program& program);
	public:
		InjectConstraints(const PartialModelType& partialModel, const ProgramType& program);
		std::stringstream& stream();
	};

	typedef ExpanderInterface<PartialModelType, ChoiceType> LocalExpanderInterfaceType;
	typedef ExpanderInterface<PartialModelType, DelegatableChoiceType> DelegatableExpanderInterfaceType;

	template<class Interface = LocalExpanderInterfaceType>
	class Expander : public Interface
	{
	public:	
		typedef typename Interface::ExpanderState ExpanderState;
		typedef typename Interface::PartialModelType PartialModelType;
		typedef typename Interface::ChoiceType ChoiceType;
	private:
		PartialModelType partialModel_;
		
		std::auto_ptr<SmodelsExpander> pimpl_;
		void updatePartialModel()
		{
			pimpl_->update(partialModel_);
		}
		void calcExpansionAndUpdatePM()
		{
			pimpl_->expand();
			updatePartialModel();
		}
	public:
		Expander(const PartialModelType& partialModel, const Program& program)
			:	partialModel_(partialModel)
		{
			InjectConstraints injector(partialModel, program);
			std::auto_ptr<SmodelsExpander> gcc_295_fix(new SmodelsExpander(injector.stream()));
			pimpl_ = gcc_295_fix;
			updatePartialModel();
		}
		Expander(const PartialModelType& partialModel, const SmodelsEnhancedProgram& program)
			:	partialModel_(partialModel)
		{
			std::auto_ptr<SmodelsExpander> gcc_295_fix(new SmodelsExpander(program, partialModel));
			pimpl_ = gcc_295_fix;
			updatePartialModel();
		}
		// returns the current state of the Expander
		// OR'd together
		// This could be any combination of the 
		// values above except those hat mutually
		// exclude each other
		// So a valid state would be HAS_UNKNOWN | HAS_CONFLICT
		ExpanderState state() const
		{
			return	(pimpl_->hasUnknown()	? Interface::HAS_UNKNOWN  : Interface::NIL) |
					(pimpl_->hasConflict()	? Interface::HAS_CONFLICT : Interface::NIL);
		}
		
		// shall return false whether the Expander is in shape
		// for another call to expand, true otherwise.
		bool done() const
		{
			return (state() & HAS_CONFLICT) || !(state() &  HAS_UNKNOWN);
		}
		
		// expands the partial model located in the implementation
		// of the Expander with the choice passed.	
		void expand(const ChoiceType& choice)
		{
			pimpl_->makeChoice(choice.atom_.name(), choice.positive_);
			calcExpansionAndUpdatePM();
		}
		
		// returns a PartialModel representation of the internal
		// partial model. 
		PartialModelType partialModel() const
		{
			return partialModel_;
		}
		
		// backtracks the internal partial model to the choice indicated
		// by the argument. If no choice has been made the corresponding
		// exception shall be raised. The method shall raise 
		// NoTruthValue if the atom specified has not yet been assigned
		// a truth value (that is it wasn't a choice or is yet to be one).
		void backtrackTo(const Atom& choice)
		{
			pimpl_->backtrackTo(choice.name());
			partialModel_.reset();
			calcExpansionAndUpdatePM();
		}
	};
}

#endif

