// $Id: program_impl.cpp,v 1.6 2004/08/02 18:34:42 jean Exp $

#include <interfaces/program_impl.h>
#include <interfaces/parser.h>
#include <interfaces/name_generator.h>
#include <interfaces/atom.h>

using namespace std;

namespace Platypus
{
	namespace
	{
		class Names : public LParserCallback
		{
			typedef std::vector<AtomId> Atoms;
			Atoms atoms_;
			NameGenerator generator_;
			string ruleLines_;
			string bodyPlusLines_;
			string bodyMinusLines_;
			string noModels_;
			static void addLine(string& s, const string& line)
			{
				s += line;
				s += '\n';
			}
		public:
			void addAtom(int atom)
			{
				if(generator_.addAtom(atom))
					atoms_.push_back(atom);
			}
			void setName(int atom, const std::string& name)
			{
				generator_.setName(atom, name);
			}
			// This is called by LParser before the parsing loop is left for good
			void done() 
			{
				generator_.generate();
			}
		
			void addRuleLine(const std::string& line)
			{
				addLine(ruleLines_, line);
			}
			
			void addBodyPlusLine(const std::string& line)
			{
				addLine(bodyPlusLines_, line);
			}
			void addBodyMinusLine(const std::string& line)
			{
				addLine(bodyMinusLines_, line);
			}
			void addNoModels(unsigned no)
			{
				stringstream s;
				s << no;
				noModels_ = s.str();
			}
			Atoms& atoms() { return atoms_; }
			const NameGenerator& generator() const { return generator_; }
			const string& ruleLines() const { return ruleLines_; }
			const string& bodyPlusLines() const { return bodyPlusLines_; }
			const string& bodyMinusLines() const { return bodyMinusLines_; }
			const string& noModels() const { return noModels_; }
		};

		string generateLparse(const Names& names, const Program::Id2Name& ids_)
		{
			// add rules
			string lparse_ = names.ruleLines();
			lparse_ += "0\n";

			// add atoms and their names
			for(Program::Id2Name::const_iterator it = ids_.begin();
				it != ids_.end();
				++it)
			{
				stringstream s;
				s << it->first << ' ' << it->second << '\n';
				lparse_ += s.str();
			}
			// add B+
			lparse_ += "0\nB+\n";
			lparse_ += names.bodyPlusLines();

			// add B-
			lparse_ += "0\nB-\n";
			lparse_ += names.bodyMinusLines();

			// add no models
			lparse_ += "0\n";
			lparse_ += names.noModels();
			lparse_ += "\n";
			return lparse_;
		}
	}
	
	void Program::setupMappings(std::istream& is)
	{
		Names names;
		try
		{
			LParser(is, &names).parse();
		}
		catch(const ::ParseError& e)
		{
			throw Platypus::ParseError(e.what());
		}

		atoms_.swap(names.atoms());
		ids_.clear(), names_.clear();
		for(CollectionType::const_iterator it = atoms_.begin();
			it != atoms_.end();
			++it)
		{
			const std::string& name = names.generator().getName(*it);
			ids_.insert(Id2Name::value_type(static_cast<AtomId>(*it), name));
			names_.insert(Name2Id::value_type(name, static_cast<AtomId>(*it)));
		}

		// regenerate lparse output now that each atom has a name
		lparse_ = generateLparse(names, ids_);
	}

	Program::Program()	
	{
		setup(EMPTY_PROGRAM);
	}
	
	void Program::setup(const std::string& input)
	{
		istringstream is(input);
		setupMappings(is);
	}
	void Program::setup(std::istream& input)
	{
		setupMappings(input);
	}
	
	const std::string& Program::idToName(AtomId id) const
	{
		Id2Name::const_iterator res = ids_.find(id);
		if(res == ids_.end())
			throw UnknownAtom();
		return res->second;
	}
	
	// returns the id associated with an atom name
	// NOTE: 	Names MUST be returned for all atoms, wether
	// 			they have a "name" (like a, p(v4), ...) or not.	
	AtomId Program::nameToId(const std::string& name) const
	{
		Name2Id::const_iterator res = names_.find(name);
		if(res == names_.end())
			throw UnknownAtom();
		return res->second;
	}
	
	// returns the collection of atoms of the logic program
	const Program::CollectionType& Program::atoms() const
	{
		return atoms_;
	}
	
	// Returns the definite number of atoms in the logic program.
	// For instance the logic program
	// d :- {c} 1.
	// c.
	//
	// after being preprocessed by lparse contains FOUR atoms.
	size_t Program::numberOfAtoms() const
	{
		return atoms_.size();
	}
	
	// returns true if the atom had a name in the original (textual program), false otherwise.
	bool Program::hasDisplayName(AtomId id) const
	{
		return idToName(id)[0] != '_';
	}
	
	// returns an lparse representation of the preprocessed program.
	const std::string& Program::lparse() const
	{
		return lparse_;
	}
	

}
