// $Id: smodels_expander_impl.cpp,v 1.9 2005/02/20 21:09:30 rtichy Exp $

#include <cassert>
#include <vector>
#include <algorithm>
#include <iostream>
#include <functional>
#include <interfaces/smodels_expander_impl.h>
#include <interfaces/atom.h>
#include <interfaces/partial_model_impl.h>
#include <interfaces/from_idl/exception.h>
#include <interfaces/smodels_program_impl.h>
#include <smodels/read.h>
#include <smodels/smodels.h>
#include <smodels/api.h>
#include <smodels/atomrule.h>

using namespace std;

namespace Platypus
{
	namespace
	{
		class FindByName : public std::unary_function<std::string, bool>
		{
			std::string name_;
		public:
			FindByName(const std::string& name)
				:	name_(name)
			{}
			bool operator()(const ::Atom& atom) const
			{
				assert(atom.name);
				return name_ == atom.name;
			}
		};
	}

	void SmodelsExpander::addComputeStatements(Api& api_, const PartialModel& partialModel)
	{
		for(PartialModel::CollectionType::const_iterator it = partialModel.positiveAtoms().begin();
			it != partialModel.positiveAtoms().end(); ++it)
		{
			iterator res = find_if(iterator(smodels_->program.atoms.head()), iterator(), FindByName(it->name()));
			assert(res != iterator());
			api_.set_compute(&*res, true);
		}

		for(PartialModel::CollectionType::const_iterator it = partialModel.negativeAtoms().begin();
			it != partialModel.negativeAtoms().end(); ++it)
		{
			iterator res = find_if(iterator(smodels_->program.atoms.head()), iterator(), FindByName(it->name()));
			assert(res != iterator());
			api_.set_compute(&*res, false);
		}
	}

	void SmodelsExpander::addComputeStatements(SmodelsHolder* holder, const PartialModel& partialModel)
	{
		for(PartialModel::CollectionType::const_iterator it = partialModel.positiveAtoms().begin();
			it != partialModel.positiveAtoms().end(); ++it)
		{
			SmodelsHolder::const_iterator res = find_if(holder->begin(), holder->end(), FindByName(it->name()));
			assert(res != holder->end());
			holder->addComputeStatement(res, true);
		}

		for(PartialModel::CollectionType::const_iterator it = partialModel.negativeAtoms().begin();
			it != partialModel.negativeAtoms().end(); ++it)
		{
			SmodelsHolder::const_iterator res = find_if(holder->begin(), holder->end(), FindByName(it->name()));
			assert(res != holder->end());
			holder->addComputeStatement(res, false);
		}
	}

	void SmodelsExpander::finalize(Api& api_)
	{
		api_.done();
		smodels_->init();
		// copying is over at this point
		// now start calculating the wellfounded model 
		smodels_->setup_with_lookahead ();
		smodels_->expand ();
	}
	
	void SmodelsExpander::finalize(SmodelsHolder* holder)
	{
		auto_ptr<Smodels> gcc_295_fix(holder->finalize());
		smodels_ = gcc_295_fix;
		// copying is over at this point
		// now start calculating the wellfounded model 
		smodels_->setup_with_lookahead ();
		smodels_->expand ();
	}

	SmodelsExpander::~SmodelsExpander()
	{
		// inhibits compiler warnings about Smodels being an incomplete type
	}

	SmodelsExpander::SmodelsExpander(const Api& api, const PartialModel& partialModel)
		:	smodels_(new Smodels)
	{
		Api api_(&smodels_->program);
		api_.copy(&api);

		// Maybe need to delete old compute statements
		addComputeStatements(api_, partialModel);

		finalize(api_);
	}

	SmodelsExpander::SmodelsExpander(std::istream& is)
		:	smodels_(new Smodels)
	{
		Api api(&smodels_->program);
		Read reader(&smodels_->program, &api);
		if(reader.read(is))
			throw ParseError();
		
		finalize(api);
	}

	SmodelsExpander::SmodelsExpander(const SmodelsEnhancedProgram& program, const PartialModel& partialModel)
	{
		std::auto_ptr<SmodelsHolder> safeHolder(program.create());
		addComputeStatements(safeHolder.get(), partialModel);
		finalize(safeHolder.get());
	}

	void SmodelsExpander::expand()
	{
		// Here we don't assert that the partial model is yet uncovered
		// because the choice set before may have just covered the model
		//
		assert(!hasConflict());
		smodels_->expand();
	}
	void SmodelsExpander::makeChoice(const std::string& name, bool setTrue)
	{
		bool foundit = false;
		for(long i = 0; i < smodels_->atomtop; ++i)
		{
			::Atom *a = smodels_->atom[i];
			assert(a->name);
			if(name == a->name)
			{
				if(a->Bpos || a->Bneg)
					throw InvalidChoice();

				foundit = true;
				smodels_->hi_index = i;
				smodels_->hi_is_positive = setTrue;
				break;
			}
		}
		if(!foundit)
			throw InvalidChoice("[smodels] atom " + name + " is not a valid choice!");
	
		smodels_->choose();
	}

	// This method sort of implements Smodels::backtrack(false)
	// with a few modifications:
	//
	// 1.	The atom to backtrack to is specified
	// 2.	It has to be taken into account whether we are
	//		backtracking from a conflict or not. In Smodels
	//		this is done in the main procedures (smodels, model)
	void SmodelsExpander::backtrackTo(const std::string& name)
	{
		// check with smodels whether there are any choices to 
		// backtrack from
		if(smodels_->guesses == 0)
			throw NoChoiceMade();

		// in case we are backtracking for a conflict, reset smodels
		// conflict flag and some queues
		if(smodels_->conflict_found)
			smodels_->conflict();

		// assert sanity of this operation by checking if the 
		// atom really has been assigned a truth value
		// If we leave this out we'll end up popping the stack
		// empty and not find the atom we are looking for.
		//
		// NOTE: This heavily depends on smodels stack implementation
		::Atom* atom = 0;
		for(long index = smodels_->stack.top - 1; index >= 0; --index)
		{
			// check if the we are passing as backtrack parameter really
			// has been assigned a truth value yet.
			assert(smodels_->stack.stack[index]->name);
			if(name == smodels_->stack.stack[index]->name)
			{
				atom = smodels_->stack.stack[index];
				break;
			}
		}

		if(!atom)
			throw NoTruthValue();

		// reset the truth value of all atoms succeeding
		// the choice
		// this is an implementation of Smodels::unwind()
		// which unwinds the smodels stack to the choice
		// specified
		::Atom *a = smodels_->stack.pop();
		for(; a != atom; a = smodels_->stack.pop())
		{
			if (a->Bpos)
			{
				if (a->posScore > smodels_->score)
					a->posScore = smodels_->score;
				a->backtrackFromBTrue ();
			}
			else if (a->Bneg)
			{
				if (a->negScore > smodels_->score)
					a->negScore = smodels_->score;
				a->backtrackFromBFalse ();
			}
			if (a->guess != false)
			{
				a->guess = false;
				--smodels_->guesses;
			}
		}

		// now that we have worked our way back to the
		// choice see if it really a choice
		// check if the choice has been made
		if(!a->Bpos && !a->Bneg)
			throw NoTruthValue();

		// check if the atom was a choice
		if(!a->guess)
			throw NoChoiceMade("Atom "  + name + " wasn't a choice");
		
		// continue Smodels::backtrack

		// flip the choice		
		if (a->Bneg)
		{
			a->backtrackFromBFalse ();
			a->setBTrue ();
		}
		else
		{
			a->backtrackFromBTrue ();
			a->setBFalse ();
		}
	
		smodels_->stack.push (a);
		smodels_->addAtom(a);
	}

	SmodelsExpander::const_iterator SmodelsExpander::begin() const
	{
		return const_iterator(smodels_->program.atoms.head());
	}
	SmodelsExpander::const_iterator SmodelsExpander::end() const
	{
		return const_iterator();
	}
	bool SmodelsExpander::hasConflict() const
	{
		// Smodels::conflict() returns true one
		// time if there has been a conflict, then resets the conflict state
		// -> this is not what we want
		return smodels_->conflict_found;
	}
	bool SmodelsExpander::hasUnknown() const
	{
		return !smodels_->covered();
	}

	// this method assumes that the positive and negative
	// part of the partial model agree with smodels' idea of 
	// positive and negative atoms
	void SmodelsExpander::update(PartialModel& partialModel) const
	{
		typedef std::vector< std::pair<Atom, bool> > Atom2Truth;
		Atom2Truth atoms;

		// find all the atoms that are now either positive or negative
		// NOTE: A copy is made to ensure no iterators are invalidated in 
		// PartialModel
		for(PartialModel::CollectionType::const_iterator it = partialModel.unknownAtoms().begin();
			it != partialModel.unknownAtoms().end(); ++it)
		{
			const_iterator res = find_if(begin(), end(), FindByName(it->name()));
			assert(res != end());
			if(res->Bpos)
				atoms.push_back(make_pair(*it, true));
			else if(res->Bneg)
				atoms.push_back(make_pair(*it, false));
		}

		// now that we have copies move atoms into their
		// new state
		for(Atom2Truth::const_iterator it = atoms.begin();
			it != atoms.end();	++it)
		{
			if(it->second)
				partialModel.setTrue(it->first);
			else
				partialModel.setFalse(it->first);
		}
	}
}
