#include "ProcessControl.h"
#include <iostream>

/*======================================================================
 * ProcessControl.cc
 * Richard Tichy
 *
 * Modifications:
 *     May 25th - code level commenting complete
 *======================================================================*/

using Platypus::FORK_ERROR;
using Platypus::CHILD;
using Platypus::MUTEX_EXPANDFN;
using Platypus::MUTEX_SHAREDMEM;

/*
 * Control the creation of new processes using the 
 * Unix/Linux fork call.
 */
int Platypus::forkProcess(int semid, SharedMemory * shm){
 
  int childpid = FORK_ERROR;

  /*******************************************
   *        LOCK
   *******************************************/
  waitSem(semid, MUTEX_SHAREDMEM);

  if(shm->pidIndex < (MAX_NUM_PROCESSES)){
    //if we haven't exceeded the max number of forks that we've set
    if((--(shm->maxNumberOfForks) + 1) > 0){
      
      childpid=fork();
      
      if(childpid == FORK_ERROR){
	shm->maxNumberOfForks++;

	signalSem(semid, MUTEX_SHAREDMEM); 
	/*******************************************
	 *        RELEASE
	 *******************************************/
      }	    
      else if(childpid==CHILD){
	shm->ppids[shm->pidIndex] = getpid();
	shm->ppidWaiting[shm->pidIndex++] = false;

	//std::cout << std::endl << "shm->pidIndex: " << shm->pidIndex << std::endl;

	signalSem(semid, MUTEX_SHAREDMEM); 
	/*******************************************
	 *        RELEASE
	 *******************************************/
      }
    }
    //if we've exceeded our max_number of forks
    else{
      shm->maxNumberOfForks++;
      childpid=FORK_ERROR;

      signalSem(semid, MUTEX_SHAREDMEM);
      /*******************************************
       *        RELEASE
       *******************************************/
    }
  }else{
    childpid=FORK_ERROR;

    std::cout << "**************************************" << std:: endl 
	      << "**************************************" << std:: endl
	      << "HIT MAX NUM FORKS CEILING ***" << std::endl;

    signalSem(semid, MUTEX_SHAREDMEM);
    /*******************************************
     *        RELEASE
     *******************************************/
  } 

  return childpid;
  
}

/*
 * Method allows child processes to exit without creating orphaned
 * zombie processes.
 */
void Platypus::childExiting(int semid, SharedMemory * shm){
  
  /*******************************************
   *        LOCK
   *******************************************/
  waitSem(semid, MUTEX_SHAREDMEM);
  
  //find yourself in the ppid array
  int i;
  pid_t pid;
  pid = getpid();
  
  //no need for continous polling, if you do not find yourself
  //in the array you have larger problem
  bool found = false;
  while(!found){
    for (i=0; i < shm->pidIndex; i++){
      if (shm->ppids[i] == pid){
	found = true;
	break;
      }
    }
  }

  //if the process did not find himself in the array
  //cry for help
  assert(found);

  //place your exit status in the ppidWaiting array
  shm->ppidWaiting[i] = true;

  signalSem(semid, MUTEX_SHAREDMEM);
  /*******************************************
   *        RELEASE
   *******************************************/
  
  exit(0);

}

/*
 * Allows the parent of a process to reap created children
 * and avoid orphaned zombie processes.
 */
void Platypus::parentReturning(int semid, SharedMemory * shm, vector<pid_t> & cids){
  
  /*******************************************
   *        LOCK
   *******************************************/
  //waitSem(semid, MUTEX_SHAREDMEM);
  
  //shm->maxNumberOfForks++;
  
  //signalSem(semid, MUTEX_SHAREDMEM);
  /*******************************************
   *        RELEASE
   *******************************************/

  while(!cids.empty()){

    vector<pid_t>::iterator it = cids.begin();

    for(;it != cids.end(); it++){

      pid_t waitForPid= *it;

      //PARENT IS EXITING (now needs to "wait" for each of it's children)
      /*******************************************
       *        LOCK
       *******************************************/
      waitSem(semid, MUTEX_SHAREDMEM);
  
      //Find waitForPid in the ppid table
      bool found = false;
      int i;
      while(!found){
	for (i=0; i < shm->pidIndex; i++){
	  if (shm->ppids[i] == waitForPid){
	    found = true;
	    break;
	  }
	}
      }

      assert(found);
      
      //if the child is ready to be reaped
      if(shm->ppidWaiting[i] == true){
       
	waitpid(waitForPid,NULL,0);
	
	//remove the child from the array
	for (int kk=i+1; kk<shm->pidIndex; kk++){
	  shm->ppids[kk-1] = shm->ppids[kk];
	  shm->ppidWaiting[kk-1] = shm->ppidWaiting[kk];
	}

	//std::cout << std::endl << getpid() << " REMOVED CHILD " <<  waitForPid << " FROM ARRAY" << std::endl;

	shm->pidIndex--;
	//shm->maxNumberOfForks++;
	cids.erase(it);
	signalSem(semid, MUTEX_SHAREDMEM);
	/*******************************************
	 *        RELEASE
	 *******************************************/
	break;//break out the for loop iterating through the vector
      }
      
      signalSem(semid, MUTEX_SHAREDMEM);
      /*******************************************
       *        RELEASE
       *******************************************/
      
    }//end for

    //sleep for a little bit to give the children a little bit
    //of time to exit, provents a busy which could realy detriment 
    //performance if many parents a re waiting to return on a single
    //processor
    usleep(5000);

  }//end while

  return;

}


