#ifndef __SHAREDMEMORY__INTERFACE__H__
#define __SHAREDMEMORY__INTERFACE__H__

/*======================================================================== 
 * sharedMemoryInterface.h
 * Christine Bailey and Richard Tichy
 *
 * This file holds the interfaces of functions specific to creating, 
 * accessing and destroying sharedmemory.
 *
 * Modification:
 *     May 25th - doxygen style documentation done
 *=========================================================================*/

#include <sys/shm.h>
#include <limits.h>
#include <iostream>
//#include "Types.h"


namespace Platypus{

  /**   
   * This function creates a key that can be used to create shared memory
   * segments or semaphores.
   *
   * @param char c - unique id
   * @param key_t - key value
   */
  key_t getSharedMemoryKey(char c);

  /**    
   * This function creates a shared memory segment from the given key and 
   * returns the id of the segment.
   * 
   * @param key_t key - key used to access the segment
   * @return int - id of the shared memory accessed
   */
  int createSegment(key_t key);

  /**  
   * This function accesses and already existing shared memory segment using 
   * the given key and returns the id of the segment.
   *
   * @param key_t key - the key to be used to access the segment
   * @return int - the id of the shared memory just accessed
   */
  int getSegment(key_t key);

  /** 
   * This function attaches a pointer to the shared memory associated with
   * the shared memory id shmid for access; will return
   * (void*)(-1) if there was an error.
   *
   * @param void* data - the pointer we want to attach to the shared memory
   * @param int shmid - the id of the shared memory we want to attach to
   * @return void* - the data that's now attached to the shared memory. 
   */
  void* attachSegment(void* data, int shmid);
  
  /** 
   * This function detaches a pointer from shared memory.
   *
   * @param void* data - the pointer we want to detach from shared memory
   * @return bool - returns SUCCESS if detach succeeds, FAILURE otherwise.
   */
  bool detachSegment(void* data);
  
  /**
   * This function deallocates the shared memory associated with the
   * given shared memory id.
   *
   * @param int shmid - the id of the shared memory we want to remove.
   * @return bool - returns SUCCESS if detach succeeds, else FAILURE
   */
  bool removeSegment(int shmid);

}

#endif
