/*	$Id: ipc.h 1728 2005-05-06 08:08:53Z jgressma $
 *
 *  Copyright 2005 University of Potsdam, Germany
 * 
 *	This file is part of Platypus.
 *
 *  Platypus is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  Platypus is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with Platypus; if not, write to the Free Software
 *  Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 *
 */


#ifndef IPC_H
#define IPC_H

#include <string>
#include <stdexcept>
#include <vector>

namespace IPC
{
	class IPCException : public std::runtime_error
	{
	public:
		IPCException(const std::string& reason = std::string())
			:	std::runtime_error(reason)
		{}
	};

	class IPCMQError : public IPCException
	{
	public:
		IPCMQError(const std::string& reason = std::string())
			:	IPCException(reason)
		{}
	};
	
	class IPCSHMError : public IPCException
	{
	public:
		IPCSHMError(const std::string& reason = std::string())
			:	IPCException(reason)
		{}
	};

	class IPCMessageQueue;
	class IPCMessageBuffer
	{
		friend class IPCMessageQueue;
	public:
		IPCMessageBuffer();
		void write(const char* data, size_t length);
		void resize(size_t length = 0);
		const char* data() const;
		char* data();
		size_t size() const; 
	private:
		void setType(unsigned long type);
		inline const char* bytes() const { return &data_[0]; }
		inline char* bytes() { return &data_[0]; }
		void grow();
	private:
		std::vector<char> data_;
	};

	class IPCMessageQueue
	{
	public:
		~IPCMessageQueue();
		IPCMessageQueue(const char* name, int mode = 0600, bool create = false, bool remove = true);
		bool send(unsigned long type, const IPCMessageBuffer& buffer, bool synchronous = true);
		bool receive(unsigned long type, IPCMessageBuffer& buffer, bool synchronous = true);
		void remove(bool b) { remove_ = b; }
	private:
		IPCMessageQueue();
	private:
		int queue_;
		bool remove_;
	};
	
	class IPCSharedMemory
	{
	public:
		~IPCSharedMemory();
		IPCSharedMemory(size_t size, const char* name, int mode = 0600, bool create = false, bool remove = true);
		inline void* raw() { return raw_; }
		inline const void* raw() const { return raw_; }
		inline size_t size() const { return size_; }
		void remove(bool b) { remove_ = b; }
	private:
		IPCSharedMemory();
	private:
		size_t size_;
		void* raw_;
		int id_;		
		bool remove_;
	};
	
	template<class T>
	class IPCSHMObject
	{
	public:
		~IPCSHMObject()
		{
			// remember, this object is in shared memory!
			// only ONE process should destroy it
			if(remove_)
				t_->~T();
		}
		IPCSHMObject(IPCSharedMemory& shm, const T& t = T(), size_t byteoffset = 0, bool remove = true)
			:	t_(0)
			,	remove_(remove)
		{
			assert(sizeof(T) <= shm.size());
			assert(byteoffset + sizeof(T) <= shm.size());
			t_ = new (static_cast<void*>(static_cast<char*>(shm.raw())+byteoffset)) T(t);
			assert(t_);
		}
		inline void remove(bool b) { remove_ = b; }
		inline T& reference() { return *t_; }
		inline const T& const_reference() const { return *t_; }
		inline operator T&() { return reference(); }
		inline operator const T&() const { return const_reference(); }
	private:
		IPCSHMObject();
	private:
		T* t_;
		bool remove_;
	};
}

#endif
