/*	$Id: foreman.cpp 1728 2005-05-06 08:08:53Z jgressma $
 *
 *  Copyright 2005 University of Potsdam, Germany
 * 
 *	This file is part of Platypus.
 *
 *  Platypus is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  Platypus is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with Platypus; if not, write to the Free Software
 *  Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 *
 */

#include <cstdio>
#include <stdexcept>
#include <cassert>
#include <nop_stream.h>
#include <portablethreads/utility.h>
#include <platypus/types.h>
#include <platypus/types_factory.h>
#include <platypus/options.h>
#include <platypus/cores/worker.h>
#include <platypus/cores/foreman.h>

using namespace std;

namespace Platypus
{
	/************************************************************************/
	/* ForemanAssistant                                                     */
	/************************************************************************/
	ForemanAssistant::ForemanAssistant()
		:	processId_(-1)
		,	program_(0)
		,	factory_(0)
		,	options_(0)
		,	os_(0)
	{}
	bool ForemanAssistant::silent() const
	{
		assert(options_);
		return options_->silent();
	}
	ProbingManager* ForemanAssistant::probingManager()
	{
		assert(options_);
		return new ProbingManager(
			options_->probingInitialCutoff(),
			options_->probingSteps(),
			options_->probing());
	}
	NopStream* ForemanAssistant::output()
	{
		assert(os_);
		return os_;
	}
	size_t ForemanAssistant::atoms() const
	{
		assert(program_);
		return program_->numberOfAtoms();
	}
	DelegationHeuristicInterface* ForemanAssistant::delegationHeuristic()
	{
		assert(options_);
		return new DelegateIfPercentageExceeded(options_->percentage());
	}
	void ForemanAssistant::program(ProgramInterface& prog)
	{
		program_ = &prog;
	}
	void ForemanAssistant::factory(const PlatypusTypesFactory& factory)
	{
		factory_ = &factory;
	}
	void ForemanAssistant::output(NopStream& str)
	{
		os_ = &str;
	}
	void ForemanAssistant::options(const PlatypusOptions& values)
	{
		options_ = &values;
	}
	Expander* ForemanAssistant::expander(const DelegatableChoice& dc)
	{
		assert(options_);
		assert(factory_);
		std::auto_ptr<Expander> expander(factory_->expanderFactory().create(*program_, dc, options_->expander()));
		expander->setChoicePolicy(factory_->choicePolicyFactory().create(options_->localChoicePolicy()));
		expander->setManualChoice(options_->manualChoices());
		
		return expander.release();
	}
	
	DCPolicy* ForemanAssistant::delegatableChoicePolicy()
	{
		assert(options_);
		assert(factory_);
		return factory_->delegatableChoicePolicyFactory().create(options_->delegatableChoicePolicy());
	}
	int ForemanAssistant::processId() const
	{
		return processId_;
	}
	void ForemanAssistant::processId(int id)
	{
		processId_ = id;
	}

	/************************************************************************/
	/* Foreman                                                              */
	/************************************************************************/
	Foreman::~Foreman()
	{}
	Foreman::Foreman()
		:	assistant_(0)
		,	traits_(0)
		,	program_(0)
		,	os_(0)
		,	threads_(0)
		,	filedForDCWithTraits_(false)
		,	silent_(false)
		,	shutdown_(false)
		,	flagNeedForDC_(false)
	{}
	

	void Foreman::run()
	{
		createThreads();
		while(!shutdown())
		{
			iterate();
			PortableThreads::pt_milli_sleep(50);
		}
		joinThreads();

		updateStatistics();

		// The way things currently work distribution might 
		// just shut us down b/c we have reported say 6 answer
		// sets to be found. However, so far we have printed only
		// two. So the remaining 4 must be in the buffer. B/c
		// the printing engine expects them, forward
		handleAnswerSets();
	}
	
	void Foreman::updateStatistics()
	{
		int_type current = 0;

		current = deltaAnswerSets_.get();
		deltaAnswerSets_.dec(current);
		traits_->incAnswerSets(static_cast<size_t>(current));

		current = deltaConflicts_.get();
		deltaConflicts_.dec(current);
		traits_->incConflicts(static_cast<size_t>(current));

		current = deltaBacktracks_.get();
		deltaBacktracks_.dec(current);
		traits_->incBacktracks(static_cast<size_t>(current));

		current = deltaBacktracks_.get();
		deltaBacktracks_.dec(current);
		traits_->incBacktracks(static_cast<size_t>(current));

		current = deltaInits_.get();
		deltaInits_.dec(current);
		traits_->incExpanderInitializations(static_cast<size_t>(current));

		current = deltaDelegations_.get();
		deltaDelegations_.dec(current);
		traits_->incThreadDelegations(static_cast<size_t>(current));
	}
	void Foreman::handleDelegation()
	{
		if(traits_->needDelegatableChoice())
		{
			DelegatableChoice dc;
			if(delegateQueue_.popFront(dc))
			{
				traits_->delegate(dc);
			}
			else
			{
				flagNeedForDC_ = true;
			}
		}
		// no else, thread might find the flag set even though no dc is
		// needed. In this case it will put a dc in the queue which will
		// be retrieved later.
	}
	void Foreman::handleAnswerSets()
	{
		if(!silent_)
		{
			PartialAssignment pa(*program_);
			while(answerSets_.pop(pa))
				traits_->answerSet(pa);
		}
	}
	void Foreman::handleNoWork()
	{ 
		assert(threads_ >= threadsThatNeedSomethingToDo_.get());
		if(threads_ == threadsThatNeedSomethingToDo_.get())
		{
			DelegatableChoice dc;
			if(delegateQueue_.popFront(dc))
			{
				addToThreadQueue(dc, true);
			}
			else
			{
				if(!filedForDCWithTraits_)
				{
					filedForDCWithTraits_ = true;
					traits_->fileDelegatableChoiceRequest();				
				}
				else
				{
					if(traits_->hasDelegatableChoice())
					{
						filedForDCWithTraits_ = false;
						addToThreadQueue(traits_->delegatableChoice(), true);
					}
				}
			}
		}
	}
	void Foreman::iterate()
	{
		updateStatistics();
		handleNoWork();
		handleDelegation();
		handleAnswerSets();
		shutdown_ = traits_->shutdown();	
	}
	bool Foreman::shutdown()
	{
		return shutdown_;
	}
	bool Foreman::delegate()
	{
		return flagNeedForDC_;
	}
	void Foreman::delegate(const DelegatableChoice& dc)
	{
		flagNeedForDC_ = false;
		delegateQueue_.pushBack(dc);
	}
	
	void Foreman::answerSet(const PartialAssignment& pa)
	{
		answerSets_.push(pa);
	}
	void Foreman::markConflict()
	{
		++deltaConflicts_;
	}
	void Foreman::markAnswerSet()
	{
		++deltaAnswerSets_;
	}
	void Foreman::markBacktrack()
	{
		++deltaBacktracks_;
	}
	void Foreman::markExpanderInitialization()
	{
		++deltaInits_;
	}
	bool Foreman::branch()
	{
		return threadsThatNeedSomethingToDo_.get() > 0;
	}
	void Foreman::addToThreadQueue(const DelegatableChoice& dc, bool forman)
	{
		if(!forman)
			++deltaDelegations_;
		localQueue_.pushBack(dc);
		--threadsThatNeedSomethingToDo_;
		blockOnLocalQueue_.up();
	}
	void Foreman::branch(const DelegatableChoice& dc)
	{
		addToThreadQueue(dc, false);
	}
	bool Foreman::wait()
	{
		++threadsThatNeedSomethingToDo_;
		blockOnLocalQueue_.down();
		
		return shutdown_;
	}
	DelegatableChoice Foreman::retrieve()
	{
		DelegatableChoice dc;
#ifdef NDEBUG
		localQueue_.popFront(dc);
#else
		const bool ret = localQueue_.popFront(dc);
		assert(ret);
#endif
		return dc;
	}

	void Foreman::createThreads()
	{
		workers_.resize(threads_);
		for(int_type i = 0; i < threads_; ++i)
		{
			workers_[i] = new Worker(i, *this, *assistant_);
		}
		for(int_type i = 0; i < threads_; ++i)
		{
			workers_[i]->run();
		}
	}
	void Foreman::joinThreads()
	{
		assert(shutdown_);
		for(int_type i = 0; i < threads_; ++i)
		{
			blockOnLocalQueue_.up();
		}
		for(int_type i = 0; i < threads_; ++i)
		{
			workers_[i]->join();
		}
	}
	void Foreman::threads(unsigned t)
	{
		threads_ = t;
	}
	void Foreman::silent(bool yes)
	{
		silent_ = yes;
	}
	void Foreman::program(const ProgramInterface& prog)
	{
		program_ = &prog;
	}
	void Foreman::output(NopStream& str)
	{
		os_ = &str;
	}
	void Foreman::distribution(CoreDistributionCallback& callback)
	{
		traits_ = &callback;
	}
	void Foreman::assistant(ForemanAssistant& assistant)
	{
		assistant_ = &assistant;
	}
	void Foreman::setup()
	{
		assert(program_);
		assert(traits_);
		assert(assistant_);
		assert(threads_ >= 1);

		assistant_->processId(traits_->id());
	}
}
