/*	$Id: kernel.cpp 1728 2005-05-06 08:08:53Z jgressma $
 *
 *  Copyright 2005 University of Potsdam, Germany
 * 
 *	This file is part of Platypus.
 *
 *  Platypus is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  Platypus is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with Platypus; if not, write to the Free Software
 *  Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 *
 */

#include <cassert>
#include <iostream>
#include <nop_stream.h>
#include <platypus/types.h>
#include <platypus/cores/kernel.h>

using namespace std;

namespace Platypus
{
	Kernel::~Kernel()
	{}
	Kernel::Kernel(KernelCallback* callback)
		:	callback_(callback)
		,	os_(0)
		,	suppressAnswerSetPrinting_(false)
	{}
	void Kernel::setCallback(KernelCallback& callback)
	{
		callback_ = &callback;
	}
	
	void Kernel::configure()
	{
		assert(callback_);
		delegationHeuristic_.reset(callback_->delegationHeuristic());
		assert(delegationHeuristic_.get());
		suppressAnswerSetPrinting_ = callback_->silent();
		probing_.reset(callback_->probingManager());
		assert(probing_.get());
		expander_.reset(callback_->expander());
		assert(expander_.get());
		atoms_ = callback_->atoms();
		os_ = callback_->output();
		assert(os_);
	}
	void Kernel::initialize()
	{
		assert(!probing_->exhausted());
		choicesToDo_.reset(probing_->select());

		(*os_) << 'P' << callback_->processId() << 'T' << callback_->threadId() << " reinitializing expander\n";
		expander_->reinitialize(choicesToDo_->core());
		callback_->markExpanderInitialization();

		ChoiceFeeder feeder(choicesToDo_->feeder());
		feeder.feed(*expander_);
	}
	void Kernel::run(const DelegatableChoice& dc)
	{
		assert(callback_);
		assert(probing_.get());

		choicesToDo_.reset(new ChoicesToDo(dc));
		choicesToDo_->setDelegatableChoicePolicy(callback_->delegatableChoicePolicy());
		probing_->add(choicesToDo_.release());

		while(!callback_->shutdown() && !probing_->exhausted())
		{
			initialize();
			bool done = false;
			while(!callback_->shutdown() && !done)
			{
				done = !iteration();
			}
		}
	}
	bool Kernel::iteration()
	{
		if(expander_->hasConflict())
		{
			return conflict();
		}
		else if (!expander_->hasUnknown())
		{
			return answerSet();
		}
		return expand();
	}
	bool Kernel::answerSet()
	{
		assert(!expander_->hasConflict());
		assert(!expander_->hasUnknown());

		callback_->markAnswerSet();
		probing_->markAnswerSet();
		if(!suppressAnswerSetPrinting_)
		{
			callback_->answerSet(expander_->partialAssignment());
		}
		
		if(choicesToDo_->hasChoice())
		{
			return backtrack();
		}
		return false;
	}
	bool Kernel::conflict()
	{
		assert(expander_->hasConflict());
		callback_->markConflict();
		if(choicesToDo_->hasChoice())
		{
			return backtrack();
		}
		return false;
	}
	
	bool Kernel::expand()
	{
		assert(expander_->hasUnknown());
		assert(!expander_->hasConflict());
		

		const Choice c(expander_->getChoice());
		choicesToDo_->add(c);

		if(callback_->delegate())
		{
			delegate();
		}
		if(callback_->branch())
		{
			branch();
		}
		
		expander_->expand(c);
		return true;
	}
	void Kernel::delegate()
	{
		if(delegationHeuristic_->delegate(*expander_, atoms_))
		{
			const DelegatableChoice dc(choicesToDo_->nextDelegatableChoice());
			(*os_) << 'P' << callback_->processId() << 'T' << callback_->threadId() << ' ' << "giving " << dc << " away\n"; 
			
			callback_->delegate(dc);
		}
	}
	void Kernel::branch()
	{
		if(!probing_->exhausted())
		{
			auto_ptr<ChoicesToDo> reinsert(probing_->select());
			auto_ptr<ChoicesToDo> giveaway;
			if(reinsert->hasChoice())
			{
				auto_ptr<ChoicesToDo> giveaway(reinsert->split());
				probing_->add(reinsert.release());
			}
			else
			{
				giveaway = reinsert;
			}
			assert(!giveaway->hasChoice());
			callback_->branch(giveaway->core());
		}
		else
		{
			if(choicesToDo_->hasChoice())
			{
				auto_ptr<ChoicesToDo> giveaway(choicesToDo_->split());
				assert(!giveaway->hasChoice());
				callback_->branch(giveaway->core());
			}
		}
	}
	bool Kernel::backtrack()
	{
		assert(choicesToDo_->hasChoice());
		do
		{
			(*os_) << 'P' << callback_->processId() << 'T' << callback_->threadId() << " backtracking expander\n";
			probing_->markBacktrack();				
			callback_->markBacktrack();

			const Choice choice = choicesToDo_->nextChoice();			
			expander_->backtrackTo(choice);
		}
		while(expander_->hasConflict() && choicesToDo_->hasChoice());

		if(!expander_->hasConflict() && expander_->hasUnknown() && probing_->restart())
		{
			probing_->add(choicesToDo_.release());
			return false;
		}
		return !expander_->hasConflict();
	}
}
