/*	$Id: worker.cpp 1728 2005-05-06 08:08:53Z jgressma $
 *
 *  Copyright 2005 University of Potsdam, Germany
 * 
 *	This file is part of Platypus.
 *
 *  Platypus is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  Platypus is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with Platypus; if not, write to the Free Software
 *  Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 *
 */

#include <cassert>
#include <platypus/cores/kernel.h>
#include <platypus/cores/foreman.h>
#include <platypus/cores/worker.h>

namespace Platypus
{
	/************************************************************************/
	/* WorkerCallbackAdaptor                                                    */
	/************************************************************************/
	class WorkerCallbackAdaptor	: public KernelCallback
	{
	public:
		WorkerCallbackAdaptor(int id, Foreman& df, ForemanAssistant& tf);
	private:
		bool shutdown();
		bool delegate();		
		void delegate(const DelegatableChoice& dc);
		bool branch();
		void branch(const DelegatableChoice& dc);
		Expander* expander(const DelegatableChoice& dc = DelegatableChoice()); 
		DCPolicy* delegatableChoicePolicy();
		void answerSet(const PartialAssignment& pa);
		void markConflict();
		void markAnswerSet();
		void markBacktrack();
		void markExpanderInitialization();
		int processId() const;
		int threadId() const;
		bool silent() const;
		ProbingManager* probingManager();
		NopStream* output();
		size_t atoms() const;
		DelegationHeuristicInterface* delegationHeuristic();
	private:
		const int id_;
		Foreman* foreman_;
		ForemanAssistant* assistant_;
	};

	WorkerCallbackAdaptor::WorkerCallbackAdaptor(int id, Foreman& df, ForemanAssistant& tf)
		:	id_(id)
		,	foreman_(&df)
		,	assistant_(&tf)
	{}
	
	bool WorkerCallbackAdaptor::delegate()
	{
		return foreman_->delegate();
	}
	bool WorkerCallbackAdaptor::shutdown()
	{
		return foreman_->shutdown();
	}
	void WorkerCallbackAdaptor::delegate(const DelegatableChoice& dc)
	{
		foreman_->delegate(dc);
	}
	Expander* WorkerCallbackAdaptor::expander(const DelegatableChoice& dc)
	{
		return assistant_->expander(dc);
	}
	
	DCPolicy* WorkerCallbackAdaptor::delegatableChoicePolicy()
	{
		return assistant_->delegatableChoicePolicy();
	}
	void WorkerCallbackAdaptor::answerSet(const PartialAssignment& pa)
	{
		foreman_->answerSet(pa);
	}
	void WorkerCallbackAdaptor::markConflict()
	{
		foreman_->markConflict();
	}
	void WorkerCallbackAdaptor::markAnswerSet()
	{
		foreman_->markAnswerSet();
	}
	void WorkerCallbackAdaptor::markBacktrack()
	{
		foreman_->markBacktrack();
	}
	void WorkerCallbackAdaptor::markExpanderInitialization()
	{
		foreman_->markExpanderInitialization();
	}
	int WorkerCallbackAdaptor::processId() const
	{
		return assistant_->processId();
	}
	int WorkerCallbackAdaptor::threadId() const
	{
		return id_;
	}
	bool WorkerCallbackAdaptor::silent() const
	{
		return assistant_->silent();
	}
	ProbingManager* WorkerCallbackAdaptor::probingManager()
	{
		return assistant_->probingManager();
	}
	NopStream* WorkerCallbackAdaptor::output()
	{
		return assistant_->output();
	}
	size_t WorkerCallbackAdaptor::atoms() const
	{
		return assistant_->atoms();
	}
	DelegationHeuristicInterface* WorkerCallbackAdaptor::delegationHeuristic()
	{
		return assistant_->delegationHeuristic();
	}
	bool WorkerCallbackAdaptor::branch()
	{
		return foreman_->branch();
	}
	void WorkerCallbackAdaptor::branch(const DelegatableChoice& dc)
	{
		foreman_->branch(dc);
	}

	/************************************************************************/
	/* Worker                                                               */
	/************************************************************************/
	Worker::~Worker()
	{}
	Worker::Worker(int id, Foreman& df, ForemanAssistant& tf)
		:	foreman_(&df)
		,	callback_(new WorkerCallbackAdaptor(id, df, tf))
		,	kernel_(new Kernel(callback_.get()))
	{}
	void Worker::threadImpl()
	{
		kernel_->configure();
		while(!foreman_->wait())
		{
			const DelegatableChoice dc = foreman_->retrieve();
			kernel_->run(dc);
		}
	}
}
