/*	$Id: choices_to_do.cpp 1728 2005-05-06 08:08:53Z jgressma $
 *
 *  Copyright 2005 University of Potsdam, Germany
 * 
 *	This file is part of Platypus.
 *
 *  Platypus is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  Platypus is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with Platypus; if not, write to the Free Software
 *  Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 *
 */

#include <iostream>
#include <functional>
#include <algorithm>
#include <cassert>
#include <memory>
#include <platypus/program_base.h>
#include <platypus/types/choices_to_do.h>
#include <platypus/types/delegatable_choice_policy.h>
#include <platypus/types/exceptions.h>

using namespace std;

namespace Platypus
{
	///////////////////////////////////////////////////////////
	// DCPolicy
	///////////////////////////////////////////////////////////
	bool DCPolicy::possibleChoice(const ChoiceInfo& info) const
	{
		return !info.core_ && !info.delegated_ && !info.toggled_;
	}

	///////////////////////////////////////////////////////////
	// ChoicesToDo
	///////////////////////////////////////////////////////////
	namespace
	{
		struct CompareById : public binary_function<ChoiceInfo, ChoiceId, bool>
		{
			inline bool operator()(const ChoiceInfo& lhs, ChoiceId rhs) const
			{
				return lhs.choice_.id() == rhs;
			}
		};

		static inline const Choice& project(const ChoiceInfo& ci)
		{
			return ci.choice_;
		}
	}

	ChoicesToDo::ChoicesToDo(const DelegatableChoice& dc)
		:	latest_(-1)
		,	coreSize_(0)
	{
		reinitialize(dc);
	}

	void ChoicesToDo::reinitialize(const DelegatableChoice& dc)
	{
		choiceHistory_.clear();
		choiceHistory_.reserve(dc.size());
		for(DelegatableChoice::const_iterator it = dc.begin(); 
			it != dc.end(); ++it)
		{
			//                                 choice, toggled, core
			choiceHistory_.push_back(ChoiceInfo(*it, false, true));
		}
		latest_ = -1;
		coreSize_ = choiceHistory_.size();
	}	

	void ChoicesToDo::setDelegatableChoicePolicy(const DelegatableChoicePolicyType& policy)
	{
		policy_.reset(policy.clone());
	}
	void ChoicesToDo::setDelegatableChoicePolicy(DelegatableChoicePolicyType* policy)
	{
		assert(policy);
		policy_.reset(policy);
	}
	void ChoicesToDo::add(const Choice& choice)
	{
		// make sure we don't have this choice in store
		assert(find_if(choiceHistory_.begin(), choiceHistory_.end(), bind2nd(CompareById(), choice.id())) 
			== choiceHistory_.end());

		choiceHistory_.push_back(choice);
		latest_ = static_cast<int>(choiceHistory_.size()) - 1;
	}
	bool ChoicesToDo::hasChoice() const
	{
		return latest_ != -1;
	}
	Choice ChoicesToDo::nextChoice()
	{
		if(!hasChoice())
			throw NoChoiceLeft();

		popUsed();

		choiceHistory_.back().choice_.toggle();
		choiceHistory_.back().toggled_ = true;

		findLatestChoice();

		return choiceHistory_.back().choice_;
	}
	bool ChoicesToDo::flippable(const ChoiceInfo& ci)
	{
		return !ci.toggled_ && !ci.delegated_ && !ci.core_;
	}
	void ChoicesToDo::findLatestChoice()
	{
		assert(latest_ == -1 || !flippable(choiceHistory_[latest_]));
		for(latest_ = static_cast<int>(choiceHistory_.size()) - 1; latest_ >= 0; --latest_)
		{
			if(flippable(choiceHistory_[latest_]))
			{
				break;
			}
		}
		
	}
	void ChoicesToDo::popUsed()
	{
		while(choiceHistory_.back().toggled_ || choiceHistory_.back().delegated_)
		{
			assert(!choiceHistory_.back().core_);
			choiceHistory_.pop_back();
		}
		assert(!choiceHistory_.empty());
	}
	DelegatableChoice ChoicesToDo::nextDelegatableChoice()
	{
		assert(policy_.get());
		if(!hasChoice())
			throw NoChoiceLeft();

		assert(coreSize_ >= 0 && coreSize_ <= choiceHistory_.size());

		const const_iterator res = policy_->choose(choiceHistory_.begin() + coreSize_, choiceHistory_.end());
		return createDelegatableChoice(res);
	}

	DelegatableChoice ChoicesToDo::createDelegatableChoice(const const_iterator chosen)
	{
		assert(chosen != choiceHistory_.end());
		assert(flippable(*chosen));

		const int d = (int)std::distance(const_cast<const ChoicesDone&>(choiceHistory_).begin(), chosen);
		choiceHistory_[d].delegated_ = true;

		// The choice found must either be 
		// before the latest choice (chronologically)
		// made or must be the last choice. In the 
		// later case we need to find a new latest
		// choice.
		if(d == latest_) // latest is valid here, hasChoice makes sure
			findLatestChoice();
		

		DelegatableChoice dc;
		dc.reserve(d+1);
		transform(choiceHistory_.begin(), choiceHistory_.begin()+d, back_inserter(dc), project);
		dc.push_back(chosen->choice_);
		dc.back().toggle();
		return dc;
	}

	DelegatableChoice ChoicesToDo::choiceHistory() const
	{
		DelegatableChoice history;
		history.reserve(choiceHistory_.size());
		transform(choiceHistory_.begin(), choiceHistory_.end(), back_inserter(history), project);
		return history;
	}

	void ChoicesToDo::extendCore()
	{
		const size_t size = choiceHistory_.size();
		if(size)
		{
			for(size_t coreIndex = coreSize_; coreSize_ < size; ++coreIndex)
			{
				assert(!choiceHistory_[coreIndex].core_);
				if(!choiceHistory_[coreIndex].delegated_ && !choiceHistory_[coreIndex].toggled_)
					break;

				choiceHistory_[coreIndex].core_ = true;
				++coreSize_;
			}
		}
	}

	DelegatableChoice ChoicesToDo::core()
	{
		extendCore();

		DelegatableChoice core;
		core.reserve(coreSize_);
		transform(choiceHistory_.begin(), choiceHistory_.begin() + coreSize_, back_inserter(core), project);
		return core;
	}

	ChoiceFeeder ChoicesToDo::feeder() const
	{
		DelegatableChoice eatthis;
		eatthis.reserve(choiceHistory_.size() - coreSize_);
		transform(choiceHistory_.begin() + coreSize_, choiceHistory_.end(), back_inserter(eatthis), project);


		// construction takes ownership of eatthis!!!
		return ChoiceFeeder(eatthis);
	}

	ChoicesToDo* ChoicesToDo::split()
	{
		assert(hasChoice());
		extendCore();

		assert(coreSize_ < choiceHistory_.size());
		DelegatableChoice dc = createDelegatableChoice(choiceHistory_.begin() + coreSize_);

		// find new core size
		extendCore();

		ChoicesToDo* ctd = new ChoicesToDo(dc);
		ctd->setDelegatableChoicePolicy(policy_->clone());
		/*
		ChoicesToDo* ctd = new ChoicesToDo(nextDelegatableChoice());
		ctd->setDelegatableChoicePolicy(policy_->clone());
		*/
		return ctd;
	}
	size_t ChoicesToDo::coreSize() const
	{
		assert(coreSize_ >= 0);
		return coreSize_;
	}

	std::ostream& operator<<(std::ostream& os, const ChoicesToDo& ctd)
	{
		os << "Stack:\n";
		for(ChoicesDone::const_iterator it = ctd.choiceHistory_.begin();
			it != ctd.choiceHistory_.end(); ++it)
		{
			os	<< "\t" << it->choice_.id() << ": " << (it->choice_.isPositive() ? '+' : '-')
				<< (it->toggled_ ? ", toggled" : "")
				<< (it->delegated_ ? ", delegated" : "")
				<< (it->core_ ? ", core" : "") << '\n';
		}
		return os;
	}
}
