/*	$Id: expander_strategies.cpp 1728 2005-05-06 08:08:53Z jgressma $
 *
 *  Copyright 2005 University of Potsdam, Germany
 * 
 *	This file is part of Platypus.
 *
 *  Platypus is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  Platypus is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with Platypus; if not, write to the Free Software
 *  Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 *
 */

#include <functional>
#include <algorithm>
#include <cassert>
#include <platypus/types/chooser.h>
#include <platypus/types/expander_strategies_callback.h>
#include <platypus/types/expander_strategies.h>

using namespace std;

namespace Platypus
{
	/////////////////////////////////////////////////////////////////////////
	// NativeChooser
	/////////////////////////////////////////////////////////////////////////
	NativeChooser::NativeChooser(ChooseCallback& chooser)
		:	chooser_(&chooser)
	{}
	Choice NativeChooser::choose()
	{
		return chooser_->strategyGetChoice();
	}
	void NativeChooser::setPolicy(ChoicePolicy& policy)
	{}
	

	/////////////////////////////////////////////////////////////////////////
	// ManualChooser
	/////////////////////////////////////////////////////////////////////////
	ManualChooser::ManualChooser(ChooseCallback& chooser)
		:	chooser_(&chooser)
		,	policy_(0)
	{}
	Choice ManualChooser::choose()
	{
		assert(policy_ && "Forgot to set choice policy in Expander!");
		const PossibleChoices pc(chooser_->possibleChoices());
		
		policy_->reinitialize(pc);
		return policy_->getChoice();
	}
	void ManualChooser::setPolicy(ChoicePolicy& policy)
	{
		policy_ = &policy;
	}

	/////////////////////////////////////////////////////////////////////////
	// NativeBacktracker
	/////////////////////////////////////////////////////////////////////////	
	NativeBacktracker::NativeBacktracker(BacktrackCallback& expander)
		:	expander_(&expander)
	{}
	void NativeBacktracker::expand(const Choice& choice)
	{
		static_cast<BacktrackCallback*>(expander_)->strategyExpand(choice);
	}
	void NativeBacktracker::backtrack(const Choice& choice)
	{
		static_cast<BacktrackCallback*>(expander_)->strategyBacktrackTo(choice);
	}
	void NativeBacktracker::reinitialize(const DelegatableChoice& dc)
	{
		static_cast<BacktrackCallback*>(expander_)->strategyReinitialize(dc);
	}


	/////////////////////////////////////////////////////////////////////////
	// ManualBacktracker
	/////////////////////////////////////////////////////////////////////////	
	namespace
	{
		struct FindById : public std::binary_function<Choice, Choice, bool>
		{
			inline bool operator()(const Choice& lhs, const Choice& rhs) const
			{
				return lhs.id() == rhs.id();
			}
		};
	}
	ManualBacktracker::ManualBacktracker(BacktrackCallback& expander, const DelegatableChoice& dc)
		:	expander_(&expander)
		,	stack_(dc)
	{}

	void ManualBacktracker::expand(const Choice& choice)
	{
		assert(find_if(stack_.begin(), stack_.end(), bind2nd(FindById(), choice)) == stack_.end());
		stack_.push_back(choice);
		static_cast<BacktrackCallback*>(expander_)->strategyExpand(choice);
	}
	void ManualBacktracker::backtrack(const Choice& choice)
	{
		assert(find_if(stack_.begin(), stack_.end(), bind2nd(FindById(), choice)) != stack_.end());

		while(stack_.back().id() != choice.id())
		{
			stack_.pop_back();
		}
		stack_.back().toggle();
		static_cast<BacktrackCallback*>(expander_)->strategyReinitialize(stack_);
	}
	void ManualBacktracker::reinitialize(const DelegatableChoice& dc)
	{
		stack_ = dc;
		static_cast<BacktrackCallback*>(expander_)->strategyReinitialize(dc);
	}
}
