/*	$Id: partial_assignment.cpp 1728 2005-05-06 08:08:53Z jgressma $
 *
 *  Copyright 2005 University of Potsdam, Germany
 * 
 *	This file is part of Platypus.
 *
 *  Platypus is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  Platypus is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with Platypus; if not, write to the Free Software
 *  Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 *
 */

#include <iostream>
#include <algorithm>
#include <iterator>
#include <cassert>
#include <functional>
#include <platypus/program_base.h>
#include <platypus/types/partial_assignment.h>

using namespace std;

namespace Platypus
{
	namespace
	{
		struct AtomSort : public binary_function<Atom, Atom, bool>
		{
			inline bool operator()(const Atom& lhs, const Atom& rhs) const
			{
				return lhs.id() < rhs.id();
			}
		};
	} 

	///////////////////////////////////////////////////////////
	// PartialAssignment::Proxy
	///////////////////////////////////////////////////////////
	PartialAssignment::Proxy::Proxy(const Atoms& atoms,
									const States& states, 
									char mine)
		:	atoms_(&atoms)
		,	states_(&states)
		,	mine_(mine)
	{}
	PartialAssignment::Proxy::const_iterator PartialAssignment::Proxy::begin() const
	{
		assert(atoms_->size() == states_->size());	
		return const_iterator(	atoms_->begin(), 
								states_->begin(), states_->end(),
								mine_);
	}
	PartialAssignment::Proxy::const_iterator PartialAssignment::Proxy::end() const
	{
		assert(atoms_->size() == states_->size());
		return const_iterator(	atoms_->end(), 
								states_->end(), states_->end(),
								mine_);
	}
	size_t PartialAssignment::Proxy::size() const
	{
		return static_cast<size_t>(
			count(states_->begin(), states_->end(), mine_));
	}
	bool PartialAssignment::Proxy::empty() const
	{
		return size() == 0;
	}

	///////////////////////////////////////////////////////////
	// PartialAssignment
	///////////////////////////////////////////////////////////
	const char PartialAssignment::UNKNOWN = 'U';
	const char PartialAssignment::POSITIVE = 'P';
	const char PartialAssignment::NEGATIVE = 'N';

	PartialAssignment::PartialAssignment(const ProgramInterface& program)
		:	states_(program.atoms().size(), UNKNOWN) // all unknown
		,	positive_(atoms_, states_, POSITIVE)
		,	negative_(atoms_, states_, NEGATIVE)
		,	unknown_(atoms_, states_, UNKNOWN)
	{
		atoms_.reserve(program.atoms().size());
		for(ProgramInterface::CollectionType::const_iterator it = program.atoms().begin();
			it != program.atoms().end(); ++it)
		{
			atoms_.push_back(Atom(*it, program));
		}
		sort(atoms_.begin(), atoms_.end(), AtomSort());
		assert(atoms_.size() == states_.size());
	}
	PartialAssignment::PartialAssignment(const PartialAssignment& pa)
		:	atoms_(pa.atoms_)
		,	states_(pa.states_)
		,	positive_(atoms_, states_, POSITIVE)
		,	negative_(atoms_, states_, NEGATIVE)
		,	unknown_(atoms_, states_, UNKNOWN)
	{}
	PartialAssignment& PartialAssignment::operator=(const PartialAssignment& pa)
	{
		PartialAssignment(pa).swap(*this);
		return *this;
	}

	void PartialAssignment::swap(PartialAssignment& other)
	{
		// Atoms should be the same throuout the entire program
		// Thus we only need to swap states
		//atoms_.swap(other.atoms_);
		assert(atoms_ == other.atoms_);
		states_.swap(other.states_);
	}
	// Program already makes the assumption that lparse generates linear
	// 1...MAX_ID atom ids. Use this assumption here for constant lookup
	void PartialAssignment::set(const Atom& atom, char value)
	{
		assert(value == UNKNOWN || value == POSITIVE || value == NEGATIVE);
		//const Atoms::iterator res = lower_bound(atoms_.begin(), atoms_.end(), atom, AtomSort());
		//assert(res != atoms_.end());
		const size_t index = atom.id() - 1; // Ids start at 1
		assert(index >= 0 && index < atoms_.size()); 
		
		states_[index] = value;
	}

	bool PartialAssignment::hasUnknownAtoms() const
	{
		return find(states_.begin(), states_.end(), UNKNOWN) != states_.end();
	}

	void PartialAssignment::reset()
	{
		states_.clear();
		states_.resize(atoms_.size(), UNKNOWN);		
	}
	bool PartialAssignment::equals(const PartialAssignment& other) const
	{
		return atoms_ == other.atoms_ && states_ == other.states_;
	}
	std::ostream& operator<<(std::ostream& os, const PartialAssignment& pa)
	{
		os << "Positive: ";
		copy(pa.positiveAtoms().begin(), pa.positiveAtoms().end(), ostream_iterator<Atom>(os, " "));
		os << "\nNegative: ";
		copy(pa.negativeAtoms().begin(), pa.negativeAtoms().end(), ostream_iterator<Atom>(os, " "));
		os << "\nUnknown: ";
		copy(pa.unknownAtoms().begin(), pa.unknownAtoms().end(), ostream_iterator<Atom>(os, " "));
		os << "\n";
		return os;
	}
};
