/*	$Id: partial_assignment.h 1728 2005-05-06 08:08:53Z jgressma $
 *
 *  Copyright 2005 University of Potsdam, Germany
 * 
 *	This file is part of Platypus.
 *
 *  Platypus is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  Platypus is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with Platypus; if not, write to the Free Software
 *  Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 *
 */

#ifndef PLATYPUS_PARTIAL_ASSIGNMENT_IMPL_H
#define PLATYPUS_PARTIAL_ASSIGNMENT_IMPL_H

#include <iosfwd>
#include <vector>
#include <iterator>
#include <platypus/types/atom.h>

namespace Platypus
{
	template<class T, class TIterator>
	class HoleIterator : public std::iterator<std::forward_iterator_tag, T> 
	{
		typedef std::iterator<std::forward_iterator_tag, T> Base;
	public:
		typedef typename Base::pointer pointer;
		typedef typename Base::reference reference;
		
		HoleIterator(TIterator beg, TIterator end, const T& needle = T())
			:	current_(beg)
			,	end_(end)
			,	distance_(0)
			,	needle_(needle)
		{
			moveForward();
		}
		inline pointer operator->()
		{
			return &*current_;
		}
		inline reference operator*()
		{
			return *current_;
		}
		inline HoleIterator& operator++()
		{
			inc();
			return *this;
		}
		inline HoleIterator operator++(int)
		{
			const HoleIterator temp(*this);
			inc();
			return temp;
		}
		inline bool equals(const HoleIterator& other) const
		{
			return needle_ == other.needle_ && current_ == other.current_ && end_ == other.end_;
		}
		inline size_t distance() const { return distance_; }
	private:
		HoleIterator();
		inline void inc()
		{
			++current_;
			++distance_;
			moveForward();
		}
		inline void moveForward()
		{
			while(current_ != end_ && *current_ != needle_)
				++current_, ++distance_;
		}
	private:
		TIterator current_, end_;
		size_t distance_;
		T needle_;
	};

	template<class T, class TIterator>
	inline bool operator==(const HoleIterator<T, TIterator>& lhs, const HoleIterator<T, TIterator>& rhs)
	{
		return lhs.equals(rhs);
	}

	template<class T, class TIterator>
	inline bool operator!=(const HoleIterator<T, TIterator>& lhs, const HoleIterator<T, TIterator>& rhs)
	{
		return !(lhs == rhs);
	}

	class PartialAssignment
	{
		typedef std::vector<Atom> Atoms;
		typedef std::vector<char> States;
		typedef std::iterator<std::forward_iterator_tag, const Atom> IteratorBase;
	public:
		class Iterator : public IteratorBase
		{
			typedef IteratorBase Base;
			typedef Atoms::const_iterator AtomIterator;
			typedef HoleIterator<const char, States::const_iterator> StateIterator;
		public:
			typedef Base::pointer pointer;
			typedef Base::reference reference;	
			
			Iterator(	Atoms::const_iterator atomBegin,
						States::const_iterator stateBegin, States::const_iterator stateEnd, 
						char mine)
				:	begin_(atomBegin)
				,	currentState_(stateBegin, stateEnd, mine)
				,	stateEnd_(stateEnd, stateEnd, mine)
			{}
			inline pointer operator->()
			{
				return &*current();
			}
			inline reference operator*()
			{
				return *current();
			}
			inline Iterator& operator++()
			{
				++currentState_;
				return *this;
			}
			inline Iterator operator++(int)
			{
				const Iterator temp(*this);
				++currentState_;
				return temp;
			}
			inline bool equals(const Iterator& other) const
			{
				return	currentState_ == other.currentState_ && 
						stateEnd_ == other.stateEnd_; 
			}
		private:
			inline Atoms::const_iterator current()
			{
				return (begin_ + currentState_.distance());
			}
			Iterator();
		private:
			AtomIterator begin_;
			StateIterator currentState_, stateEnd_;
		};
	public:
		// Proxy class that mocks a very simple, readonly container
		class Proxy
		{
		public:
			typedef Iterator const_iterator;
			typedef Atom value_type;
			Proxy(const Atoms& atoms, const States& states, char mine);
			const_iterator begin() const;
			const_iterator end() const;
			size_t size() const;
			bool empty() const;
		private:
			Proxy();
		private:
			const Atoms* atoms_;
			const States* states_;
			char mine_;
		};
		typedef Proxy CollectionType;
		// initializes the partial model to all 
		// atoms unknown
		PartialAssignment(const ProgramInterface& program);	
		PartialAssignment(const PartialAssignment& pa);	
		PartialAssignment& operator=(const PartialAssignment& pa);	
		inline void setTrue(const Atom& atom)
		{
			set(atom, POSITIVE);
		}
		inline void setFalse(const Atom& atom)
		{
			set(atom, NEGATIVE);
		}
		inline void setUnknown(const Atom& atom)
		{
			set(atom, UNKNOWN);
		}
		
		// returns true, if there are unknown atoms left in the partial model, false otherwise
		bool hasUnknownAtoms() const;			
		inline const Proxy& positiveAtoms() const
		{
			return positive_;
		}
		inline const Proxy& negativeAtoms() const
		{
			return negative_;
		}
		inline const Proxy& unknownAtoms() const
		{
			return unknown_;
		}
		void reset();
		bool equals(const PartialAssignment& other) const;
		void swap(PartialAssignment& other);
	private:
		PartialAssignment();
		void set(const Atom& atom, char value);
	private:
		Atoms atoms_;
		States states_;
		const Proxy positive_, negative_, unknown_;

		static const char UNKNOWN;
		static const char POSITIVE;
		static const char NEGATIVE;
	};

	inline bool operator==(const PartialAssignment::Iterator& lhs, const PartialAssignment::Iterator& rhs)
	{
		return lhs.equals(rhs);
	}
	inline bool operator!=(const PartialAssignment::Iterator& lhs, const PartialAssignment::Iterator& rhs)
	{
		return !(lhs == rhs);
	}
	
	inline bool operator==(const PartialAssignment& lhs, const PartialAssignment& rhs)
	{
		return lhs.equals(rhs);
	}
	inline bool operator!=(const PartialAssignment& lhs, const PartialAssignment& rhs)
	{
		return !(lhs == rhs);
	}

	std::ostream& operator<<(std::ostream& os, const PartialAssignment& pa);
};

#endif
