/*	$Id: program.cpp 1728 2005-05-06 08:08:53Z jgressma $
 *
 *  Copyright 2005 University of Potsdam, Germany
 * 
 *	This file is part of Platypus.
 *
 *  Platypus is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  Platypus is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with Platypus; if not, write to the Free Software
 *  Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 *
 */

#include <algorithm>
#include <cassert>
#include <sstream>
#include <platypus/types/program.h>
#include <platypus/types/parser.h>
#include <platypus/types/name_generator.h>
#include <platypus/types/atom.h>
#include <platypus/types/exceptions.h>

using namespace std;

namespace Platypus
{
	namespace
	{
		class SimpleProgram : public LParserCallback
		{
		public:
			typedef std::vector<AtomId> Atoms;
			SimpleProgram()
				:	rules_(0)
			{}
			void addAtom(int atom)
			{
				if(generator_.addAtom(atom))
					atoms_.push_back(atom);
			}
			void setName(int atom, const std::string& name)
			{
				generator_.setName(atom, name);
			}
			// This is called by LParser before the parsing loop is left for good
			void done() 
			{
				generator_.generate();
			}
		
			void addRuleLine(const std::string& line)
			{
				++rules_;
				addLine(ruleLines_, line);
			}
			
			void addBodyPlusLine(const std::string& line)
			{
				addLine(bodyPlusLines_, line);
			}
			void addBodyMinusLine(const std::string& line)
			{
				addLine(bodyMinusLines_, line);
			}
			void addNoModels(unsigned no)
			{
				ostringstream s;
				s << no;
				noModels_ = s.str();
			}
			Atoms& atoms() { return atoms_; }
			size_t rules() const { return rules_; }
			const NameGenerator& generator() const { return generator_; }
			const string& ruleLines() const { return ruleLines_; }
			const string& bodyPlusLines() const { return bodyPlusLines_; }
			const string& bodyMinusLines() const { return bodyMinusLines_; }
			const string& noModels() const { return noModels_; }
		private:
			static void addLine(string& s, const string& line)
			{
				s += line;
			}
		private:
			Atoms atoms_;
			NameGenerator generator_;
			string ruleLines_;
			string bodyPlusLines_;
			string bodyMinusLines_;
			string noModels_;
			size_t rules_;
		};

		string generateLparse(const SimpleProgram& program, const Program::Id2Name& ids_)
		{
			// add rules
			string lparse_ = program.ruleLines();
			lparse_ += "0\n";

			// add atoms and their names
			for(unsigned i = 1; i < ids_.size(); ++i)
			{
				assert(!ids_[i].empty());
				ostringstream s;
				s << i << ' ' << ids_[i] << '\n';
				lparse_ += s.str();
			}
			// add B+
			lparse_ += "0\nB+\n";
			lparse_ += program.bodyPlusLines();

			// add B-
			lparse_ += "0\nB-\n";
			lparse_ += program.bodyMinusLines();

			// add no models
			lparse_ += "0\n";
			lparse_ += program.noModels();
			lparse_ += '\n';
			return lparse_;
		}
	}
	
	
	void Program::setupMappings(std::istream& is)
	{
		SimpleProgram simpleprogram;
		try
		{
			LParser(is, &simpleprogram).parse();
		}
		catch(const ::ParseError& e)
		{
			throw Platypus::ParseError(e.what());
		}

		ids_.clear(), names_.clear();
		atoms_.swap(simpleprogram.atoms());
		rules_ = simpleprogram.rules();
		if(!atoms_.empty())
		{
			// assume there is a maximum id -> reasonable AND
			// all other ids are sequential, that the ids are in
			// range 1...MAX_ID
			const AtomId maxId = *max_element(atoms_.begin(), atoms_.end());
			ids_.resize(maxId+1);
		}
		
		for(CollectionType::const_iterator it = atoms_.begin();
			it != atoms_.end();
			++it)
		{
			const std::string& name = simpleprogram.generator().getName(*it);
			// we assume lparse generated linear ids
			assert(static_cast<AtomId>(*it) < ids_.size());
			ids_[static_cast<AtomId>(*it)] = name;
			names_.insert(Name2Id::value_type(name, static_cast<AtomId>(*it)));
		}

		// regenerate lparse output now that each atom has a name
		lparse_ = generateLparse(simpleprogram, ids_);
	}

	Program::Program()
		:	rules_(0)
	{
		setup(EMPTY_PROGRAM);
	}
	
	void Program::setup(const std::string& input)
	{
		istringstream is(input);
		setupMappings(is);
	}
	void Program::setup(std::istream& input)
	{
		setupMappings(input);
	}
	
	const std::string& Program::idToName(AtomId id) const
	{
		if(id == 0 || id >= ids_.size())
		{
			ostringstream s;
			s << "The atom with id: " << id << " does not exist in the logic program!";
			throw UnknownAtom(s.str());
		}
		return ids_[id];
	}
	
	// returns the id associated with an atom name
	// NOTE: 	Names MUST be returned for all atoms, wether
	// 			they have a "name" (like a, p(v4), ...) or not.	
	AtomId Program::nameToId(const std::string& name) const
	{
		Name2Id::const_iterator res = names_.find(name);
		if(res == names_.end())
			throw UnknownAtom("Atom: " + name + " does not exist in the logic program!");
		
		return res->second;
	}
	
	// returns the collection of atoms of the logic program
	const Program::CollectionType& Program::atoms() const
	{
		return atoms_;
	}
	
	// Returns the definite number of atoms in the logic program.
	// For instance the logic program
	// d :- {c} 1.
	// c.
	//
	// after being preprocessed by lparse contains FOUR atoms.
	size_t Program::numberOfAtoms() const
	{
		return atoms_.size();
	}
	size_t Program::numberOfRules() const
	{
		return rules_;
	}
	
	// returns true if the atom had a name in the original (textual program), false otherwise.
	bool Program::hasDisplayName(AtomId id) const
	{
		return idToName(id)[0] != '_';
	}
	
	// returns an lparse representation of the preprocessed program.
	const std::string& Program::lparse() const
	{
		return lparse_;
	}
	ProgramInterface* Program::create()
	{
		return new Program;
	}

}
