/*  Copyright (c) January 2005 Jean Gressmann (jsg@rz.uni-potsdam.de)
 *
 *  This is free software; you can redistribute it and/or modify
 *	it under the terms of the GNU General Public License as published by
 *	the Free Software Foundation; either version 2 of the License, or
 *	(at your option) any later version. 
 * 
 *	This file is distributed in the hope that it will be useful,
 *	but WITHOUT ANY WARRANTY; without even the implied warranty of
 *	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *	GNU General Public License for more details.
 *
 *	You should have received a copy of the GNU General Public License
 *	along with this file; if not, write to the Free Software
 *	Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#ifndef X86_32_LINUX_GCC_H
#define X86_32_LINUX_GCC_H

PT_NAMESPACE_BEGIN
typedef signed char int8;
typedef unsigned char uint8;
typedef short int16;
typedef unsigned short uint16;
typedef int int32;
typedef unsigned int uint32;
typedef long long int64;
typedef unsigned long long uint64;


// Most of these functions are due to the Linux kernel.

inline void pt_barrier()
{
	__asm__ __volatile__("mfence\n\t": : :"memory");
}

inline int32 pt_atomic_add(volatile int32* counter, int32 value)
{
	const int32 res = value;
	__asm__ __volatile__(
		"lock xaddl %0, %1	\n\t"
		:"=r"(value)
		:"m"(*counter), "0"(value)
	);
	return res + value;
}

inline int32 pt_atomic_sub(volatile int32* counter, int32 value)
{
	return pt_atomic_add(counter, -value);
}

inline int32 pt_atomic_inc(volatile int32* counter)
{
	return pt_atomic_add(counter, 1);
}

inline int32 pt_atomic_dec(volatile int32* counter)
{
	return pt_atomic_add(counter, -1);
}

/*
 * Atomic compare and exchange.  Compare OLD with MEM, if identical,
 * store NEW in MEM.  Return the initial value in MEM.  Success is
 * indicated by comparing RETURN with OLD.
 */
inline int32 pt_atomic_cas_return_memory(volatile int32* inMemory, int32 newValue, int32 oldValue)
{
	int32 prev;	
	__asm__ __volatile__(
		"lock cmpxchgl %1, %2	\n\t"
		:	"=a"(prev)
	    :	"q"(newValue), "m"(*inMemory), "0"(oldValue)
	    :	"memory"
	);
	return prev;
}

inline bool pt_atomic_cas(volatile int32* mem, int32 nv, int32 ov)
{
	return pt_atomic_cas_return_memory(mem, nv, ov) == ov;
}

inline int32 pt_atomic_set(volatile int32* inMemory, int32 newValue)
{
	// This always asserts a lock (Pentium Manual)
	__asm__ __volatile__(
		"xchgl %0, %1	\n\t"
		:	"=r"(newValue)
		:	"m"(*inMemory), "0"(newValue)
		:	"memory"
	);
	return newValue;
}

// 64 bit
inline bool pt_atomic_cas(volatile int64* inMemory, int64 newValue, int64 oldValue)
{
/* (from Pentium Manual)
Compares the 64-bit value in EDX:EAX with the operand (destination operand). If the values
are equal, the 64-bit value in ECX:EBX is stored in the destination operand. Otherwise, the
value in the destination operand is loaded into EDX:EAX. The destination operand is an 8-byte
memory location. For the EDX:EAX and ECX:EBX register pairs, EDX and ECX contain the
high-order 32 bits and EAX and EBX contain the low-order 32 bits of a 64-bit value.
*/
	const int32 oldLow = static_cast<int32>(oldValue);
	const int32 oldHigh = static_cast<int32>(oldValue >> 32);
	const int32 newLow = static_cast<int32>(newValue);
	const int32 newHigh = static_cast<int32>(newValue >> 32);
	bool yes;
	__asm__ __volatile__(
		"lock cmpxchg8b %5	\n\t"
		"sete %0			\n\t"
		:	"=q"(yes)
		:	"a"(oldLow), "d"(oldHigh), "b"(newLow), "c"(newHigh), "m"(*inMemory)
	    :	"cc", "memory" // no eax, edx here, gas doesn't like it.
	);
	return yes;
}

inline int32 pt_rol(int32 value, unsigned char shift = 1)
{
	__asm__ (
		"roll %1, %0	\n\t"
		:"=r"(value)
		:"c"(shift), "0"(value)
		:"cc"
	);
	return value;
}
inline int32 pt_ror(int32 value, unsigned char shift = 1)
{
	__asm__ (
		"rorl %1, %0;	\n\t"
		:"=r"(value)
		:"c"(shift), "0"(value)
		:"cc"
	);
	return value;
}

PT_NAMESPACE_END

#include <portablethreads/arch/arch-common.h>
#include <portablethreads/arch/native-pointer-cas.h>

#endif
