/*  Copyright (c) January 2005 Jean Gressmann (jsg@rz.uni-potsdam.de)
 *
 *  This is free software; you can redistribute it and/or modify
 *	it under the terms of the GNU General Public License as published by
 *	the Free Software Foundation; either version 2 of the License, or
 *	(at your option) any later version. 
 * 
 *	This file is distributed in the hope that it will be useful,
 *	but WITHOUT ANY WARRANTY; without even the implied warranty of
 *	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *	GNU General Public License for more details.
 *
 *	You should have received a copy of the GNU General Public License
 *	along with this file; if not, write to the Free Software
 *	Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#ifndef PT_LOCK_FREE_H
#define PT_LOCK_FREE_H
#include <portablethreads/config.h>
#include <portablethreads/atomic_number.h>
#include <cassert>
#include <vector>
#include <memory>

#ifdef _MSC_VER
// This warning also appears when a 32-bit pointer gets cast into a 4 byte int
#	pragma warning(disable:4311) // A 64-bit pointer was truncated to a 32-bit int or 32-bit long.
// This warning also appears when a 32-bit int gets cast into a 32-bit pointer
#	pragma warning(disable:4312) // You attempted to assign a 32-bit value to a 64-bit integer. For example, casting a 32-bit int or 32-bit long to a 64-bit pointer
#endif

PT_NAMESPACE_BEGIN

inline bool operator==(const PTPointerCAS::token_t& lhs, const PTPointerCAS::token_t& rhs)
{
	return lhs.value() == rhs.value() && lhs.count() == rhs.count();
}

inline bool operator!=(const PTPointerCAS::token_t& lhs, const PTPointerCAS::token_t& rhs)
{
	return !(lhs == rhs);
}

class PSpinlock
{
public:
	PSpinlock()
		:	spin_(0)
	{}
	inline void lock()
	{
		bool done;
		do { done = pt_atomic_cas(&spin_, 1, 0); } while(!done);
	}
	inline void unlock()
	{
		pt_atomic_cas(&spin_, 0, 1);
	}
private:
	PSpinlock(const PSpinlock&);
	PSpinlock& operator=(const PSpinlock&);
private:
	volatile pt_int_type spin_;
};

namespace PTPrivate
{
	template<typename T>
	class SingleLinkedNode
	{
	public:
		typedef T value_type;
		typedef T& reference;
		typedef const T& const_reference;
		~SingleLinkedNode()
		{
			destruct();
		}
		SingleLinkedNode(bool pod = false)
			:	hasObject_(false)	
			,	next_(0)
			,	pod_(pod)
		{}
		inline void construct(const T& t) volatile
		{
			if(!pod_)
			{
				assert(!hasObject_);
			}
			const_cast<PTPointerCAS&>(next_).assign(0);
			new (const_cast<char*>(raw_)) T(t);
			if(!pod_)
			{
				hasObject_ = true;
			}
			
		}
		inline void destruct() volatile
		{
			// make sure only one thread ever destroys the object
			if(!pod_)
			{
				if(hasObject_)
				{
					hasObject_ = false;
					reinterpret_cast<T*>(const_cast<char*>(raw_))->~T();
				}
			}
		}
		inline const_reference data() volatile const 
		{
			if(!pod_)
			{
				assert(hasObject_);
			}
			return *reinterpret_cast<const T*>(const_cast<char*>(raw_));
		}
		inline PTPointerCAS::token_t next() const { return next_.get(); }
		inline void next(PTPointerCAS::int_t n) { next_.assign(n); }
		inline bool hasObject() const volatile { return hasObject_; }
		inline bool cas(PTPointerCAS::int_t nv, const PTPointerCAS::token_t ov)
		{
			return next_.cas(nv, ov);
		}
		inline bool pod() const { return pod_; }
	private:
		SingleLinkedNode(const SingleLinkedNode&);
		SingleLinkedNode& operator=(const SingleLinkedNode&);
	private:	
		volatile char raw_[sizeof(T)];
		volatile bool hasObject_;
		PTPointerCAS next_;
		const bool pod_;
	};

	template<typename T>
	SingleLinkedNode<T>* recycle(PTPointerCAS& head, bool pod = false)
	{
		typedef SingleLinkedNode<T> Node;

		PTPointerCAS::token_t current = head.get();
		while(current.value())
		{
			if(head.cas(((Node*)current.value())->next().value(), current))
			{
				return (Node*)current.value();
			}
			current = head.get();
		}
		return new SingleLinkedNode<T>(pod);
	}

	template<typename T>
	void recycle(PTPointerCAS& head, SingleLinkedNode<T>* node)
	{
		PTPointerCAS::token_t h;
		do
		{
			h = head.get();
			node->next(h.value());
		}
		while(!head.cas((PTPointerCAS::int_t)node, h));
	}
}


// A lock-free stack (Treiber) suitable for concurrent access without 
// synchronisation.
// Memory management:
// PLockFreeStack allows its user to make time-space trade offs.
// By passing "true" to PLockFreeStack's c'tor internal memory
// management is disabled. This results in every push operation
// causing a heap allocation and every pop operation in freeing
// the memory previously allocated. While this solution is strictly
// lock-free, the drawback is there may exist several unused nodes.
// Passing "false", the default setting, PLockFreeStack reuses 
// allocated memory the best it can. In this mode the stack is
// lock-free when inserting, however, a spinlock must be obtained
// and released in order to safely delete the freed node.
template<class T>
class PLockFreeStack
{
	typedef PTPrivate::SingleLinkedNode<T> Node;
public:
	typedef T value_type;
	typedef T& reference;
	typedef const T& const_reference;
	~PLockFreeStack()
	{
		Node* current = ((Node*)head_.get().value());
		while(current)
		{
			Node* temp = current;
			current = ((Node*)current->next().value());
			delete temp;
		}
		current = ((Node*)storeHead_.get().value());
		while(current)
		{
			Node* temp = current;
			current = ((Node*)current->next().value());
			delete temp;
		}
	}
	PLockFreeStack(bool lean = false)
		:	head_(0)
		,	storeHead_(0)
		,	lean_(lean)
	{}
	
	// Push one element on the stack.
	// This method may be safely invoked simultanously from
	// multiple threads.
	void push(const_reference t)
	{
		std::auto_ptr<Node> safe(recycle());
		safe->construct(t);
		
		PTPointerCAS::token_t h;
		do
		{
			h = head_.get();
			safe->next(h.value());
		}
		while(!head_.cas((PTPointerCAS::int_t)safe.get(), h));
		safe.release();
	}
	
	// Try to pop the topmost element from the stack. If the stack
	// is not empty, true is returned and the reference passed
	// is assigned the value of the topmost element. Otherwise
	// the method returns false and leaves the reference unchanged.
	bool pop(reference t)
	{
		if(lean_)
			headGuard_.lock();
		bool removed = false;
		PTPointerCAS::token_t current = head_.get();
		while(current.value())
		{
			if(head_.cas(((Node*)current.value())->next().value(), current))
			{
				removed = true;
				break;
			}
			current = head_.get();
		}
		if(lean_)
			headGuard_.unlock();
		if(removed)
		{
			t = ((Node*)current.value())->data();
			((Node*)current.value())->destruct();
			recycle(((Node*)current.value()));
		}
		return removed;
	}
private:	
	inline Node* recycle()
	{
		return PTPrivate::recycle<T>(storeHead_);
	}
	inline void recycle(Node* node)
	{
		assert(node);
		assert(!node->hasObject());
		if(lean_)
		{
			delete node;
		}
		else
		{
			PTPrivate::recycle<T>(storeHead_, node);
		}
	}
	
private:
	PTPointerCAS head_, storeHead_;
	PSpinlock headGuard_;
	const bool lean_;
};


// A lock-free queue class based on Maged M. Michael's paper (http://citeseer.ist.psu.edu/michael96simple.html).
// I added support for node recycling.
template<class T>
class PLockFreeQueue
{
	typedef PTPrivate::SingleLinkedNode<T> TNode;
	typedef PTPrivate::SingleLinkedNode<TNode*> Node;	
public:
	typedef T value_type;
	typedef T& reference;
	typedef const T& const_reference;
	~PLockFreeQueue()
	{
		const PTPointerCAS::token_t h = head_.get();
		Node* c = (Node*)h.value();
		while(c)
		{
			if(c != (Node*)h.value())
			{
				delete (TNode*)c->data();
			} 
			Node* temp = c;
			c = ((Node*)c->next().value());
			delete temp;
		}
		Node* n = (Node*)storeNode_.get().value();
		while(n)
		{
			Node* temp = n;
			n = ((Node*)n->next().value());
			delete temp;
		}
		TNode* t = (TNode*)storeT_.get().value();
		while(t)
		{
			TNode* temp = t;
			t = ((TNode*)t->next().value());
			delete temp;
		}
	}
	PLockFreeQueue(bool lean = false)
		:	head_((PTPointerCAS::int_t)new Node(true))
		,	tail_(head_.get().value())
		,	storeT_(0)
		,	storeNode_(0)
	{}
	// Add the element passed to the end of the queue.
	void pushBack(const_reference value)
	{
		std::auto_ptr<Node> safe(recycleNode());
		{
			std::auto_ptr<TNode> tsafe(recycleT());
			tsafe->construct(value);
			safe->construct(tsafe.release());
		}
		
		PTPointerCAS::token_t t, n;
		while(true)
		{
			t = tail_.get();
			n = ((Node*)t.value())->next();
			if(t == tail_.get()) // diese Zeile ist wichtig!
			{
				if(n.value() == 0)
				{
					if(((Node*)t.value())->cas((PTPointerCAS::int_t)safe.get(), n))
						break;
				}
				else
				{
					tail_.cas(n.value(), t);
				}
			}
		}
		tail_.cas((PTPointerCAS::int_t)safe.get(), t);
		safe.release();
	}

	
	// Try to remove the first element. Fail if the queue is empty.
	// Otherwise assign the first element to the reference and remove
	// that element from the queue.
	bool popFront(reference value)
	{
		TNode* tnode = 0;
		PTPointerCAS::token_t h, t, n;
		while(true)
		{
			h = head_.get();
			t = tail_.get();
			n = ((Node*)h.value())->next();
			if(h == head_.get()) // diese Zeile ist wichtig!
			{
				if(h.value() == t.value())
				{
					if(n.value() == 0)
						return false;
					tail_.cas(n.value(), t);
				}
				else
				{
					tnode = ((Node*)n.value())->data();
					if(head_.cas(n.value(), h))
						break;
				}
			}
		}
		recycleNode((Node*)h.value());

		assert(tnode);
		value = tnode->data();
		tnode->destruct();
		recycleT(tnode);
		
		return true;
	}
private:
	inline Node* recycleNode()
	{
		return PTPrivate::recycle<TNode*>(storeNode_, true);
	}
	inline void recycleNode(Node* node)
	{
		assert(node);
		assert(node->pod());
		PTPrivate::recycle<TNode*>(storeNode_, node);
	}
	inline TNode* recycleT()
	{
		return PTPrivate::recycle<T>(storeT_);
	}
	inline void recycleT(TNode* node)
	{
		assert(node);
		assert(!node->hasObject());
		PTPrivate::recycle<T>(storeT_, node);
	}
private:
	PTPointerCAS head_, tail_;
	PTPointerCAS storeT_, storeNode_;
};

template<class T, class A = std::allocator<T> >
class PRingBuffer
{
	typedef std::vector<T, A> BufferType;
	typedef typename BufferType::size_type size_type;
public:
	PRingBuffer(size_t size, const T& t = T())
		:	buffer_(size, t)
		,	size_(0)
		,	inserter_(0)
		,	remover_(0)
	{
		assert(size);
	}
	bool popFront(T& item)
	{
		if(--size_ == 0)
		{
			++size_;
			return false;
		}
		const size_type index = (remover_++) % buffer_.size();
		item = buffer_[index];
		
		return true;
	}
	bool pushBack(const T& t)
	{
		if(++size_ > (PAtomicNumber::int_type)buffer_.size())
		{
			--size_;
			return false;
		}
		const size_type index = (inserter_++) % buffer_.size();
		buffer_[index] = t;
		return true;
	}
	inline size_t capacity() const { return buffer_.size(); }
	inline size_t size() const { return size_.get(); }
private:	
	PRingBuffer();
private:
	BufferType buffer_;
	PAtomicNumber size_;
	volatile size_type inserter_, remover_;
};

PT_NAMESPACE_END

#ifdef _MSC_VER
#	pragma warning(disable:4311)
#	pragma warning(disable:4312)
#endif

#endif

