/*  Copyright (c) January 2005 Jean Gressmann (jsg@rz.uni-potsdam.de)
 *
 *  This is free software; you can redistribute it and/or modify
 *	it under the terms of the GNU General Public License as published by
 *	the Free Software Foundation; either version 2 of the License, or
 *	(at your option) any later version. 
 * 
 *	This file is distributed in the hope that it will be useful,
 *	but WITHOUT ANY WARRANTY; without even the implied warranty of
 *	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *	GNU General Public License for more details.
 *
 *	You should have received a copy of the GNU General Public License
 *	along with this file; if not, write to the Free Software
 *	Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#ifndef PT_MESSAGE_QUEUE_H
#define PT_MESSAGE_QUEUE_H
#include <portablethreads/config.h>
#include <portablethreads/condition.h>
#include <portablethreads/mutex.h>
#include <deque>
#include <cassert>

PT_NAMESPACE_BEGIN

// PMessageQueue is a blocking and waitable message queue implementation. It's access is always
// synchronized, that is there is always at most one thread accessing the queue at all times.
template<typename T, typename LT = PMutex, typename C = PCondition, typename A = std::allocator<T> > 
class PMessageQueue
{
    typedef std::deque<T, A> Queue; 
public:
	typedef typename std::deque<T>::size_type size_type;
	typedef LT LockType;
	typedef C ConditionType;
private:
	typedef typename Queue::iterator QueueIt;
	typedef PMutexHelper<LockType> LockGuard;

 	Queue queue_;
	mutable LockType lock_;	
	mutable ConditionType cond_;
	// MessageQueues cannot be copied or assigned b/c
	// neither mutexes nor conditions can.
	PMessageQueue(const PMessageQueue&);
	PMessageQueue& operator=(const PMessageQueue&);
public:
	typedef T value_type;
	typedef T& reference;
	typedef const T& const_reference;
	PMessageQueue() 
	{}
	
	// Adds a message to the end of the message queue
	// and wakes up waiting threads.
	// Adding a message is guaranteed to be an exclusive operation
	void pushBack(const T& message)
	{
  		const LockGuard dynUnlock(lock_);  		
		const_cast<Queue&>(queue_).push_back(message);
		cond_.signal();	
	}

	// same thing as above but adds to the front of the queue
	void pushFront(const T& message)
	{
  		const LockGuard dynUnlock(lock_);  		
		const_cast<Queue&>(queue_).push_front(message);
		cond_.signal();		
	}
	
	// Remove foremost message in the queue. If the queue is
	// empty the parameter message is left unchanged.
	// Removing a message is guaranteed to be an exclusive operation
	bool popFront(T& message)
	{
		const LockGuard dynUnlock(lock_);
		if(const_cast<Queue&>(queue_).empty()) 
			return false;
  		
    	message = const_cast<Queue&>(queue_).front();
    	const_cast<Queue&>(queue_).pop_front();
		if(!const_cast<Queue&>(queue_).empty())
			cond_.signal();
    	return true;		
	}

	// same thing as above, but message is taken from the
	// end of the queue
	bool popBack(T& message)
	{
		const LockGuard dynUnlock(lock_);
		if(const_cast<Queue&>(queue_).empty()) 
			return false;
  		
    	message = const_cast<Queue&>(queue_).back();
    	const_cast<Queue&>(queue_).pop_back();
		if(!const_cast<Queue&>(queue_).empty())
			cond_.signal();
    	return true;		
	}
	
	// Check if the queue is empty
	// This is guaranteed to be an exclusive operation
	inline bool empty() const
	{
		const LockGuard dynUnlock(lock_);
		return const_cast<Queue&>(queue_).empty();				
	}

	// Check queues size
	// This is guaranteed to be an exclusive operation
	inline size_type size() const
	{
		const LockGuard dynUnlock(lock_);
		return const_cast<Queue&>(queue_).size();				
	}

	// wait blocking for a message to arrive
	void waitForMessage() const
	{
		cond_.wait();
	}

	bool waitForMessage(unsigned long milliseconds) const
	{
		return cond_.wait(milliseconds);
	}
};

// This class enables you to add a MessageQueue to 
// a PWait object. WaitForMessage queries the MessageQueues
// empty-state without blocking.
template<class T>
class WaitForMessage : public PWaitable
{
	T& queue_;
public:
	WaitForMessage(T& q)
		:	queue_(q)
	{}
	bool queryState()
	{
		return !queue_.empty();
	}
};

// Does the same as the above exept it wait's
// for the MessageQueue to become empty.
template<class T>
class WaitForEmptyQueue : public PWaitable
{
	T& queue_;
public:
	WaitForEmptyQueue(T& q)
		:	queue_(q)
	{}
	bool queryState()
	{
		return queue_.empty();
	}
};

PT_NAMESPACE_END
#endif
