/*  Copyright (c) January 2005 Jean Gressmann (jsg@rz.uni-potsdam.de)
 *
 *  This is free software; you can redistribute it and/or modify
 *	it under the terms of the GNU General Public License as published by
 *	the Free Software Foundation; either version 2 of the License, or
 *	(at your option) any later version. 
 * 
 *	This file is distributed in the hope that it will be useful,
 *	but WITHOUT ANY WARRANTY; without even the implied warranty of
 *	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *	GNU General Public License for more details.
 *
 *	You should have received a copy of the GNU General Public License
 *	along with this file; if not, write to the Free Software
 *	Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#ifndef PT_THREAD_H
#define PT_THREAD_H
#include <portablethreads/config.h>
#ifdef WIN32
#	include <portablethreads/win32/thread.h>
#endif
#ifdef UNIX
#	include <portablethreads/unix/thread.h>
#endif
#include <portablethreads/utility.h>
#include <portablethreads/mutex.h>
#include <portablethreads/condition.h>
#include <portablethreads/semaphore.h>
#include <portablethreads/lock_free.h>
#include <portablethreads/smartpointer.h>

#include <string>
#include <vector>
#include <stdexcept>
#include <algorithm>

PT_NAMESPACE_BEGIN
// This exception gets thrown by PThreadObjectSemantic
// if a thread could not be created. This shouldn't ever
// happen but it is better to notify the user if it does.
class CouldNotCreateThread : public std::runtime_error
{
public:
	CouldNotCreateThread(const std::string& what = "[PThreadObjectSemantic] A thread could NOT be created")
		:	std::runtime_error(what)
	{}
};

// Policy class for PThreadObjectSemantic
// There are currently three different policies:
// 1.	RunOnce -> invokes loopImpl() once
// 2.	RunN -> invokes loopImpl() N times
// 3.	RunForever -> repeatedly invoked loopImpl() 
//		util the thread gets terminated
// 
// If you want to provide your own policies, create
// a class that has three public methods:
// 1.	void reset(); -> resets the policy to initial state
// 2.	bool next(); -> is evaluated in each iteration. This
//		method should return true for as long as you want the
//		thread to run
// 3.	void shutdown(); -> must make next() return false
template<unsigned int N>
class RunN
{
	unsigned int count_;
public:
	RunN()
		:	count_(0)
	{}
	inline bool next()
	{
		return count_++ < N;
	}
	inline void reset()
	{
		count_ = 0;
	}
	inline void shutdown()
	{
		count_ = N;
	}
	
};

typedef RunN<1> RunOnce;

class RunForever
{	
	bool keepRunning_;
public:	
	RunForever()
		:	keepRunning_(true)
	{}
	inline bool next() { return keepRunning_; }	
	inline void reset() { keepRunning_ = true; }
	inline void shutdown() { keepRunning_ = false; }	
};

// This class allows you to plug a normal class into a thread.
// The plugged class needs to provide a public or proteced
// method loopImpl() which will be called by the thread.
// The thread gets created in PThreadObjectSemantic's
// c'tor and terminated in its d'tor. The policy used
// (template parameter U) determins how often loopImpl() gets called.
template<class T, class U = RunOnce>
class PThreadObjectSemantic : public T, private PThread
{
	U runCondition_;
	bool isRunning_;	
	PMutex threadStarted_;	
	
	void threadImpl()
	{
		isRunning_ = true;
		threadStarted_.unlock();		
		
		while(runCondition_.next())
			T::operator()();
		
		isRunning_ = false;
	}
	
	void startUp()
	{
		runCondition_.reset();
		threadStarted_.lock();
		
		if(!run())
			throw CouldNotCreateThread();
		
		threadStarted_.lock();
	}
public:
	// If the thread has terminated when PThreadObjectSemantic's
	// d'tor is called the object will be destroyed. Otherwise
	// the thread is told to shut down via the policy method
	// shutdown(). Note that for PThreadObjectSemantic objects
	// that use the "RunForever" policy you need to invoke
	// the shutdown() method manually with false as parameter
	// to exit the thread.
	virtual ~PThreadObjectSemantic()
	{
		shutdown();
	}
	// Spawn off a thread that runs at least once
	// while the object exists. When the c'tor has
	// been worked off, either a thread is running
	// OR an exception of type CouldNotCreateThread
	// has been thrown.
	PThreadObjectSemantic()	
		:	T()		
		,	isRunning_(false)
	{
		startUp();
	}
	// Constructor that passes its single argument to T's
	// c'tor.
	template<class X>
	PThreadObjectSemantic(X& arg)	
		:	T(arg)		
		,	isRunning_(false)
	{
		startUp();
	}
	// c'tor.
	template<class X, class Y>
	PThreadObjectSemantic(X& arg1, Y& arg2)	
		:	T(arg1, arg2)		
		,	isRunning_(false)
	{
		startUp();
	}

	// c'tor.
	template<class X, class Y, class Z>
	PThreadObjectSemantic(X& arg1, Y& arg2, Z& arg3)	
		:	T(arg1, arg2, arg3)		
		,	isRunning_(false)
	{
		startUp();
	}
	// c'tor.
	template<class X, class Y, class Z, class A>
	PThreadObjectSemantic(X& arg1, Y& arg2, Z& arg3, A& arg4)	
		:	T(arg1, arg2, arg3, arg4)		
		,	isRunning_(false)
	{
		startUp();
	}
	// c'tor.
	template<class X, class Y, class Z, class A, class B>
	PThreadObjectSemantic(X& arg1, Y& arg2, Z& arg3, A& arg4, B& arg5)	
		:	T(arg1, arg2, arg3, arg4, arg5)		
		,	isRunning_(false)
	{
		startUp();
	}
	// c'tor.
	template<class X, class Y, class Z, class A, class B, class C>
	PThreadObjectSemantic(X& arg1, Y& arg2, Z& arg3, A& arg4, B& arg5, C& arg6)	
		:	T(arg1, arg2, arg3, arg4, arg5, arg6)		
		,	isRunning_(false)
	{
		startUp();
	}
	// You may call this method to shut down the 
	// thread before the d'tor is reached.
	// Call this with false as parameter to 
	// terminate the thread as soon as possible
	void shutdown(bool waitTillCompletion = true) 
	{ 
		if(!waitTillCompletion)
			runCondition_.shutdown();

		join();
	}
	// Call this if you wish to rerun the thread
	// You may only call this method if there is
	// no thread currently running. This method
	// may cause a CouldNotCreateThread exception.
	bool start()
	{
		if(isRunning_)
			return false;
		
		startUp();
		return true;
	}
};

/************************************************************************/
/* ThreadPool                                                           */
/************************************************************************/

// Thread pool with N worker threads. This template class takes a functor
// and an argument to the functor as template arguments. If the argument
// to the functor is void, no argument is passed. 
// The thread pool provided synchronous and asynchronous assignment of
// worker threads.
// By default, the argument passed is copied, however, if a pointer is passed
// it is assumned that it points to an object that was allocated with new and
// may safely be deleted by a worker thread.
// Allocation and deallocation of threads is synchronous with respect to a
// thread pool object. Upon creation N worker threads are created. When a 
// thread pool goes out of scope, all threads are shut down before the 
// object is destroyed.


template<class RequestProzessor /* functor */, class ArgumentT /* Argument to functor, may be void */>
class WorkerControl_;

template<class RequestProzessor, class ArgumentT>
class PThreadPool
{
public:
	typedef RequestProzessor WorkerType;
	typedef ArgumentT ArgumentType;
public:
	// Construct a thread pool with n workers.
	PThreadPool(unsigned int noThreads)
		:	controller_(noThreads)
	{}	
	// Process one item. Call this if the 
	// argument type is void (no argument needed).
	// Caller is blocked until a worker was assigned.
	inline void process()
	{
		controller_.assignWorker()->process();
	}

	// Process one item. Call this if the 
	// argument type is X. The item passed is
	// copied.
	// Caller is blocked until a worker was assigned.
	template<class X>
	inline void process(const X& arg)
	{
		controller_.assignWorker()->process(new X(arg));
	}

	// Same as above but the argument passed is passed
	// as is. NOTE: This must be a pointer to a heap
	// allocated object.
	template<class X>
	void process(X* arg)
	{
		controller_.assignWorker()->process(arg);
	}

	// Same as process() but the method returns immediately
	// indicating if a worker was assigned.
	bool tryProcess()
	{
		InternalWorkerType* worker = controller_.tryAssignWorker();
		if(worker)
			worker->process();
		return worker != 0;
	}

	// Same as template<class X> process(const X& arg) but the method returns immediately
	// indicating if a worker was assigned.
	template<class X>
	bool tryProcess(const X& arg)
	{
		InternalWorkerType* worker = controller_.tryAssignWorker();
		if(worker)
			worker->process(new X(arg));
		return worker != 0;
	}

	// Returns the (current) number of busy workers.
	inline unsigned int active() const
	{
		return controller_.active();
	}
	// Returns the size of the thread pool.
	inline unsigned size() const
	{
		return controller_.size();
	}
private:
	PThreadPool();
private:
	typedef WorkerControl_<RequestProzessor, ArgumentT> ControlType;
	typedef typename ControlType::WorkerType InternalWorkerType;
	ControlType controller_;	
};

/************************************************************************/
/* ThreadPool (Implementation)                                          */
/************************************************************************/

#if !defined(_MSC_VER) || _MSC_VER >= 1300
	template<class RequestProzessor, class ArgumentT>
	struct Select_
	{
		static void process(ArgumentT* arg)
		{
			RequestProzessor()(*arg);
		}	
	};

	template<class RequestProzessor>
	struct Select_<RequestProzessor, void>
	{
		static void process(void*)
		{
			RequestProzessor()();
		}	
	};
#endif

template<class RequestProzessor, class ArgumentT>
class Worker_ : private PThread
{
	template<class U>
	class ThinSmrtPtr
	{	
		U* ptr;			
	public:
		explicit 
		ThinSmrtPtr(U* p = 0) 
			:	ptr(p) 
		{}
		ThinSmrtPtr(ThinSmrtPtr& foo)
		{
			Copy(foo);
		}

		~ThinSmrtPtr()
		{
			PointerTraits_<U>::freeResource(ptr);
		}
		
		
		ThinSmrtPtr& operator=(ThinSmrtPtr& foo)
		{
			if(&foo == this) return *this;
			PointerTraits_<U>::freeResource(ptr);
			Copy(foo);
			return *this;
		}

		ThinSmrtPtr& operator=(U* a)
		{
			if(a == ptr) return *this;
			PointerTraits_<U>::freeResource(ptr);
			ptr = a;
			return *this;
		}

		U* get() const
		{
			return ptr;
		}
	private:		
		void Copy(ThinSmrtPtr& foo)
		{
			ptr = foo.ptr;
			foo.ptr = 0;
		}
	};
	
#if defined(_MSC_VER) && _MSC_VER < 1300
	template<class T>
	struct Select_
	{
		static void process(T* arg)
		{
			RequestProzessor()(*arg);
		}
	};

	template<>
	struct Select_<void>
	{
		static void process(void*)
		{
			RequestProzessor()();
		}		
	};
#endif

	
	typedef WorkerControl_<RequestProzessor, ArgumentT> ControlType;
public:	
	~Worker_()
	{
		shutdown();
		join();		
	}
	Worker_(ControlType* wc)
		:	arg_(0)
		,	wc_(wc)
		,	keepRunning_(true)
	{
		assert(wc);
		run();
	}
	void process(ArgumentT* arg = 0)
	{		
		setArgument(arg);
		unblockThread();
	}
	void shutdown()
	{
		keepRunning_ = false;		
		startProcessing_.signal();		
	}
private:
	Worker_();
	void threadImpl()
	{
		while(true)
		{
			waitForRequest();			
			
			if(shutdownRequest())
				break;
		
			#if defined(_MSC_VER) && _MSC_VER < 1300
				Select_<ArgumentT>::process(ThinSmrtPtr<ArgumentT>(arg_).get());
			#else
				Select_<RequestProzessor, ArgumentT>::process(ThinSmrtPtr<ArgumentT>(arg_).get());
			#endif			
			enableRequests();
		}
	}
	
	inline bool shutdownRequest()
	{
		return !keepRunning_;
	}
	inline void waitForRequest()
	{
		startProcessing_.wait();
	}
	inline void enableRequests()
	{
		wc_->addFreeWorker(this);
	}
	inline void setArgument(ArgumentT* arg)
	{
		arg_ = arg;
	}
	inline void unblockThread()
	{
		startProcessing_.signal();
	}
private:
	ThinSmrtPtr<ArgumentT> arg_;
	ControlType* wc_;
	PCondition startProcessing_;
	volatile bool keepRunning_;	
};

template<class RequestProzessor, class ArgumentT>
class WorkerControl_
{
public:
	typedef Worker_<RequestProzessor, ArgumentT> WorkerType;
	~WorkerControl_()
	{
		// decrementing the semaphore will ensure that no
		// worker is still active
		for(size_t i = 0; i < workers_.size(); ++i)
			freeCount_.down();
	}
	WorkerControl_(unsigned size)
		:	workers_(size)
		,	freeCount_(size)
		,	active_(0)
	{
		for(size_t i = 0; i < workers_.size(); ++i)
		{
			workers_[i] = new WorkerType(this);
			assert(workers_[i]);
			free_.push(workers_[i].get());
		}
	}
	inline void addFreeWorker(WorkerType* worker)
	{
		--active_;
		free_.push(worker);
		freeCount_.up();
	}
	WorkerType* assignWorker()
	{
		freeCount_.down();
		
		WorkerType* ret = 0;
		free_.pop(ret);
		assert(ret);
		++active_;
		return ret;
	}
	WorkerType* tryAssignWorker()
	{
		WorkerType* ret = 0;
		free_.pop(ret);
		if(ret)
			++active_;
		return ret;
	}
	inline size_t size() const { return workers_.size(); }
	inline size_t active() const { return (size_t)active_.get(); }
private:
	WorkerControl_();
private:
	typedef std::vector< StackPtr<WorkerType> > Workers;
	Workers workers_;
	typedef PLockFreeStack<WorkerType*> FreeWorkersQueue;
	FreeWorkersQueue free_;
	PSemaphore freeCount_;
	PAtomicNumber active_;
};


template<>
class PSingleWait<PThread>
{
	PThread& waitFor_;
	PSingleWait();
public:	
	PSingleWait(PThread& waitfor)
		:	waitFor_(waitfor)
	{}
	void wait()
	{
		waitFor_.join();	
	}
};
PT_NAMESPACE_END

#endif
