/*  Copyright (c) January 2005 Jean Gressmann (jsg@rz.uni-potsdam.de)
 *
 *  This is free software; you can redistribute it and/or modify
 *	it under the terms of the GNU General Public License as published by
 *	the Free Software Foundation; either version 2 of the License, or
 *	(at your option) any later version. 
 * 
 *	This file is distributed in the hope that it will be useful,
 *	but WITHOUT ANY WARRANTY; without even the implied warranty of
 *	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *	GNU General Public License for more details.
 *
 *	You should have received a copy of the GNU General Public License
 *	along with this file; if not, write to the Free Software
 *	Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */
 
#ifndef UNIX_CONDITION_H
#define UNIX_CONDITION_H
#include <portablethreads/unix/config.h>
#include <portablethreads/unix/mutex.h>
#include <time.h>

PT_NAMESPACE_BEGIN
class PCondition : public PWaitable
{
	volatile bool predicate_;
	PMutex mutex_;
	pthread_cond_t cond_;

	PCondition(const PCondition& o);
	PCondition& operator=(const PCondition& o);
public:
	PCondition() 
		:	predicate_(false)
	{
		while(pthread_cond_init(&cond_, 0) < 0);
	}
	
	~PCondition()
	{
		while(pthread_cond_destroy(&cond_) < 0);			
	}

	void signal()
	{ 
		mutex_.lock();
		predicate_ = true;
		pthread_cond_signal(&cond_);
		mutex_.unlock();
	}
	/*
	void broadcast()
	{ 
		mutex_.lock();
		predicate_ = true;
		pthread_cond_broadcast(&cond_);
		mutex_.unlock();
	}
	*/
	void wait()
	{
		mutex_.lock();
		while(!predicate_)
		{ 
			// upon exit, mutex_ is owned by the calling thread
			// ok, only possible error should be "invalid mutex" or "not owned"
  			pthread_cond_wait(&cond_, mutex_);
     	}
		predicate_ = false;
		mutex_.unlock();
	} 
	bool wait(unsigned long sec, unsigned long nano)
	{
		//			struct timespec
		//  {
		//    long int tv_sec;            /* Seconds.  */
		//    long int tv_nsec;           /* Nanoseconds.  */
		//  };
		
		if(nano >= 1000000000)
		{
			const unsigned long extraSecs = nano / 1000000000;
			nano -= extraSecs*1000000000;
			sec += extraSecs;
		}
		
		timespec t;
		clock_gettime(CLOCK_REALTIME, &t);
		t.tv_sec += sec;
		t.tv_nsec += nano;	
		mutex_.lock();
		bool waitedSuccessfully = true;
		
		while(!predicate_)
		{
			// Nobody has called signal or broadcast yet, hence no state to collect
			// Thus we wait till the condition is miracly set
			const int ret = pthread_cond_timedwait(&cond_, mutex_, &t);
			if(ret == ETIMEDOUT || ret == EINVAL)
			{
				// report back that the condition did not 
				// become signaled within timeout
				waitedSuccessfully = false;
				break;
			}			
			else // Hier war das Warten erfolgreich, sprich ein anderer Thread hat predicate_ ber signal oder broadcast war gemacht
			{
				// Here the condition was signaled...
				break;
			}
		}
		// We collected the condition's state, hence reset the condition to nonsignaled
		// Note this is mutual exclusive with broadcast, because each thread will
		// set the flag back to false hence leaving other threads to wait for another call
		// to signal
		predicate_ = false;
		mutex_.unlock();
		return waitedSuccessfully;				
	}
	
	inline bool wait(unsigned long milliseconds)
	{
		const unsigned long sec = milliseconds / 1000;
		const unsigned long nano = (milliseconds - (sec*1000)) * 1000000;
		return wait(sec, nano);
	}
	
	inline operator pthread_cond_t*() { return &cond_; }
	inline operator pthread_mutex_t*() { return mutex_.operator pthread_mutex_t*(); }
	bool queryState()
	{ 
		return wait(0, 0);
	}
};

PT_NAMESPACE_END


#endif
