/*  Copyright (c) November 2004 Jean Gressmann (jsg@rz.uni-potsdam.de)
 *
 *  This is free software; you can redistribute it and/or modify
 *	it under the terms of the GNU General Public License as published by
 *	the Free Software Foundation; either version 2 of the License, or
 *	(at your option) any later version. 
 * 
 *	This file is distributed in the hope that it will be useful,
 *	but WITHOUT ANY WARRANTY; without even the implied warranty of
 *	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *	GNU General Public License for more details.
 *
 *	You should have received a copy of the GNU General Public License
 *	along with this file; if not, write to the Free Software
 *	Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */
 
#ifndef ARRAY_ISTREAM_H
#define ARRAY_ISTREAM_H

#include <string>
#include <cassert>
#include <algorithm>
#include <streambuf>
#include <istream>

template<class T, class Traits = std::char_traits<T> >
class basic_array_streambuf : public std::basic_streambuf<T, Traits>
{
	typedef std::basic_streambuf<T, Traits> Base;
	typedef typename Traits::char_type* PT;
	typedef const typename Traits::char_type* CPT;
public:
	explicit
	basic_array_streambuf(CPT p, size_t size)
		:	buffer_(const_cast<PT>(p))
		,	size_(size)
	{
		assert(p && "null pointer passed");
		// first, disable write support in basic_streambuf ...
		Base::setp(0, 0);
		// then tell the base class about our string.
		Base::setg(buffer_, buffer_, buffer_+size_);
	}
	virtual std::streamsize xsgetn(PT to, std::streamsize size)
	{
		const std::streamsize extracted = 
			std::min(static_cast<std::streamsize>(Base::egptr() - Base::gptr()), size);
		Traits::copy(to, Base::gptr(), extracted);
		Base::setg(buffer_, buffer_+extracted, buffer_+size_);
		return extracted;
	}
	virtual typename Base::pos_type seekoff(typename Base::off_type offset,
											std::ios_base::seekdir dir,
											std::ios_base::openmode which = std::ios_base::in | std::ios_base::out)
	{
		if(which & std::ios_base::out)
			return Base::seekoff(offset, dir, which);

		if(dir == std::ios_base::cur)
			offset += static_cast<typename Base::off_type>(Base::gptr() - Base::eback());
		else if(dir == std::ios_base::end)
			offset = static_cast<typename Base::off_type>(size_) - offset;

		return seekpos(offset, which);
	}
	virtual typename Base::pos_type seekpos(typename Base::pos_type offset,
											std::ios_base::openmode which = std::ios_base::in | std::ios_base::out)
	{
		if(which & std::ios_base::out)
			return Base::seekpos(offset, which);

		if(offset < 0 || static_cast<size_t>(offset) > size_)
			offset = Base::seekpos(offset, which);
		else
			// do seeking now that the offset has been transformed into 
			// a positive value counting from the beginning of the
			// buffer.
			Base::setg(buffer_, buffer_+offset, buffer_+size_);

		return static_cast<typename Base::pos_type>(offset);
	}
   	inline void set_array(CPT p, size_t size = 0)
	{
		assert(p && "null pointer passed");
		buffer_ = const_cast<PT>(p);
		size_ = size;
		Base::setg(buffer_, buffer_, buffer_+size_);
	}
private:
	basic_array_streambuf();
	basic_array_streambuf(const basic_array_streambuf&);
	basic_array_streambuf& operator=(const basic_array_streambuf&);
protected:
	PT buffer_;
	size_t size_;
};

typedef basic_array_streambuf<char> array_streambuf;
typedef basic_array_streambuf<wchar_t> warray_streambuf;

template<class T, class Traits = std::char_traits<T> >
class basic_array_istream : public std::basic_istream<T, Traits>
{
	typedef std::basic_istream<T, Traits> Base;
	typedef basic_array_streambuf<T, Traits> BufferType;
	typedef typename Traits::char_type* PT;
	typedef const typename Traits::char_type* CPT;
public:
	explicit
	basic_array_istream(CPT p)
		:	Base(0)
		,	buffer_(p, Traits::length(p))
	{
		assert(p && "null pointer passed");
		Base::rdbuf(&buffer_);
	}
	explicit
	basic_array_istream(CPT p, size_t size)
		:	Base(0)
		,	buffer_(p, size)
	{
		assert(p && "null pointer passed");
		Base::rdbuf(&buffer_);
	}
	explicit
	basic_array_istream(const std::basic_string<T>& s)
		:	Base(0)
		,	buffer_(s.c_str(), s.size())
	{
		Base::rdbuf(&buffer_);
	}
	inline void set_array(CPT p)
	{
		assert(p && "null pointer passed");
		buffer_.set_array(p, Traits::length(p));
	}
	inline void set_array(CPT p, size_t size)
	{
		assert(p && "null pointer passed");
		buffer_.set_array(p, size);
	}
	inline void set_array(const std::basic_string<T>& s)
	{
		buffer_.set_array(s.c_str(), s.size());
	}
private:
	basic_array_istream();
	basic_array_istream(const basic_array_istream&);
	basic_array_istream& operator=(const basic_array_istream&);
protected:
	BufferType buffer_;
};

typedef basic_array_istream<char> array_istream;
typedef basic_array_istream<wchar_t> warray_istream;

#endif
